package de.dim.diamant.connection;

import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import de.dim.diamant.model.OutboundLogistic;
import de.dim.diamant.model.OutboundSerializer;
import de.dim.diamant.model.Product;
import de.dim.diamant.model.ProductSerializer;

public abstract class ConnectionFacade {

  private String connectionURL = null;
  private int timeout = 3000; 


  public enum ConnectionMethod {
    GET,
    POST,
    PUT,
    DELETE
  }

  /**
   * Constructor with default data and GUID parameter
   * @param url the connection url
   */
  public ConnectionFacade(String url) {
    setConnectionURL(url);
  }
  
  /**
   * Sends a GET request
   * @param objectType the EClass type of the result to be loaded
   * @return the {@link Object} or null in case an error occurs
   */
  public <T> T sendGetRequest(Class<T> objectType, String path) {
    HttpURLConnection conn = null;
    try {
      // prepare GET request
      URL url = getConnectionURL();
      if (path != null) {
        url = new URL(getConnectionURL().toString() + path);
      }
      conn = prepareRequest(ConnectionMethod.GET, url);
      // load into EMF data model
      DataInputStream dis = new DataInputStream(conn.getInputStream());
      Gson gson = new GsonBuilder().create();
      return gson.fromJson(new InputStreamReader(dis), objectType);
    } catch (IOException e) {
      Log.e("Connection", "sendGetRequest: Error executing GET request because of I/O exception: ", e);
      return null;
    } catch (Exception e) {
      Log.e("Connection", "sendGetRequest: Error executing GET request: ", e);
      return null;
    } finally {
      if (conn != null) {
        conn.disconnect();
      }
    }
  }

  /**
   * Sends a GET request
   * @return the {@link Object} or null in case an error occurs
   */
  public Object sendGetRequest(String path) {
    HttpURLConnection conn = null;
    try {
      // prepare GET request
      URL url = getConnectionURL();
      if (path != null) {
        url = new URL(getConnectionURL().toString() + path);
      }
      conn = prepareRequest(ConnectionMethod.GET, url);
      // load into EMF data model
      Map<String, Object> loadProperties = new HashMap<>();
      //resource.load(conn.getInputStream(), loadProperties);
      return null;
    } catch (IOException e) {
      Log.e("Connection", "sendGetRequest: Error executing GET request because of I/O exception: ", e);
      return null;
    } catch (Exception e) {
      Log.e("Connection", "sendGetRequest: Error executing GET request: ", e);
      return null;
    } finally {
      if (conn != null) {
        conn.disconnect();
      }
    }
  }
  
  /**
   * Sends a get request and only returns its HTTP response code
   * @return the HTTP repospnse code
   */
  public int sendGetRequestStatusOnly(){
    return sendRequest(ConnectionMethod.GET, null);
  }
  /**
   * Sends a POST request
   * @param object the object to be posted
   * @return the HTTP response code or -1 on errors
   */
  public int sendPostRequest(Object object, String path) {
    return sendRequest(object, ConnectionMethod.POST, path);
  }

  /**
   * Sends a PUT request
   * @param object the object to be posted
   * @return the HTTP response code or -1 on errors
   */
  public int sendPutRequest(Object object, String path) {
    return sendRequest(object, ConnectionMethod.PUT, path);
  }
  
  /**
   * Sends a DELETE request
   * @return the HTTP response code or -1 on errors
   */
  public int sendDeleteRequest() {
    return sendRequest(ConnectionMethod.DELETE, null);
  }

  /**
   * Sends a request
   * @param object the object to be whatever
   * @param method the connection method, that should be executed 
   * @return the HTTP response code or -1 on errors
   */
  public int sendRequest(Object object, ConnectionMethod method, String path) {
    if (object == null) {
      throw new IllegalStateException("Error executing using method: " + method + " request because object or content type is/are null: " + object);
    }
    HttpURLConnection conn = null;
    try {
      URL url = getConnectionURL();
      if (path != null) {
        url = new URL(getConnectionURL().toString() + path);
      }
      // prepare POST request
      conn = prepareRequest(method, url);
      // add feedbacks to the resource
      // create output stream
      //DataOutputStream printout = new DataOutputStream(conn.getOutputStream());
      Gson gson = new GsonBuilder().
              registerTypeAdapter(OutboundLogistic.class, new OutboundSerializer()).
              registerTypeAdapter(Product.class, new ProductSerializer()).
              create();
      String jsonString =  gson.toJson(object);
      Log.i("JSON", "sendRequest: " + jsonString);
      OutputStream os = conn.getOutputStream();
      byte[] input = jsonString.getBytes("utf-8");
      os.write(input, 0, input.length);
      //resource.save(printout, options);
      int responseCode = conn.getResponseCode();
      if (responseCode == 200) {
        conn.getInputStream().close();
      }
//            logger.log(Level.INFO, "[ConnectionFacade#sendPostRequest] Content: " + new String(printout.toByteArray()));
     return responseCode;
    } catch (Exception e) {
      throw new IllegalStateException("Error executing using method: " + method + " request for URL " + connectionURL, e);
    } finally {
      if (conn != null) {
        conn.disconnect();
      }
    }
  }
  
  /**
   * Sends a request without object
   * @return the HTTP response code or -1 on errors
   */
  public int sendRequest(ConnectionMethod method, URL url) {
    int responseCode = HttpURLConnection.HTTP_BAD_REQUEST;
    HttpURLConnection conn = null;
    try {
      // prepare request
      conn = prepareRequest(method, url);
      responseCode = conn.getResponseCode();      
      return responseCode;
    } catch (IOException e) {
      Log.e("Connection", "sendRequest: Error executing request: " + method, e);
      return HttpURLConnection.HTTP_INTERNAL_ERROR;
    } finally {
      if (conn != null) {
        conn.disconnect();
      }
    }
  }

  public int getTimeout() {
    return timeout;
  }

  public void setTimeout(int timeout) {
    this.timeout = timeout;
  }

  /**
   * Sets the connection url
   * @param url the connection url, must be not null
   */
  protected void setConnectionURL(String url) {
    if (url == null) {
      throw new IllegalArgumentException("Url parameter of the connection facade must be not null");
    }
    connectionURL = url;
  }

  /**
   * Returns the connection uri
   * @return the connection uri
   * @throws java.net.MalformedURLException thrown id the uri string is wrong
   */
  protected final URL getConnectionURL() throws MalformedURLException {
    return new URL(connectionURL);
  }

  /**
   * Prepares a request for the given HTTP method
   * @param method the HTTP method
   * @return the {@link HttpURLConnection}
   * @throws IOException
   */
  protected HttpURLConnection prepareRequest(ConnectionMethod method, URL url) throws IOException {
    url = url == null ? getConnectionURL() : url;
    HttpURLConnection conn = (HttpURLConnection) url.openConnection();
    Map<String, String> header = getHeaderMap();
    if (header != null) {
      header.forEach((k,v)->conn.setRequestProperty(k, v));
    }
    if (ConnectionMethod.PUT.equals(method) || ConnectionMethod.POST.equals(method)) {
      conn.setDoOutput(true);
    }
    conn.setRequestMethod(method.name());
    if (!ConnectionMethod.DELETE.equals(method)) {
      conn.setDoInput(true);
    }
    conn.setConnectTimeout(getTimeout());
    return conn;
  }

  protected abstract Map<String, String> getHeaderMap();

}