package de.dim.diamant.model;

import android.content.Context;
import android.content.SharedPreferences;

import androidx.preference.PreferenceManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import de.dim.diamant.connection.BackendService;

public class ModelCache implements SharedPreferences.OnSharedPreferenceChangeListener {

    private BackendService backend;
    private final List<Product> productCache = new LinkedList<>();
    private final Map<String, List<OutboundLogistic>> productOutboundCache = new HashMap<>();
    private final Map<String, List<TransactionEntry>> productTransactionCache = new HashMap<>();
    private final SharedPreferences sharedPreferences;
    private static ModelCache INSTANCE = null;
    private static Map<String, String> productCatalog = new HashMap<>();
    private static Map<String, String> outboundCatalog = new HashMap<>();


    static {
        productCatalog.put("4047075147724", "Dist. Femurplatte m.kon.Gew. 4,5 - 6; 10L;214x18;D 6,0;R;T");
        productCatalog.put("4047075082315", "Spiralbohrer f.SK;3,5x195;SpL. 60;2-lippig");
        productCatalog.put("4047075068876", "Kortikalisschraube;3,5x10;VG;ss;S");
        productCatalog.put("4047075069095", "Kortikalisschraube;3,5x22;VG;ss;S");
        productCatalog.put("4047075068449", "Kortikalisschraube;3,5x30;VG;S");
        productCatalog.put("4260001210754", "LipoNit Augenspray Sensitiv");
        outboundCatalog.put("9002236311036", "Hermes");
    }

    public static ModelCache getInstance(Context context) {
        if (INSTANCE == null) {
            INSTANCE = new ModelCache(context);
        }
        return INSTANCE;
    }

    private ModelCache(Context context) {
        sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);
        updatePreferences(sharedPreferences);
        sharedPreferences.registerOnSharedPreferenceChangeListener(this);
    }

    public void updatePreferences(SharedPreferences sharedPreferences) {
        String host = sharedPreferences.getString("host", "diamant.data-in-motion.biz/rest");
        String protocol = sharedPreferences.getString("protocol", "https");
        String urlString = String.format("%s://%s", protocol, host);
        if (backend == null) {
            backend = new BackendService(urlString);
        } else {
            backend.setUrl(urlString);
        }
    }

    public void addProduct(Product product) {
        if (product == null) {
            return;
        }
        backend.createProduct(product);
        getProducts();
    }

    public void addOutbound(OutboundLogistic outbound) {
        if (outbound == null) {
            return;
        }
        if (outbound.getProduct() != null) {
            backend.createOutbound(outbound);
            // update cache
            getTransactions(outbound.getProduct());
        }
    }

    public int getProductsSize() {
        return productCache.size();
    }

    public List<Product> getProducts() {
        synchronized (productCache) {
            List<Product> products = backend.getProducts();
            if (products != null && products.size() > 0) {
                productCache.clear();
                productCache.addAll(products);
            }
            return Collections.unmodifiableList(productCache);
        }
    }

    public Product getProduct(int position) {
        synchronized (productCache) {
            return productCache.get(position);
        }
    }

    public OutboundLogistic getOutbound(Product product, int position) {
        if (product == null) {
            return null;
        }
        synchronized (productOutboundCache) {
            List<OutboundLogistic> outbounds = productOutboundCache.get(product.getId());
            if (outbounds == null) {
                return null;
            }
            return outbounds.get(position);
        }
    }

    public TransactionEntry getTransaction(Product product, int position) {
        if (product == null) {
            return null;
        }
        synchronized (productTransactionCache) {
            List<TransactionEntry> entries = productTransactionCache.get(product.getId());
            if (entries == null) {
                return null;
            }
            return entries.get(position);
        }
    }

    public List<OutboundLogistic> getOutbounds(Product product) {
        if (product == null) {
            return Collections.emptyList();
        }
        synchronized (productOutboundCache) {
            getTransactions(product);
            List<OutboundLogistic> outbounds = productOutboundCache.get(product.getId());
            if (outbounds == null) {
                return Collections.emptyList();
            }
            return Collections.unmodifiableList(outbounds);
        }
    }

    public List<TransactionEntry> getTransactions(Product product) {
        if (product == null) {
            return Collections.emptyList();
        }
        synchronized (productTransactionCache) {
            List<TransactionEntry> entries = backend.getEntries(product);
            if (entries != null) {
                productTransactionCache.put(product.getId(), entries);
                List<OutboundLogistic> ols = entries.stream().
                        filter(e->"outboundLogisticsX".equals(e.getType())).
                        map(e->new OutboundLogistic(e)).
                        collect(Collectors.toList());
                productOutboundCache.put(product.getId(), ols);
            } else {
                entries = Collections.emptyList();
            }
            return Collections.unmodifiableList(entries);
        }
    }

    public static Product createProduct(String ean) {
        Product p = new Product();
        if (ean != null) {
            p.setEan(ean);
            String desc = productCatalog.get(ean);
            if (desc != null) {
                p.setDescription(desc);
            }
        }
        return p;
    }

    public static OutboundLogistic createOutboundLogistic(String tracking, Product product) {
        if (product == null) {
            return null;
        }
        OutboundLogistic ol = new OutboundLogistic(product);
        if (tracking != null) {
            ol.setTrackingId(tracking);
            String provider = getProvider(tracking);
            if (provider != null) {
                ol.setProvider(provider);
            }
        }
        return ol;
    }

    public static String getProvider(String code) {
        if (code == null) {
            return null;
        }
        if (code.startsWith("JJ") || code.startsWith("2L")) {
            return "DHL";
        } else if (code.indexOf("92") == 8 || code.indexOf("94") == 8) {
            return "UPS";
        } else {
            return outboundCatalog.get(code);
        }
    }

    @Override
    public void onSharedPreferenceChanged(SharedPreferences sharedPreferences, String s) {
       updatePreferences(sharedPreferences);
    }
}
