/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.api.mmt.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.UUID;

import org.gecko.qvt.osgi.api.ConfigurableModelTransformatorPool;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.playertour.backend.api.CourseType;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.apis.mmt.common.PlayertourModelTransformator;
import com.playertour.backend.golfcourse.model.golfcourse.CourseDetails;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.Location;

/**
 * Tests the implementation of the m2m transformation for the GolfCourse from the backend model to the api model
 * @since 1.0
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class BEApiGolfCourseTransformationTest {

	@InjectBundleContext
	BundleContext bundleContext;
	
	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 1000) ServiceAware<ConfigurableModelTransformatorPool> transformatorPoolAware,
			@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=APIModelTransformator)") ServiceAware<PlayertourModelTransformator> transformatorAware, 
			@InjectService ConfigurationAdmin ca)
			throws Exception {
		
		assertThat(transformatorPoolAware.getServices()).hasSize(1);	
		ServiceReference<ConfigurableModelTransformatorPool> poolReference = transformatorPoolAware.getServiceReference();
		assertThat(poolReference).isNotNull();		
				
		assertThat(transformatorAware.getServices()).hasSize(1);	
		ServiceReference<PlayertourModelTransformator> transformatorReference = transformatorAware.getServiceReference();
		assertThat(transformatorReference).isNotNull();
		
		ConfigurableModelTransformatorPool poolService = transformatorPoolAware.getService();
		assertThat(poolService).isNotNull();		
		assertThat(poolService.getPoolMap()).hasSize(1);
	}
	
	@Test
	public void testCourseTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=APIModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> transformatorAware) throws Exception {
		
		assertThat(transformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = transformatorAware.getService();
		assertThat(transformatorService).isNotNull();
				
		GolfCourse golfCourse = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse.setId(UUID.randomUUID().toString());
		
		CourseDetails details = GolfCourseFactory.eINSTANCE.createCourseDetails();
		details.setCourseName("Golf Club Jena");
		Location loc = GolfCourseFactory.eINSTANCE.createLocation();
		loc.setLatitude(50.9234);
		loc.setLongitude(11.5087);
		details.setLocation(loc);
		golfCourse.setCourseDetails(details);
		
		CourseGPSVector gpsVector = GolfCourseFactory.eINSTANCE.createCourseGPSVector();
		gpsVector.setHoleCount(9);
		golfCourse.setCourseGPSVector(gpsVector);
		
		com.playertour.backend.api.GolfCourse apiCourse = transformatorService.transform(golfCourse, PlayerApiPackage.Literals.GOLF_COURSE);
		
		assertThat(apiCourse).isNotNull();
		assertThat(apiCourse.getId()).isEqualTo(golfCourse.getId());
		assertThat(apiCourse.getName()).isEqualTo("Golf Club Jena");
		assertThat(apiCourse.getLocation()).isNotNull();
		assertThat(apiCourse.getLocation().getLat()).isEqualTo(50.9234);
		assertThat(apiCourse.getLocation().getLng()).isEqualTo(11.5087);
		assertThat(apiCourse.getType()).isEqualTo(CourseType.NINE);
	}
	
}
