package com.playertour.backend.api.mmt;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.gecko.core.pool.Pool;
import org.gecko.qvt.osgi.api.ConfigurableModelTransformatorPool;
import org.gecko.qvt.osgi.api.ModelTransformator;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.mmt.common.PlayertourModelTransformator;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;

@Component(name = "APIModelTransformator")
public class APIModelTransformatorImpl implements PlayertourModelTransformator {

	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	@Reference
	private ConfigurableModelTransformatorPool poolComponent;
	
	@Override
	public <T extends EObject> T transform(EObject source, EClass to) throws UnknownTransformationException {
		return transform(source, source.eClass(), to);
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> T transform(EObject source, EClass sourceEClass, EClass to)
			throws UnknownTransformationException {
		String trafo = buildTemplatename(sourceEClass, to);
		logger.debug("searching for Trafo " + trafo);
		Pool<ModelTransformator> pool = poolComponent.getPoolMap().get(trafo);
		if(pool == null) {
			logger.warn("No Trafo found for " + trafo);
			throw new UnknownTransformationException("Could not find " + trafo);
		}
		ModelTransformator modelTransformator = pool.poll();
		try {
			return (T) modelTransformator.startTransformation(source);
		} finally {
			pool.release(modelTransformator);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T extends EObject> List<T> transform(List<? extends EObject> sourceList, EClass to)
			throws UnknownTransformationException {
		if(sourceList.isEmpty()) {
			return Collections.emptyList();
		}
		EClass sourceEClass = getSourceEClass(sourceList);
		String trafo = buildTemplatename(sourceEClass, to);
		logger.debug("searching for Trafo " + trafo);
		Pool<ModelTransformator> pool = poolComponent.getPoolMap().get(trafo);
		
		if(pool == null) {
			logger.warn("No Trafo found for " + trafo);
			throw new UnknownTransformationException("Could not find " + trafo);
		}
		ModelTransformator modelTransformator = pool.poll();
		try {
			List<? extends EObject> result = modelTransformator.startTransformations(sourceList);
			result = result.stream().filter(eo -> eo.eClass().isSuperTypeOf(to)).collect(Collectors.toList());
			return (List<T>) result;
		} finally {
			pool.release(modelTransformator);
		}
	}
	
	private String buildTemplatename(EClass source, EClass to) {
		String poolComponentName = "apiModelTransformatorService";
		String trafo = source.getEPackage().getNsPrefix() + "2" + to.getEPackage().getNsPrefix();
		String combinedId = poolComponentName+"-"+trafo;
		return combinedId;
	}
	
	private EClass getSourceEClass(List<? extends EObject> sourceList) {
		EClass sourceEClass = sourceList.get(0).eClass();
		for(int i = 0; i < sourceList.size(); i++) {
			EObject eo = sourceList.get(i);
			if(eo.eClass() != sourceEClass) {
				throw new IllegalArgumentException("All EObjects in the list must be of the same type found first Object was of type " + sourceEClass.getName() + " but EClass at index " + i + " was " + eo.eClass());
			}
		}
		return sourceEClass;
	}
}