/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.challenges.service.api;

import java.util.List;
import java.util.Optional;

import org.osgi.annotation.versioning.ProviderType;

import com.playertour.backend.challenges.model.challenges.Challenge;
import com.playertour.backend.challenges.model.challenges.ChallengeContent;
import com.playertour.backend.challenges.model.challenges.ChallengeResult;
import com.playertour.backend.challenges.model.challenges.ChallengeSearchResult;
import com.playertour.backend.challenges.model.challenges.PossibleChallengeParticipant;
import com.playertour.backend.challenges.model.challenges.PossibleChallengeParticipantResult;

@ProviderType
public interface ChallengesService {

	/**
	 * Retrieves the list of Players whose username matches the query and who currently have an open Scorecard for the provided 
	 * golf courseId, or an empty list.
	 * 
	 * @param courseId
	 * @param query
	 * @return
	 */
	List<PossibleChallengeParticipant> getPossibleChallengeParticipants(String courseId, String query);
	
	/**
	 * Retrieves the list of all Players who currently have an open Scorecard for the provided golf courseId, or an empty list.
	 * 
	 * @param courseId
	 * @return
	 */
	List<PossibleChallengeParticipant> getAllPossibleChallengeParticipants(String courseId);
	
	/**
	 * Open a {@link Challenge}, launched by the Player with the provided <code>challengeIssuerPlayerLoginId<code>
	 * and with the provided {@link ChallengeContent}
	 * 
	 * @param challengeIssuerPlayerLoginId
	 * @param challengeContent
	 * @return the newly opened {@link Challenge}
	 */
	Challenge openChallenge(String challengeIssuerPlayerLoginId, ChallengeContent challengeContent);
	
	/**
	 * The challenged Player can accept the {@link Challenge}. In this case the {@link Challenge} is 
	 * marked as ACCEPTED. If the {@link Challenge} is already CLOSED, then the Player cannot
	 * accept the {@link Challenge} and an Exception it thrown. If the Player is not the challenged Player, 
	 * an Exception is thrown. 
	 * 
	 * @param challengeId
	 * @param playerLoginId
	 * @return
	 */
	Challenge acceptChallenge(String challengeId, String playerLoginId);
	
	/**
	 * The challenged Player can decline the {@link Challenge}. In this case the {@link Challenge} is 
	 * marked as DECLINED. If the {@link Challenge} is already CLOSED, then the Player cannot
	 * decline the {@link Challenge} and an Exception it thrown. If the Player is not the challenged Player, 
	 * an Exception is thrown.
	 * 
	 * @param challengeId
	 * @param playerLoginId
	 * @return
	 */
	Challenge declineChallenge(String challengeId, String playerLoginId);
	
	/**
	 * A Player can invite another one to the {@link Challenge}.
	 * 
	 * @param challengeId
	 * @param senderLoginId the loginId of the inviting Player
	 * @param receiverLoginId the loginId of the invited Player
	 */
	Challenge inviteToChallenge(String challengeId, String senderLoginId, String receiverLoginId);
	
	/**
	 * A Player can join the {@link Challenge} and specifies whether is against or 
	 * in favor of the {@link ChallengeContent}. The Player must be a different one with 
	 * respect to the challenged Player, otherwise an Exception is thrown.
	 * The {@link Challenge} has to be in status ACCEPTED for a Player to be able to join, 
	 * otherwise an Exception is thrown.
	 * The challenged Player can use the {@link accept} or {@link decline} method.
	 * 
	 * @param challengeId
	 * @param playerLoginId
	 * @param isAgainst
	 * @return
	 */
	Challenge joinChallenge(String challengeId, String playerLoginId, Boolean isAgainst);
	
	/**
	 * Assign the {@link ChallengeResult} to the {@link Challenge}, based on whether it was satisfied or not.

	 * If <code>isChallengeSatisfied<code> is set to true, it means challenge met criteria defined in {@link ChallengeContent} and 
	 * has been won by Player who was in favor of {@link ChallengeContent} (Challenged Player). 
	 * 
	 * Otherwise it means challenge did not meet criteria defined in {@link ChallengeContent} and 
	 * has been won by Player who was against challenge content (Challenger / Challenge issuer).
	 * 
	 * @param challengeId
	 * @param isChallengeSatisfied
	 * @return
	 */
	Challenge resolveChallenge(String challengeId, boolean isChallengeSatisfied);
	
	/**
	 * Mark the {@link Challenge} as closed. The other challenge participants will get notified.
	 * 
	 * @param challengeId
	 * @param challengeIssuerPlayerLoginId
	 * @return
	 */
	Challenge closeChallenge(String challengeId, String challengeIssuerPlayerLoginId);
	
	/**
	 * Retrieve challenge with given id.
	 * 
	 * @param challengeId
	 * @return
	 */
	Challenge getChallengeById(String challengeId);
	
	/**
	 * Retrieve list of all challenges for a given Player.
	 * 
	 * @param loginId
	 * @param courseIdOptional
	 * @return
	 */
	List<Challenge> getChallengesByPlayerLoginId(String loginId, Optional<String> courseIdOptional);
	
	/**
	 * Retrieve list of all open challenges for a given Player.
	 * @param loginId
	 * @param courseIdOptional
	 * @return
	 */
	List<Challenge> getOpenChallengesByPlayerLoginId(String loginId, Optional<String> courseIdOptional);
	
	ChallengeSearchResult createChallengeSearchResult(List<Challenge> challenges);
	
	PossibleChallengeParticipantResult createPossibleChallengeParticipantResult(List<PossibleChallengeParticipant> possibleChallengeParticipants);
}
