/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.challenges.service.impl;

import java.util.Collections;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.DocumentIndexContextObjectImpl;
import org.gecko.search.util.DeferredCommitCallback;
import org.osgi.util.promise.Deferred;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.challenges.model.challenges.Challenge;
import com.playertour.backend.challenges.model.challenges.ChallengeParticipant;
import com.playertour.backend.challenges.model.challenges.ChallengesPackage;

public class ChallengesIndexHelper {
	static final String PARTICIPANTS_LOGIN_ID = "participants.playerLoginId";

	public static DocumentIndexContextObject mapChallengeNew(Challenge challenge) {
		return mapChallenge(challenge, IndexActionType.ADD, null);
	}

	public static DocumentIndexContextObject mapChallengeNew(Challenge challenge, Deferred<Boolean> deferred) {
		return mapChallenge(challenge, IndexActionType.ADD, deferred);
	}

	public static DocumentIndexContextObject mapChallengeUpdate(Challenge challenge, Deferred<Boolean> deferred) {
		return mapChallenge(challenge, IndexActionType.MODIFY, deferred);
	}

	public static DocumentIndexContextObject mapChallenge(Challenge challenge, IndexActionType indexAction, Deferred<Boolean> deferred) {
		Document doc = new Document();		
		
		doc.add(new StringField(LuceneIndexHelper.FIELD__ECLASS_URI, constructEClassUri(), Store.YES));
		doc.add(new StringField(LuceneIndexHelper.FIELD__ID, challenge.getId(), Store.YES));
		
		for(ChallengeParticipant participant : challenge.getParticipants()) {
			doc.add(new StringField(PARTICIPANTS_LOGIN_ID, participant.getPlayerLoginId(), Store.NO));
		}
		
		DocumentIndexContextObjectImpl.Builder builder = DocumentIndexContextObjectImpl.builder()
				.withDocuments(Collections.singletonList(doc))
				.withSourceObject(challenge).withIndexActionType(indexAction);

		if(IndexActionType.MODIFY.equals(indexAction) || IndexActionType.REMOVE.equals(indexAction)) {
			builder.withIdentifyingTerm(new Term("id", challenge.getId()));
		}
		if(deferred != null) {
			builder = builder.withCommitCallback(new DeferredCommitCallback(deferred));
		}

		return builder.build();
	}
	
	private static String constructEClassUri() {
		return EcoreUtil.getURI(ChallengesPackage.Literals.CHALLENGE).toString();
	}
}
