/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.challenges.service.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.document.LuceneIndexService;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.challenges.model.challenges.Challenge;
import com.playertour.backend.challenges.model.challenges.ChallengeStatusType;
import com.playertour.backend.challenges.service.api.ChallengesSearchService;

@Component(name = "ChallengesSearchService", immediate = true, scope = ServiceScope.SINGLETON)
public class ChallengesSearchServiceImpl implements ChallengesSearchService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;
	
	@Reference(target = "(id=challenges)")
	private LuceneIndexService challengesIndex;

	@Reference
	private ResourceSet resourceSet;

	@Reference(service=LoggerFactory.class)
	private Logger logger;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesSearchService#getChallengesByPlayerLoginId(java.lang.String)
	 */
	@Override
	public List<Challenge> getChallengesByPlayerLoginId(String loginId) {
		IndexSearcher searcher = challengesIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q = new TermQuery(new Term(ChallengesIndexHelper.PARTICIPANTS_LOGIN_ID, loginId));
				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if(topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (Challenge) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error( "Exception while searching for Challenge with challenge participant loginId " + loginId, e);
				return Collections.emptyList();
			}
		} finally {
			challengesIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesSearchService#getOpenChallengesByPlayerLoginId(java.lang.String)
	 */
	@Override
	public List<Challenge> getOpenChallengesByPlayerLoginId(String loginId) {
		List<Challenge> challenges = getChallengesByPlayerLoginId(loginId);
		if(challenges != null) {
			return challenges.stream().filter(b->ChallengeStatusType.OPEN.equals(b.getStatus())).collect(Collectors.toList());
		}
		return Collections.emptyList();
	}
}