/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.challenges.service.impl;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.repository.EMFRepository;
import org.javatuples.Pair;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.apis.player.APIGolfCourseService;
import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.challenges.model.challenges.Challenge;
import com.playertour.backend.challenges.model.challenges.ChallengeContent;
import com.playertour.backend.challenges.model.challenges.ChallengeParticipant;
import com.playertour.backend.challenges.model.challenges.ChallengeParticipantRole;
import com.playertour.backend.challenges.model.challenges.ChallengeResult;
import com.playertour.backend.challenges.model.challenges.ChallengeSearchResult;
import com.playertour.backend.challenges.model.challenges.ChallengeStatusType;
import com.playertour.backend.challenges.model.challenges.ChallengesFactory;
import com.playertour.backend.challenges.model.challenges.ChallengesPackage;
import com.playertour.backend.challenges.model.challenges.PossibleChallengeParticipant;
import com.playertour.backend.challenges.model.challenges.PossibleChallengeParticipantResult;
import com.playertour.backend.challenges.service.api.ChallengeMissingRequiredParticipantException;
import com.playertour.backend.challenges.service.api.ChallengeWrongGolfCourseException;
import com.playertour.backend.challenges.service.api.ChallengeWrongMeritPointsException;
import com.playertour.backend.challenges.service.api.ChallengeWrongMinStrokesException;
import com.playertour.backend.challenges.service.api.ChallengeWrongRoleException;
import com.playertour.backend.challenges.service.api.ChallengeWrongStatusException;
import com.playertour.backend.challenges.service.api.ChallengesIndexService;
import com.playertour.backend.challenges.service.api.ChallengesSearchService;
import com.playertour.backend.challenges.service.api.ChallengesService;
import com.playertour.backend.challenges.service.impl.events.publishers.ChallengesEventPublisher;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount;
import com.playertour.backend.meritpoints.service.api.MeritPointsService;
import com.playertour.backend.player.model.player.Player;

@Component(name = "ChallengesService", scope = ServiceScope.PROTOTYPE, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "ChallengesService")
@Designate(ocd = ChallengesServiceImpl.Config.class)
public class ChallengesServiceImpl implements ChallengesService {
	
	private static final String PAYEE_TRANSACTION_DESCRIPTION = "Challenges: Loss of %.2f merit point(s) with %.2f commission";
	private static final String RECEIVER_TRANSACTION_DESCRIPTION = "Challenges: Win of %.2f merit point(s) with %.2f commission";
	
	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;
	
	@Reference
	private EventAdmin eventAdmin;	
	
	@Reference(service = LoggerFactory.class)
	private Logger logger;	
	
	@Reference
	private ChallengesSearchService challengesSearchService;

	@Reference
	private ChallengesIndexService challengesIndexService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerService playerService;

	@Reference
	private PlayerSearchService playerSearchService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private APIGolfCourseService apiGolfCourseService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private MeritPointsService meritPointsService;
	
	/********************************/
	/** Externalized configuration **/	
	private Config config;
	
	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {

		// commission, taken when challenge is resolved
		double commission();
	}
	
	@Activate
	public void activate(Config config) {
		this.config = config;
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#getPossibleChallengeParticipants(java.lang.String, java.lang.String)
	 */
	@Override
	public List<PossibleChallengeParticipant> getPossibleChallengeParticipants(String courseId, String query) {
		Objects.requireNonNull(courseId, "Cannot retrieve possible challenge participants for course with null courseId!");

		List<Player> playersByUsername = playerSearchService.searchPlayersByUsername(query, Integer.MAX_VALUE);

		if (playersByUsername != null) {
			List<Player> playersByUsernameWithOpenScorecard = playersWithOpenScorecard(playersByUsername, courseId);

			if (!playersByUsernameWithOpenScorecard.isEmpty()) {
				return repackagePlayersToPossibleChallengeParticipants(playersByUsernameWithOpenScorecard);
			}
		}

		return Collections.emptyList();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#getPossibleChallengeParticipants(java.lang.String)
	 */
	@Override	
	public List<PossibleChallengeParticipant> getAllPossibleChallengeParticipants(String courseId) {
		Objects.requireNonNull(courseId, "Cannot retrieve possible challenge participants for course with null courseId!");

		List<Player> playersByScorecardCourseId = playerSearchService.searchPlayersByScorecardCourseId(courseId);

		if (playersByScorecardCourseId != null) {
			List<Player> playersByScorecardCourseIdWithOpenScorecard = playersWithOpenScorecard(
					playersByScorecardCourseId, courseId);

			if (!playersByScorecardCourseIdWithOpenScorecard.isEmpty()) {
				return repackagePlayersToPossibleChallengeParticipants(playersByScorecardCourseIdWithOpenScorecard);
			}
		}

		return Collections.emptyList();
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#openChallenge(java.lang.String, com.playertour.backend.challenges.model.challenges.ChallengeContent)
	 */
	@Override
	public Challenge openChallenge(String challengeIssuerPlayerLoginId, ChallengeContent challengeContent) {
		Objects.requireNonNull(challengeIssuerPlayerLoginId, "Login ID of player issuing challenge is required!");
		Objects.requireNonNull(challengeContent, "Challenge content is required!");

		Objects.requireNonNull(challengeContent.getChallengedPlayerLoginId(),
				"Login ID of challenged player is required!");
		if (challengeIssuerPlayerLoginId.equalsIgnoreCase(challengeContent.getChallengedPlayerLoginId())) {
			throw new ChallengeWrongRoleException("Player issuing challenge must be different from challenged player!");
		}

		if (Objects.isNull(challengeContent.getCourseId())
				|| !apiGolfCourseService.courseExists(challengeContent.getCourseId())) {
			throw new ChallengeWrongGolfCourseException("Valid golf course ID is required!");
		}

		if (challengeContent.getHoleNumber() < 1 || challengeContent.getHoleNumber() > 18) {
			throw new ChallengeWrongStatusException("Valid hole number is required!");
		}

		if (challengeContent.getMinStrokesNeeded() < 1) {
			throw new ChallengeWrongMinStrokesException("Valid minimum amount of strokes is required!");
		}

		if (challengeContent.getMeritPoints() < 1) {
			throw new ChallengeWrongMeritPointsException("Valid amount of merit points is required!");
		}

		Challenge challenge = ChallengesFactory.eINSTANCE.createChallenge();
		challenge.setStatus(ChallengeStatusType.OPEN);
		challenge.setDate(new Date());
		
		ChallengeParticipant issuer = ChallengesFactory.eINSTANCE.createChallengeParticipant();
		issuer.setPlayerLoginId(challengeIssuerPlayerLoginId);
		// challenge issuer (challenger) WINS if challenge content IS NOT satisfied, therefore they are betting AGAINST challenge content
		issuer.setAgainst(true);
		issuer.setRole(ChallengeParticipantRole.CHALLENGER);
		challenge.getParticipants().add(issuer);
		
		ChallengeParticipant challengedPlayer = ChallengesFactory.eINSTANCE.createChallengeParticipant();
		challengedPlayer.setPlayerLoginId(challengeContent.getChallengedPlayerLoginId());
		// challenged player WINS if challenge content IS satisfied, therefore they are betting FOR challenge content
		challengedPlayer.setAgainst(false);
		challengedPlayer.setRole(ChallengeParticipantRole.CHALLENGED);
		challenge.getParticipants().add(challengedPlayer);

		challenge.setContent(challengeContent);

		repository.save(challenge);
		challengesIndexService.indexChallenge(challenge, true);

		// Send notifications
		PlayerProfile challengedPlayerProfile = playerService
				.getPlayerProfileByLoginId(challengeContent.getChallengedPlayerLoginId());
		if (challengedPlayerProfile != null && challengedPlayerProfile.getLoginName() != null) {
			ChallengesEventPublisher.publishChallengeIssuedSelfEvent(eventAdmin, challengeIssuerPlayerLoginId,
					challengedPlayerProfile.getLoginName(), challenge.getId(), challenge.getDate().toString());
		}

		PlayerProfile challengeIssuerPlayerProfile = playerService
				.getPlayerProfileByLoginId(challengeIssuerPlayerLoginId);
		if (challengeIssuerPlayerProfile != null && challengeIssuerPlayerProfile.getLoginName() != null) {
			ChallengesEventPublisher.publishChallengeIssuedOtherEvent(eventAdmin,
					challengeContent.getChallengedPlayerLoginId(), challengeIssuerPlayerProfile.getLoginName(),
					challenge.getId(), challenge.getDate().toString());
		}

		return enhanceChallenge(challenge);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#acceptChallenge(java.lang.String, java.lang.String)
	 */
	@Override
	public Challenge acceptChallenge(String challengeId, String playerLoginId) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");
		Objects.requireNonNull(playerLoginId, "Login ID of player accepting challenge is required!");

		Challenge challenge = getChallengeById(challengeId);

		if (challenge != null) {

			if (ChallengeStatusType.CLOSED.equals(challenge.getStatus())) {
				throw new ChallengeWrongStatusException("Cannot accept a challenge which is in status CLOSED!");
			}

			Objects.requireNonNull(challenge.getContent(), "Cannot accept a challenge without content!");
			if (!playerLoginId.equals(challenge.getContent().getChallengedPlayerLoginId())) {
				throw new ChallengeWrongRoleException(
						"Only the challenged Player can accept the challenge. Other players must use the joinChallenge endpoint!");
			}

			challenge.setStatus(ChallengeStatusType.ACCEPTED);

			repository.save(challenge);

			challengesIndexService.indexChallenge(challenge, false);

			// Send notifications

			// @formatter:off
			Optional<ChallengeParticipant> challengeIssuerOptional = challenge.getParticipants().stream()
					.filter(c -> c.getRole() == ChallengeParticipantRole.CHALLENGER)
					.findFirst();
			// @formatter:on			

			if (challengeIssuerOptional.isPresent()) {
				PlayerProfile challengeIssuerPlayerProfile = playerService
						.getPlayerProfileByLoginId(challengeIssuerOptional.get().getPlayerLoginId());
				if (challengeIssuerPlayerProfile != null && challengeIssuerPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeAcceptedSelfEvent(eventAdmin, playerLoginId,
							challengeIssuerPlayerProfile.getLoginName(), challenge.getId(),
							challenge.getDate().toString());
				}

				PlayerProfile challengedPlayerProfile = playerService.getPlayerProfileByLoginId(playerLoginId);
				if (challengedPlayerProfile != null && challengedPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeAcceptedOtherEvent(eventAdmin,
							challengeIssuerOptional.get().getPlayerLoginId(), challengedPlayerProfile.getLoginName(),
							challenge.getId(), challenge.getDate().toString());
				}
			}
		}

		return enhanceChallenge(challenge);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#joinChallenge(java.lang.String, java.lang.String, java.lang.Boolean)
	 */
	@Override
	public Challenge joinChallenge(String challengeId, String playerLoginId, Boolean isAgainst) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");
		Objects.requireNonNull(playerLoginId, "Login ID of player invited to join challenge is required!");
		Objects.requireNonNull(isAgainst,
				"Specifying whether invited player is for or against challenge content is required!");

		Challenge challenge = getChallengeById(challengeId);

		if (challenge != null) {

			if (!ChallengeStatusType.ACCEPTED.equals(challenge.getStatus())) {
				throw new ChallengeWrongStatusException(
						"Cannot join a Challenge which has not been accepted by the challenged Player");
			}

			Objects.requireNonNull(challenge.getContent(), "Cannot accept a Challenge without content!");
			if (playerLoginId.equals(challenge.getContent().getChallengedPlayerLoginId())) {
				throw new ChallengeWrongRoleException(
						"The challenged Player can accept or decline the Challenge. This method is reserved to other Players");
			}

			// @formatter:off
			Optional<ChallengeParticipant> challengeParticipantOptional = challenge.getParticipants().stream()
					.filter(b -> playerLoginId.equals(b.getPlayerLoginId()))
					.findFirst();
			// @formatter:on

			if (!challengeParticipantOptional.isPresent()) {

				ChallengeParticipant newChallengeParticipant = ChallengesFactory.eINSTANCE.createChallengeParticipant();
				newChallengeParticipant.setPlayerLoginId(playerLoginId);
				newChallengeParticipant.setAgainst(isAgainst);
				newChallengeParticipant.setRole(ChallengeParticipantRole.INVITED);
				challenge.getParticipants().add(newChallengeParticipant);

				repository.save(challenge);

				challengesIndexService.indexChallenge(challenge, false);

				// Send notifications
				PlayerProfile invitedPlayerProfile = playerService.getPlayerProfileByLoginId(playerLoginId);
				if (invitedPlayerProfile != null && invitedPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeInviteAcceptedSelfEvent(eventAdmin, playerLoginId,
							invitedPlayerProfile.getLoginName(), challenge.getId(), challenge.getDate().toString());
				}

				// @formatter:off
				List<String> otherParticipantsLoginIds = challenge.getParticipants().stream()
						.filter(b -> !playerLoginId.equals(b.getPlayerLoginId()))
						.map(b -> b.getPlayerLoginId())
						.collect(Collectors.toList());
				// @formatter:on

				for (String otherParticipantsLoginId : otherParticipantsLoginIds) {
					PlayerProfile otherChallengeParticipantPlayerProfile = playerService
							.getPlayerProfileByLoginId(otherParticipantsLoginId);
					if (otherChallengeParticipantPlayerProfile != null && otherChallengeParticipantPlayerProfile.getLoginName() != null) {
						if (isAgainst) {
							ChallengesEventPublisher.publishChallengeInviteAcceptedOtherAgainstEvent(eventAdmin,
									otherParticipantsLoginId, otherChallengeParticipantPlayerProfile.getLoginName(),
									challenge.getId(), challenge.getDate().toString());
						} else {
							ChallengesEventPublisher.publishChallengeInviteAcceptedOtherForEvent(eventAdmin,
									otherParticipantsLoginId, otherChallengeParticipantPlayerProfile.getLoginName(),
									challenge.getId(), challenge.getDate().toString());
						}
					}
				}

			} else {
				logger.warn("Player with loginId " + playerLoginId + " is already taking part to the challenge!");
			}
		}

		return enhanceChallenge(challenge);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#declineChallenge(java.lang.String, java.lang.String)
	 */
	@Override
	public Challenge declineChallenge(String challengeId, String playerLoginId) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");
		Objects.requireNonNull(playerLoginId, "Login ID of player declining challenge is required!");

		Challenge challenge = getChallengeById(challengeId);

		if (challenge != null) {

			if (ChallengeStatusType.CLOSED.equals(challenge.getStatus())) {
				throw new ChallengeWrongStatusException("Cannot decline a challenge which is in status CLOSED!");
			}

			Objects.requireNonNull(challenge.getContent(), "Cannot decline a challenge without content!");

			if (!playerLoginId.equals(challenge.getContent().getChallengedPlayerLoginId())) {
				throw new ChallengeWrongRoleException("Only the challenged Player can decline the challenge.");
			}

			challenge.setStatus(ChallengeStatusType.DECLINED);

			repository.save(challenge);

			challengesIndexService.indexChallenge(challenge, false);

			// Send notifications

			// @formatter:off
			Optional<ChallengeParticipant> challengeIssuerOptional = challenge.getParticipants().stream()
					.filter(c -> c.getRole() == ChallengeParticipantRole.CHALLENGER)
					.findFirst();
			// @formatter:on			

			if (challengeIssuerOptional.isPresent()) {
				PlayerProfile challengeIssuerPlayerProfile = playerService
						.getPlayerProfileByLoginId(challengeIssuerOptional.get().getPlayerLoginId());
				if (challengeIssuerPlayerProfile != null && challengeIssuerPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeDeclinedSelfEvent(eventAdmin, playerLoginId,
							challengeIssuerPlayerProfile.getLoginName(), challenge.getId(),
							challenge.getDate().toString());
				}

				PlayerProfile challengedPlayerProfile = playerService.getPlayerProfileByLoginId(playerLoginId);
				if (challengedPlayerProfile != null && challengedPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeDeclinedOtherEvent(eventAdmin,
							challengeIssuerOptional.get().getPlayerLoginId(), challengedPlayerProfile.getLoginName(),
							challenge.getId(), challenge.getDate().toString());
				}
			}
		}

		return enhanceChallenge(challenge);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#inviteToChallenge(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public Challenge inviteToChallenge(String challengeId, String senderLoginId, String receiverLoginId) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");
		Objects.requireNonNull(senderLoginId, "Login ID of player inviting to join challenge is required!");
		Objects.requireNonNull(receiverLoginId, "Login ID of player invited to join challenge is required!");

		Challenge challenge = getChallengeById(challengeId);

		if (challenge != null) {

			if (ChallengeStatusType.CLOSED.equals(challenge.getStatus())
					|| ChallengeStatusType.DECLINED.equals(challenge.getStatus())) {
				throw new ChallengeWrongStatusException(
						"Cannot invite other Players to a closed or declined challenge!");
			}

			// @formatter:off
			Optional<ChallengeParticipant> challengeParticipantOptional = challenge.getParticipants().stream()
					.filter(b -> receiverLoginId.equals(b.getPlayerLoginId()))
					.findFirst();
			// @formatter:on

			if (challengeParticipantOptional.isPresent()) {
				logger.warn("Player with loginId " + receiverLoginId + " is already participating in the challenge!");
				return challenge;

			} else {

				// Send notifications
				PlayerProfile senderPlayerProfile = playerService.getPlayerProfileByLoginId(senderLoginId);
				if (senderPlayerProfile != null && senderPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeInviteSelfEvent(eventAdmin, senderLoginId,
							senderPlayerProfile.getLoginName(), challenge.getId(), challenge.getDate().toString());
				}

				PlayerProfile receiverPlayerProfile = playerService.getPlayerProfileByLoginId(receiverLoginId);
				if (receiverPlayerProfile != null && receiverPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeInviteOtherEvent(eventAdmin, receiverLoginId,
							receiverPlayerProfile.getLoginName(), challenge.getId(), challenge.getDate().toString());
				}
			}
		}

		return enhanceChallenge(challenge);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#resolveChallenge(java.lang.String, boolean)
	 */
	@Override
	public Challenge resolveChallenge(String challengeId, boolean isChallengeSatisfied) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");

		Challenge challenge = getChallengeById(challengeId);

		if (challenge != null) {

			if (ChallengeStatusType.CLOSED.equals(challenge.getStatus())
					|| ChallengeStatusType.DECLINED.equals(challenge.getStatus())) {
				throw new ChallengeWrongStatusException("Cannot resolve a closed or declined challenge!");
			}

			// @formatter:off
			Optional<ChallengeParticipant> challengeIssuerOptional = challenge.getParticipants().stream()
					.filter(c -> c.getRole() == ChallengeParticipantRole.CHALLENGER)
					.findFirst();
			// @formatter:on

			if (challengeIssuerOptional.isEmpty()) {
				throw new ChallengeMissingRequiredParticipantException(
						"Missing required challenge participant with role "
								+ ChallengeParticipantRole.CHALLENGER.getLiteral());
			}

			// @formatter:off
			Optional<ChallengeParticipant> challengedOptional = challenge.getParticipants().stream()
					.filter(c -> c.getRole() == ChallengeParticipantRole.CHALLENGED)
					.findFirst();
			// @formatter:on

			if (challengedOptional.isEmpty()) {
				throw new ChallengeMissingRequiredParticipantException(
						"Missing required challenge participant with role "
								+ ChallengeParticipantRole.CHALLENGED.getLiteral());
			}

			// @formatter:off
			List<ChallengeParticipant> participantsFor = challenge.getParticipants().stream()
					.filter(b -> !b.isAgainst())
					.collect(Collectors.toList());
			// @formatter:on

			// @formatter:off
			List<ChallengeParticipant> participantsAgainst = challenge.getParticipants().stream()
					.filter(b -> b.isAgainst())
					.collect(Collectors.toList());
			// @formatter:on

			ChallengeResult challengeResult = ChallengesFactory.eINSTANCE.createChallengeResult();

			if (isChallengeSatisfied) {
				challengeResult.getWinners().addAll(participantsFor);
				challengeResult.getLosers().addAll(participantsAgainst);
			} else {
				challengeResult.getWinners().addAll(participantsAgainst);
				challengeResult.getLosers().addAll(participantsFor);
			}

			challenge.setResult(challengeResult);

			challenge.setStatus(ChallengeStatusType.CLOSED);

			repository.save(challenge);

			challengesIndexService.indexChallenge(challenge, false);

			// Transfer merit points
			transferMeritPoints(challengeResult, new BigDecimal(challenge.getContent().getMeritPoints()),
					config.commission());

			// Send notifications
			PlayerProfile challengeIssuerPlayerProfile = playerService
					.getPlayerProfileByLoginId(challengeIssuerOptional.get().getPlayerLoginId());

			PlayerProfile challengedPlayerProfile = playerService
					.getPlayerProfileByLoginId(challengedOptional.get().getPlayerLoginId());

			if ((challengeIssuerPlayerProfile != null && challengeIssuerPlayerProfile.getLoginName() != null)
					&& (challengedPlayerProfile != null && challengedPlayerProfile.getLoginName() != null)) {

				if (isChallengeSatisfied) {
					ChallengesEventPublisher.publishChallengeWonEvent(eventAdmin,
							challengedOptional.get().getPlayerLoginId(), challengeIssuerPlayerProfile.getLoginName(),
							challenge.getId(), challenge.getDate().toString());

					ChallengesEventPublisher.publishChallengeLostEvent(eventAdmin,
							challengeIssuerOptional.get().getPlayerLoginId(), challengedPlayerProfile.getLoginName(),
							challenge.getId(), challenge.getDate().toString());

				} else {
					ChallengesEventPublisher.publishChallengeWonEvent(eventAdmin,
							challengeIssuerOptional.get().getPlayerLoginId(), challengedPlayerProfile.getLoginName(),
							challenge.getId(), challenge.getDate().toString());

					ChallengesEventPublisher.publishChallengeLostEvent(eventAdmin,
							challengedOptional.get().getPlayerLoginId(), challengeIssuerPlayerProfile.getLoginName(),
							challenge.getId(), challenge.getDate().toString());
				}
				
				// TODO: adjust accordingly when invites are supported
			}
		}

		return enhanceChallenge(challenge);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#closeChallenge(java.lang.String, java.lang.String)
	 */
	@Override
	public Challenge closeChallenge(String challengeId, String playerLoginId) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");
		Objects.requireNonNull(playerLoginId, "Login ID of player closing challenge is required!");

		Challenge challenge = getChallengeById(challengeId);

		if (challenge != null) {

			// @formatter:off
			Optional<ChallengeParticipant> challengeIssuerOptional = challenge.getParticipants().stream()
					.filter(c -> c.getRole() == ChallengeParticipantRole.CHALLENGER)
					.findFirst();
			// @formatter:on

			if (challengeIssuerOptional.isPresent()) {

				if (!playerLoginId.equals(challengeIssuerOptional.get().getPlayerLoginId())) {
					throw new ChallengeWrongRoleException("Only Player who issued challenge can close the challenge.");
				}

				challenge.setStatus(ChallengeStatusType.CLOSED);

				repository.save(challenge);

				challengesIndexService.indexChallenge(challenge, false);

				// Send notifications
				PlayerProfile challengedPlayerProfile = playerService
						.getPlayerProfileByLoginId(challenge.getContent().getChallengedPlayerLoginId());
				if (challengedPlayerProfile != null && challengedPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeClosedSelfEvent(eventAdmin, playerLoginId,
							challengedPlayerProfile.getLoginName(), challenge.getId(), challenge.getDate().toString());
				}

				PlayerProfile challengeIssuerPlayerProfile = playerService.getPlayerProfileByLoginId(playerLoginId);
				if (challengeIssuerPlayerProfile != null && challengeIssuerPlayerProfile.getLoginName() != null) {
					ChallengesEventPublisher.publishChallengeClosedOtherEvent(eventAdmin,
							challenge.getContent().getChallengedPlayerLoginId(),
							challengeIssuerPlayerProfile.getLoginName(), challenge.getId(),
							challenge.getDate().toString());
				}
			}
		}

		return enhanceChallenge(challenge);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#getChallengeById(java.lang.String)
	 */
	@Override
	public Challenge getChallengeById(String challengeId) {
		Objects.requireNonNull(challengeId, "Challenge ID is required!");

		Challenge challenge = repository.getEObject(ChallengesPackage.Literals.CHALLENGE, challengeId);

		return enhanceChallenge(challenge);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#getChallengesByPlayerLoginId(java.lang.String, java.util.Optional)
	 */
	@Override
	public List<Challenge> getChallengesByPlayerLoginId(String loginId, Optional<String> courseIdOptional) {
		Objects.requireNonNull(loginId, "Cannot retrieve challenges for Player with null loginId!");

		List<Challenge> challenges = challengesSearchService.getChallengesByPlayerLoginId(loginId);

		if (courseIdOptional.isPresent()) {
			return challengesOnCourse(challenges, courseIdOptional.get());
		}

		return enhanceChallenges(challenges);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#getOpenChallengesByPlayerLoginId(java.lang.String, java.util.Optional)
	 */
	@Override
	public List<Challenge> getOpenChallengesByPlayerLoginId(String loginId, Optional<String> courseIdOptional) {
		Objects.requireNonNull(loginId, "Cannot retrieve challenges for Player with null loginId!");

		List<Challenge> challenges = challengesSearchService.getOpenChallengesByPlayerLoginId(loginId);

		if (courseIdOptional.isPresent()) {
			return challengesOnCourse(challenges, courseIdOptional.get());
		}

		return enhanceChallenges(challenges);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#createChallengeSearchResult(java.util.List)
	 */
	@Override
	public ChallengeSearchResult createChallengeSearchResult(List<Challenge> challenges) {
		Objects.requireNonNull(challenges, "List of challenges is required!");

		ChallengeSearchResult result = ChallengesFactory.eINSTANCE.createChallengeSearchResult();
		result.getResult().addAll(challenges);

		return result;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.challenges.service.api.ChallengesService#createPossibleChallengerResult(java.util.List)
	 */
	@Override
	public PossibleChallengeParticipantResult createPossibleChallengeParticipantResult(List<PossibleChallengeParticipant> possibleChallengeParticipants) {
		Objects.requireNonNull(possibleChallengeParticipants, "List of possible challenge participants is required!");

		PossibleChallengeParticipantResult result = ChallengesFactory.eINSTANCE.createPossibleChallengeParticipantResult();
		result.getResult().addAll(possibleChallengeParticipants);

		return result;
	}
	
	private Pair<MeritPointsAccount, MeritPointsAccount> transferMeritPoints(ChallengeResult challengeResult,
			BigDecimal amount, double commission) {
		// TODO: adjust accordingly when invites are supported
		ChallengeParticipant winner = challengeResult.getWinners().get(0);
		ChallengeParticipant loser = challengeResult.getLosers().get(0);
		
		// @formatter:off
		return meritPointsService.transferWithCommission(
				loser.getPlayerLoginId(),
				winner.getPlayerLoginId(),
				amount, 
				commission, 
				String.format(PAYEE_TRANSACTION_DESCRIPTION, amount, commission), 
				String.format(RECEIVER_TRANSACTION_DESCRIPTION, amount, commission));
		// @formatter:on
	}

	private List<Player> playersWithOpenScorecard(List<Player> players, String courseId) {
		// @formatter:off
		return players.stream()
				.filter(p -> p.getPlayerScorecards().stream()
						.anyMatch(s -> s.getEnd() == null && s.isNoReturn() == false && s.getCourseId().equals(courseId)))
				.collect(Collectors.toList());
		// @formatter:on
	}

	private List<Challenge> challengesOnCourse(List<Challenge> challenges, String courseId) {
		// @formatter:off
		return challenges.stream()
				.filter(c -> c.getContent().getCourseId().equals(courseId))
				.collect(Collectors.toList());
		// @formatter:on
	}

	private List<Challenge> enhanceChallenges(List<Challenge> challenges) {
		// @formatter:off
		return challenges.stream()
				.map((ch) -> enhanceChallenge(ch))
				.collect(Collectors.toList());
		// @formatter:on
	}

	private Challenge enhanceChallenge(Challenge challenge) {
		Challenge challengeCopy = EcoreUtil.copy(challenge);

		if (challengeCopy.getContent() != null && challengeCopy.getContent().getCourseId() != null) {
			try {
				com.playertour.backend.api.GolfCourse golfCourse = apiGolfCourseService
						.getCourseById(challengeCopy.getContent().getCourseId());
				challengeCopy.getContent().setCourseName(golfCourse.getName());

			} catch (UnknownTransformationException e) {
				logger.error(e.getMessage());
			}
		}

		if (challengeCopy.getParticipants() != null && !challengeCopy.getParticipants().isEmpty()) {
			List<ChallengeParticipant> enhancedParticipants = enhanceChallengeParticipants(challengeCopy.getParticipants());
			challengeCopy.getParticipants().clear();
			challengeCopy.getParticipants().addAll(enhancedParticipants);
		}

		if (challengeCopy.getResult() != null) {

			if (challengeCopy.getResult().getWinners() != null && !challengeCopy.getResult().getWinners().isEmpty()) {

				List<ChallengeParticipant> enhancedWinners = enhanceChallengeParticipants(challengeCopy.getResult().getWinners());
				challengeCopy.getResult().getWinners().clear();
				challengeCopy.getResult().getWinners().addAll(enhancedWinners);
			}

			if (challengeCopy.getResult().getLosers() != null && !challengeCopy.getResult().getLosers().isEmpty()) {

				List<ChallengeParticipant> enhancedLosers = enhanceChallengeParticipants(challengeCopy.getResult().getLosers());
				challengeCopy.getResult().getLosers().clear();
				challengeCopy.getResult().getLosers().addAll(enhancedLosers);
			}
		}

		return challengeCopy;
	}

	private List<ChallengeParticipant> enhanceChallengeParticipants(List<ChallengeParticipant> participants) {
		// @formatter:off
		return participants.stream()
				.map((ch) -> enhanceChallengeParticipant(ch))
				.collect(Collectors.toList());
		// @formatter:on
	}

	private ChallengeParticipant enhanceChallengeParticipant(ChallengeParticipant participant) {
		ChallengeParticipant participantCopy = EcoreUtil.copy(participant);

		PlayerProfile participantPlayerProfile = playerService.getPlayerProfileByLoginId(participant.getPlayerLoginId());

		if (participantPlayerProfile != null) {

			if (participantPlayerProfile.getLoginName() != null) {
				participantCopy.setPlayerLoginName(participantPlayerProfile.getLoginName());
			}

			if (participantPlayerProfile.getName() != null) {
				participantCopy.setPlayerName(participantPlayerProfile.getName());
			}
		}

		return participantCopy;
	}

	// TODO: someday / maybe: use transformations instead ?
	private List<PossibleChallengeParticipant> repackagePlayersToPossibleChallengeParticipants(List<Player> players) {
		// @formatter:off
		return players.stream()
				.map((p) -> repackagePlayerToPossibleChallengeParticipant(p))
				.collect(Collectors.toList());
		// @formatter:on
	}

	private PossibleChallengeParticipant repackagePlayerToPossibleChallengeParticipant(Player player) {
		PossibleChallengeParticipant participant = ChallengesFactory.eINSTANCE.createPossibleChallengeParticipant();

		participant.setPlayerLoginId(player.getLoginId());
		participant.setPlayerLoginName(player.getProfile().getLoginName());
		participant.setPlayerName(player.getProfile().getName());

		return participant;
	}
}
