/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.challenges.service.tests;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatExceptionOfType;

import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.challenges.model.challenges.Challenge;
import com.playertour.backend.challenges.model.challenges.ChallengeContent;
import com.playertour.backend.challenges.model.challenges.ChallengeStatusType;
import com.playertour.backend.challenges.model.challenges.ChallengesFactory;
import com.playertour.backend.challenges.model.challenges.PossibleChallengeParticipant;
import com.playertour.backend.challenges.service.api.ChallengeWrongRoleException;
import com.playertour.backend.challenges.service.api.ChallengeWrongStatusException;
import com.playertour.backend.challenges.service.api.ChallengesService;
import com.playertour.backend.challenges.service.tests.helper.ChallengesTestHelper;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount;
import com.playertour.backend.meritpoints.service.api.MeritPointsService;
import com.playertour.backend.player.model.player.PlayerPackage;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class ChallengesIntegrationTest {

	@InjectBundleContext
	BundleContext bundleContext;
	
	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware) {

		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ServiceReference<ChallengesService> challengesServiceReference = challengesServiceAware.getServiceReference();
		assertThat(challengesServiceReference).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerServiceReference = playerServiceAware.getServiceReference();
		assertThat(playerServiceReference).isNotNull();
		
		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		ServiceReference<MeritPointsService> meritPointsServiceReference = meritPointsServiceAware
				.getServiceReference();
		assertThat(meritPointsServiceReference).isNotNull();		
	}
	
	@Test
	public void testGetAllPossibleChallengeParticipants(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			
		
		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and 2 of the 3 Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Check that the possible possible challenge participants for the GolfCourse are only the ones with an open scorecard
		List<PossibleChallengeParticipant> possibleChallengeParticipants = challengesService.getAllPossibleChallengeParticipants(courseMongoId);
		assertThat(possibleChallengeParticipants).isNotNull();
		assertThat(possibleChallengeParticipants).isNotEmpty();
		assertThat(possibleChallengeParticipants).hasSize(2);
	}	
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for that GolfCourse just for 2 of the 3 Players
	 * 4. Check that the possible challenge participants for that GolfCourse are only the 2 Players with open PlayerScorecard
	 * @throws InterruptedException 
	 */
	@Test
	public void testGetPossibleChallengeParticipants(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			
		
		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and 2 of the 3 Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Check that the possible challenge participants for the GolfCourse are only the ones with an open scorecard
		List<PossibleChallengeParticipant> possibleChallengeParticipants = challengesService.getPossibleChallengeParticipants(courseMongoId, "mario");
		assertThat(possibleChallengeParticipants).isNotNull();
		assertThat(possibleChallengeParticipants).isNotEmpty();
		assertThat(possibleChallengeParticipants).hasSize(1);
		assertThat(possibleChallengeParticipants.get(0).getPlayerLoginId()).isEqualTo(loginId1);
		
		possibleChallengeParticipants = challengesService.getPossibleChallengeParticipants(courseMongoId, "verdi");
		assertThat(possibleChallengeParticipants).isNotNull();
		assertThat(possibleChallengeParticipants).isNotEmpty();
		assertThat(possibleChallengeParticipants).hasSize(1);
		assertThat(possibleChallengeParticipants.get(0).getPlayerLoginId()).isEqualTo(loginId2);
		
		possibleChallengeParticipants = challengesService.getPossibleChallengeParticipants(courseMongoId, "viola");
		assertThat(possibleChallengeParticipants).isNotNull();
		assertThat(possibleChallengeParticipants).isEmpty();
	}
	
	/**
	 * 1. Create 2 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Check created Challenge
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testOpenChallenge(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			
		
		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		
		// 5. Check open Challenge
		assertThat(challenge).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		assertThat(challenge.getDate()).isNotNull();
		assertThat(challenge.getContent()).isNotNull();
		assertThat(challenge.getParticipants()).hasSize(2);
	}
	
	@Test
	public void testGetChallengeById(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			
		
		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		assertThat(challenge.getDate()).isNotNull();
		assertThat(challenge.getContent()).isNotNull();
		assertThat(challenge.getParticipants()).hasSize(2);		
		
		// 5. Retrieve Challenge
		challenge = challengesService.getChallengeById(challenge.getId());
		assertThat(challenge).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		assertThat(challenge.getDate()).isNotNull();
		assertThat(challenge.getContent()).isNotNull();
		assertThat(challenge.getParticipants()).hasSize(2);
	}
	
	@Test
	public void testGetOpenChallengesByPlayerLoginId(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			
		
		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		assertThat(challenge.getDate()).isNotNull();
		assertThat(challenge.getContent()).isNotNull();
		assertThat(challenge.getParticipants()).hasSize(2);		
		
		// 5. Retrieve list of Challenges for player who opened challenge
		List<Challenge> challenges = challengesService.getOpenChallengesByPlayerLoginId(loginId1, Optional.empty());
		assertThat(challenges).isNotNull();
		assertThat(challenges).hasSize(1);
	}
	
	/**
	 * 1. Create 2 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 accept the Challenge. Check that the Challenge has been properly updated
	 * 6. Check that Player 1 got a Notification when Player 2 has accepted the Challenge
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testAcceptChallengeChallengedPlayer(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		Thread.sleep(2000);
		
		PlayerProfile playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(2);
		
		// 5. Player 2 accept the Challenge
		challenge = challengesService.acceptChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.ACCEPTED);
		
		Thread.sleep(2000);
		
		// 6. Check that Player 1 got notified 
		playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(3);
	}
	
	/**
	 * 1. Create 2 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 1 closes the Challenge.
	 * 6. Player 2 accepts the Challenge. Check that an Exception is thrown because one cannot accept a CLOSED Challenge
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testInvalidAcceptChallengeClosed(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 1 closes the Challenge
		challenge = challengesService.closeChallenge(challenge.getId(), loginId1);
		String challengeId = challenge.getId();
		
		// 6. Player 2 accept the Challenge
		assertThatExceptionOfType(ChallengeWrongStatusException.class).isThrownBy(() -> {
			challengesService.acceptChallenge(challengeId, loginId2);
		});
	}
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 accept the Challenge. 
	 * 6. Player 1 invites Player 3 to the Challenge
	 * 7. Player 3 accepts. Check that an Exception is thrown because only challenged player can accept the Challenge.
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testInvalidAcceptChallengeOtherPlayer(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2. Check Player 2 get notified
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 2 accept the Challenge. Check that Player 1 get notified.
		challenge = challengesService.acceptChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.ACCEPTED);
		
		// 6. Player 1 invites Player 3 to the Challenge. Check that Player 3 get notified.
		challenge = challengesService.inviteToChallenge(challenge.getId(), loginId1, loginId3);	
		assertThat(challenge.getParticipants()).hasSize(2);
		String challengeId = challenge.getId();
		
		// 7. Player 3 accepts the Challenge. Check that Player 1 and 2 get notified.
		assertThatExceptionOfType(ChallengeWrongRoleException.class).isThrownBy(() -> {
			challengesService.acceptChallenge(challengeId, loginId3);
		});
	}
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 accept the Challenge. 
	 * 6. Player 1 invites Player 3 to the Challenge
	 * 7. Player 3 joins. Check Notifications
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testJoinChallengeOtherPlayerNotification(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2. Check Player 2 get notified
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		Thread.sleep(2000);
		
		PlayerProfile playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(2);
		
		PlayerProfile playerProfile2 = playerService.getPlayerProfileByLoginId(loginId2);
		assertThat(playerProfile2).isNotNull();
		assertThat(playerProfile2.getNotifications()).hasSize(2);
		
		// 5. Player 2 accept the Challenge. Check that Player 1 get notified.
		challenge = challengesService.acceptChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.ACCEPTED);
		
		Thread.sleep(2000);
		
		playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(3);
		
		playerProfile2 = playerService.getPlayerProfileByLoginId(loginId2);
		assertThat(playerProfile2).isNotNull();
		assertThat(playerProfile2.getNotifications()).hasSize(3);
		
		PlayerProfile playerProfile3 = playerService.getPlayerProfileByLoginId(loginId3);
		assertThat(playerProfile3).isNotNull();
		assertThat(playerProfile3.getNotifications()).hasSize(1);
		
		// 6. Player 1 invites Player 3 to the Challenge. Check that Player 3 get notified.
		challenge = challengesService.inviteToChallenge(challenge.getId(), loginId1, loginId3);	
		assertThat(challenge.getParticipants()).hasSize(2);
		
		Thread.sleep(2000);
		
		playerProfile3 = playerService.getPlayerProfileByLoginId(loginId3);
		assertThat(playerProfile3).isNotNull();
		assertThat(playerProfile3.getNotifications()).hasSize(2);
		
		// 7. Player 3 accepts the Challenge. Check that Player 1 and 2 get notified.
		challenge = challengesService.joinChallenge(challenge.getId(), loginId3, false);
		assertThat(challenge.getParticipants()).hasSize(3);
		
		Thread.sleep(2000);
		
		playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(5);
		
		playerProfile2 = playerService.getPlayerProfileByLoginId(loginId2);
		assertThat(playerProfile2).isNotNull();
		assertThat(playerProfile2.getNotifications()).hasSize(4);
		
		playerProfile3 = playerService.getPlayerProfileByLoginId(loginId3);
		assertThat(playerProfile3).isNotNull();
		assertThat(playerProfile3.getNotifications()).hasSize(3);		
	}

	/**
	 * 1. Create 2 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 decline the Challenge. Check that the Challenge has been properly updated
	 * 6. Check that Player 1 got a Notification when Player 2 has declined the Challenge
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testDeclineChallengeChallengedPlayer(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		Thread.sleep(2000);
		
		PlayerProfile playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(2);
		
		// 5. Player 2 decline the Challenge
		challenge = challengesService.declineChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.DECLINED);
		
		Thread.sleep(2000);
		
		// 6. Check that Player 1 got notified 
		playerProfile1 = playerService.getPlayerProfileByLoginId(loginId1);
		assertThat(playerProfile1).isNotNull();
		assertThat(playerProfile1.getNotifications()).hasSize(3);
	}
	
	/**
	 * 1. Create 2 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 1 closes the Challenge.
	 * 6. Player 2 declines the Challenge. Check that an Exception is thrown because one cannot decline a CLOSED Challenge
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testInvalidDeclineChallengeClosed(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 1 closes the Challenge
		challenge = challengesService.closeChallenge(challenge.getId(), loginId1);
		String challengeId = challenge.getId();
		
		// 6. Player 2 accept the Challenge
		assertThatExceptionOfType(ChallengeWrongStatusException.class).isThrownBy(() -> {
			challengesService.declineChallenge(challengeId, loginId2);
		});
	}
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 accept the Challenge. 
	 * 6. Player 1 invites Player 3 to the Challenge
	 * 7. Player 3 declines. Check that an Exception is thrown because only challenged player can decline the Challenge.
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testInvalidDeclineChallengeOtherPlayer(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();

		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2. Check Player 2 get notified
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 2 accept the Challenge. Check that Player 1 get notified.
		challenge = challengesService.acceptChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.ACCEPTED);
		
		// 6. Player 1 invites Player 3 to the Challenge. Check that Player 3 get notified.
		challenge = challengesService.inviteToChallenge(challenge.getId(), loginId1, loginId3);	
		assertThat(challenge.getParticipants()).hasSize(2);
		String challengeId = challenge.getId();
		
		// 7. Player 3 accepts the Challenge. Check that Player 1 and 2 get notified.
		assertThatExceptionOfType(ChallengeWrongRoleException.class).isThrownBy(() -> {
			challengesService.declineChallenge(challengeId, loginId3);
		});
	}
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 declines the Challenge. 
	 * 6. Player 1 invites Player 3 to the Challenge. Check that an Exception is thrown because we cannot invite to a Challenge that has been DECLINED.
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testInviteToChallengeInvalid(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2. 
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 2 declines the Challenge. 
		challenge = challengesService.declineChallenge(challenge.getId(), loginId2);
		String challengeId = challenge.getId();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.DECLINED);		
		
		// 6. Player 1 invites Player 3 to the Challenge. Check that an exception is thrown.
		assertThatExceptionOfType(ChallengeWrongStatusException.class).isThrownBy(() -> {
			challengesService.inviteToChallenge(challengeId, loginId1, loginId3);
		});
	}
	
	@Test
	public void testResolveChallenge(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();
		
		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();		

		// 1. Create 2 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Deposit merit points 
		BigDecimal depositAmount = BigDecimal.TEN;
		String depositDescription = String.format("Deposit of %.2f merit points", depositAmount);
		
		MeritPointsAccount meritPointsAccount1 = meritPointsService.deposit(loginId1, depositAmount, depositDescription);
		assertThat(meritPointsAccount1).isNotNull();
		assertThat(meritPointsAccount1.getUserID()).isEqualTo(loginId1);
		assertThat(meritPointsAccount1.getBalance()).isEqualTo(depositAmount);
		assertThat(meritPointsAccount1.getTransactions()).isNotNull();
		assertThat(meritPointsAccount1.getTransactions()).isNotEmpty();
		assertThat(meritPointsAccount1.getTransactions()).hasSize(1);
		
		MeritPointsAccount meritPointsAccount2 = meritPointsService.deposit(loginId2, depositAmount, depositDescription);
		assertThat(meritPointsAccount2).isNotNull();
		assertThat(meritPointsAccount2.getUserID()).isEqualTo(loginId2);
		assertThat(meritPointsAccount2.getBalance()).isEqualTo(depositAmount);
		assertThat(meritPointsAccount2.getTransactions()).isNotNull();
		assertThat(meritPointsAccount2.getTransactions()).isNotEmpty();
		assertThat(meritPointsAccount2.getTransactions()).hasSize(1);
		
		Thread.sleep(2000);
		
		// 4. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 5. Player 1 opens a Challenge against Player 2. 
		int meritPointsStake = 5;
		
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(meritPointsStake);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		Thread.sleep(2000);
		
		// 6. Player 2 accepts Challenge. 
		challenge = challengesService.acceptChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.ACCEPTED);
		
		Thread.sleep(2000);
		
	 	// 7. Resolve Challenge. Check that ChallengeResult is properly created
		challenge = challengesService.resolveChallenge(challenge.getId(), true);
		assertThat(challenge.getResult()).isNotNull();
		assertThat(challenge.getResult().getWinners()).hasSize(1);
		assertThat(challenge.getResult().getLosers()).hasSize(1);
		
		Thread.sleep(2000);
		
		// 8. Verify merit points were transfered
		double commission = 0.10;
		
		BigDecimal meritPointsAccountBalance1 = meritPointsService.getBalance(loginId1);
		assertThat(meritPointsAccountBalance1).isNotNull();
		assertThat(meritPointsAccountBalance1).isEqualTo(depositAmount.subtract(new BigDecimal(meritPointsStake + (meritPointsStake * commission))));
		
		BigDecimal meritPointsAccountBalance2 = meritPointsService.getBalance(loginId2);
		assertThat(meritPointsAccountBalance2).isNotNull();
		assertThat(meritPointsAccountBalance2).isEqualTo(depositAmount.add(new BigDecimal(meritPointsStake - (meritPointsStake * commission))));
	}	
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 2 accept the Challenge. 
	 * 6. Player 1 invites Player 3 to the Challenge
	 * 7. Player 3 accepts. 
	 * 8. Resolve Challenge. Check that ChallengeResult is properly created
	 * 
	 * @throws InterruptedException
	 */
	@Disabled("Invites are not supported yet")
	@Test
	public void testResolveChallengeWithJoin(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2. 
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 2 accept the Challenge. 
		challenge = challengesService.acceptChallenge(challenge.getId(), loginId2);
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.ACCEPTED);
		
		// 6. Player 1 invites Player 3 to the Challenge. 
		challenge = challengesService.inviteToChallenge(challenge.getId(), loginId1, loginId3);	
		assertThat(challenge.getParticipants()).hasSize(2);
		
		// 7. Player 3 accepts the Challenge. 
		challenge = challengesService.joinChallenge(challenge.getId(), loginId3, false);
		assertThat(challenge.getParticipants()).hasSize(3);
		
	 	// 8. Resolve Challenge. Check that ChallengeResult is properly created
		challenge = challengesService.resolveChallenge(challenge.getId(), true);
		assertThat(challenge.getResult()).isNotNull();
		assertThat(challenge.getResult().getWinners()).hasSize(2);
		assertThat(challenge.getResult().getLosers()).hasSize(1);
	}
	
	/**
	 * 1. Create 3 Players
	 * 2. Create a GolfCourse
	 * 3. Open a PlayerScorecard for each of the Player
	 * 4. Player 1 launches a Challenge to Player 2
	 * 5. Player 3 launches another Challenge to Player 2 
	 * 6. Retrieve Challenges for Player 1 and Player 3 and check they are correct
	 * 
	 * @throws InterruptedException
	 */
	@Test
	public void testGetChallengeByLoginId(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<ChallengesService> challengesServiceAware,
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> scorecardAware, 
		@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseAware) throws InterruptedException {
		
		assertThat(challengesServiceAware.getServices()).hasSize(1);
		ChallengesService challengesService = challengesServiceAware.getService();
		assertThat(challengesService).isNotNull();
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		PlayerScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(courseAware.getServices()).hasSize(1);	
		CourseService courseService = courseAware.getService();
		assertThat(courseService).isNotNull();			

		// 1. Create 3 Players
		String loginId1 = UUID.randomUUID().toString();
		String loginName1 = "mario.rossi";
		PlayerProfile profile1 = playerService.getPlayerProfile(loginId1, loginName1);	
		assertThat(profile1).isNotNull();
		
		String loginId2 = UUID.randomUUID().toString();
		String loginName2 = "carlo.verdi";
		PlayerProfile profile2 = playerService.getPlayerProfile(loginId2, loginName2);	
		assertThat(profile2).isNotNull();
		
		String loginId3= UUID.randomUUID().toString();
		String loginName3 = "paolo.viola";
		PlayerProfile profile3 = playerService.getPlayerProfile(loginId3, loginName3);	
		assertThat(profile3).isNotNull();
		
		// 2. Create a GolfCourse		
		GolfCourse golfCourse = ChallengesTestHelper.createTestGolfCourse("abcdef");
		golfCourse = courseService.saveCourse(golfCourse);
		String courseMongoId = golfCourse.getId();
		assertThat(courseMongoId).isNotNull();
		
		Thread.sleep(2000);
		
		// 3. Open a PlayerScorecard for the GolfCourse and both the Players
		PlayerScorecard playerScorecard1 = scorecardService.openScorecard(loginId1, courseMongoId);
		assertThat(playerScorecard1).isNotNull();
		
		PlayerScorecard playerScorecard2 = scorecardService.openScorecard(loginId2, courseMongoId);
		assertThat(playerScorecard2).isNotNull();
		
		Thread.sleep(2000);
		
		// 4. Player 1 opens a Challenge against Player 2. 
		ChallengeContent challengeContent = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent.setMeritPoints(10);
		challengeContent.setChallengedPlayerLoginId(loginId2);
		challengeContent.setHoleNumber(1);
		challengeContent.setMinStrokesNeeded(5);
		challengeContent.setCourseId(courseMongoId);
		
		Challenge challenge = challengesService.openChallenge(loginId1, challengeContent);
		assertThat(challenge).isNotNull();
		assertThat(challenge.getId()).isNotNull();
		assertThat(challenge.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 5. Player 3 opens a Challenge against Player 2. 
		ChallengeContent challengeContent2 = ChallengesFactory.eINSTANCE.createChallengeContent();
		challengeContent2.setMeritPoints(10);
		challengeContent2.setChallengedPlayerLoginId(loginId2);
		challengeContent2.setHoleNumber(1);
		challengeContent2.setMinStrokesNeeded(5);
		challengeContent2.setCourseId(courseMongoId);
		
		Challenge challenge2 = challengesService.openChallenge(loginId3, challengeContent);
		assertThat(challenge2).isNotNull();
		assertThat(challenge2.getId()).isNotNull();
		assertThat(challenge2.getStatus()).isEqualTo(ChallengeStatusType.OPEN);
		
		// 6. Retrieve Challenges for Player 1 and Player 3 and check they are correct
		List<Challenge> challengesPlayer1 = challengesService.getChallengesByPlayerLoginId(loginId1, Optional.empty());
		assertThat(challengesPlayer1).hasSize(1);
		assertThat(challengesPlayer1.get(0).getId()).isEqualTo(challenge.getId());
		
		List<Challenge> challengesPlayer3 = challengesService.getChallengesByPlayerLoginId(loginId3, Optional.empty());
		assertThat(challengesPlayer3).hasSize(1);
		assertThat(challengesPlayer3.get(0).getId()).isEqualTo(challenge2.getId());
	}
	
	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=CourseIndexService)") ServiceAware<CommonIndexService> courseIndexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=ChallengesIndexService)") ServiceAware<CommonIndexService> challengesIndexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> playerIndexAware) {
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(GolfCoursePackage.Literals.GOLF_COURSE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
		
		assertThat(playerIndexAware.getServices()).hasSize(1);	
		CommonIndexService playerIndexService = playerIndexAware.getService();
		assertThat(playerIndexService).isNotNull();		
		
		playerIndexService.resetIndex();
		
		assertThat(courseIndexAware.getServices()).hasSize(1);	
		CommonIndexService courseIndexService = courseIndexAware.getService();
		assertThat(courseIndexService).isNotNull();		
		
		courseIndexService.resetIndex();
		
		assertThat(challengesIndexAware.getServices()).hasSize(1);
		CommonIndexService challengesIndexService = challengesIndexAware.getService();
		assertThat(challengesIndexService).isNotNull();		
		
		challengesIndexService.resetIndex();
	}
	
	@AfterAll
	public static void brutalClean(@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
