/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.config.firebaseremoteconfig.service.impl;

import java.io.FileInputStream;
import java.io.IOException;
import java.time.DateTimeException;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.component.propertytypes.ServiceRanking;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.remoteconfig.Condition;
import com.google.firebase.remoteconfig.FirebaseRemoteConfig;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigException;
import com.google.firebase.remoteconfig.Parameter;
import com.google.firebase.remoteconfig.ParameterGroup;
import com.google.firebase.remoteconfig.ParameterValue;
import com.google.firebase.remoteconfig.Template;
import com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService;

@Component(name = "FirebaseRemoteConfigService", scope = ServiceScope.SINGLETON, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "FirebaseRemoteConfigService")
@Designate(ocd = FirebaseRemoteConfigServiceImpl.Config.class)
@ServiceRanking(Integer.MAX_VALUE - 500)
public class FirebaseRemoteConfigServiceImpl implements FirebaseRemoteConfigService {
	
	private static final DateTimeFormatter OFFSET_DATE_TIME_FORMATTER = DateTimeFormatter.ISO_OFFSET_DATE_TIME;
	private static final DateTimeFormatter LOCAL_DATE_TIME_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE_TIME;
	private static final ZoneId DEFAULT_DATE_TIME_ZONE_ID = ZoneId.of("Etc/GMT");	

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	private Config config;

	private FirebaseApp firebaseApp;

	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {

		String firebaseApplicationCredentialsPathEnv();

		String firebaseProjectId();

		String firebaseApplicationName();
	}

	@Activate
	public void activate(Config config) {
		this.config = config;

		try {
			firebaseApp = initializeFirebase();
			
		} catch (IOException e) {
			e.printStackTrace();
			logger.error("Exception was thrown while initializing Firebase app!");
		}
	}

	@Deactivate
	public void deactivate() {
		if (firebaseApp != null) {
			firebaseApp.delete();
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getParameterGroup(java.lang.String)
	 */
	@Override
	public Optional<ParameterGroup> getParameterGroup(String parameterGroupName) {
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");

		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);

			Template template = firebaseRemoteConfig.getTemplate(); // current active template is fetched each time, so
																	// relevant configuration can be modified without
																	// having to restart application

			Map<String, ParameterGroup> parameterGroups = template.getParameterGroups();

			if (parameterGroups.containsKey(parameterGroupName)) {
				return Optional.of(parameterGroups.get(parameterGroupName));
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}

		return Optional.empty();
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getParameter(java.lang.String)
	 */
	@Override
	public Optional<Parameter> getParameter(String parameterName) {
		Objects.requireNonNull(parameterName, "Parameter name is required!");

		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);

			Template template = firebaseRemoteConfig.getTemplate(); // current active template is fetched each time, so
																	// relevant configuration can be modified without
																	// having to restart application
			logger.debug("Retrieved current active remote configuration template (version: "
					+ template.getVersion().getVersionNumber() + "; eTag: " + template.getETag() + ")");

			Map<String, Parameter> parameters = template.getParameters();
			if (!parameters.isEmpty() && parameters.containsKey(parameterName)) {
				return Optional.of(parameters.get(parameterName));
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}

		return Optional.empty();
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getParameter(java.lang.String, java.lang.String)
	 */
	@Override
	public Optional<Parameter> getParameter(String parameterGroupName, String parameterName) {
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");
		Objects.requireNonNull(parameterName, "Parameter name is required!");

		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);

			Template template = firebaseRemoteConfig.getTemplate(); // current active template is fetched each time, so
																	// relevant configuration can be modified without
																	// having to restart application

			Map<String, ParameterGroup> parameterGroups = template.getParameterGroups();

			if (parameterGroups.containsKey(parameterGroupName)
					&& !parameterGroups.get(parameterGroupName).getParameters().isEmpty()) {
				Map<String, Parameter> parameterGroupParameters = parameterGroups.get(parameterGroupName)
						.getParameters();
				if (parameterGroupParameters.containsKey(parameterName)) {
					return Optional.of(parameterGroupParameters.get(parameterName));
				}
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}

		return Optional.empty();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getBooleanParameterValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public Optional<Boolean> getBooleanParameterValue(String parameterGroupName, String parameterName,
			String conditionalValueName) {
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");
		Objects.requireNonNull(parameterName, "Parameter name is required!");
		Objects.requireNonNull(conditionalValueName, "Conditional value name is required!");

		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);

			Template template = firebaseRemoteConfig.getTemplate(); // current active template is fetched each time, so
																	// relevant configuration can be modified without
																	// having to restart application

			Map<String, ParameterGroup> parameterGroups = template.getParameterGroups();

			if (parameterGroups.containsKey(parameterGroupName)
					&& !parameterGroups.get(parameterGroupName).getParameters().isEmpty()) {
				Map<String, Parameter> parameterGroupParameters = parameterGroups.get(parameterGroupName)
						.getParameters();
				if (parameterGroupParameters.containsKey(parameterName)) {

					Parameter parameter = parameterGroupParameters.get(parameterName);
					if (parameter.getConditionalValues().isEmpty()
							|| !parameter.getConditionalValues().containsKey(conditionalValueName)) {
						if (parameter.getDefaultValue() instanceof ParameterValue.Explicit) {
							return Optional.of(Boolean
									.valueOf(((ParameterValue.Explicit) parameter.getDefaultValue()).getValue()));
						}
					}

					ParameterValue parameterValue = parameter.getConditionalValues().get(conditionalValueName);
					if (parameterValue instanceof ParameterValue.Explicit) {
						return Optional.of(Boolean.valueOf(((ParameterValue.Explicit) parameterValue).getValue()));
					}
				}
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}

		return Optional.empty();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getStringParameterValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public Optional<String> getStringParameterValue(String parameterGroupName, String parameterName,
			String conditionalValueName) {
		// TODO: implement when needed; see com.playertour.backend.config.firebaseremoteconfig.service.impl.FirebaseRemoteConfigServiceImpl.getBooleanParameter(String, String, String)
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getNumberParameterValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public Optional<Number> getNumberParameterValue(String parameterGroupName, String parameterName,
			String conditionalValueName) {
		// TODO: implement when needed; see com.playertour.backend.config.firebaseremoteconfig.service.impl.FirebaseRemoteConfigServiceImpl.getBooleanParameter(String, String, String)
		return null;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#getDoubleParameterValue(java.lang.String, java.lang.String)
	 */
	@Override
	public Optional<Double> getDoubleParameterValue(String parameterGroupName, String parameterName) {
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");
		Objects.requireNonNull(parameterName, "Parameter name is required!");
		
		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);

			Template template = firebaseRemoteConfig.getTemplate(); // current active template is fetched each time, so
																	// relevant configuration can be modified without
																	// having to restart application

			Map<String, ParameterGroup> parameterGroups = template.getParameterGroups();

			if (parameterGroups.containsKey(parameterGroupName)
					&& !parameterGroups.get(parameterGroupName).getParameters().isEmpty()) {
				Map<String, Parameter> parameterGroupParameters = parameterGroups.get(parameterGroupName)
						.getParameters();
				if (parameterGroupParameters.containsKey(parameterName)) {

					Parameter parameter = parameterGroupParameters.get(parameterName);
					
					if (parameter.getDefaultValue() instanceof ParameterValue.Explicit) {
						return Optional.of(Double
								.valueOf(((ParameterValue.Explicit) parameter.getDefaultValue()).getValue()));
					}					
				}
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}

		return Optional.empty();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setBooleanParameterValue(java.lang.String, java.lang.String, java.lang.String, boolean, boolean)
	 */
	@Override
	public boolean setBooleanParameterValue(String parameterGroupName, String parameterName, String conditionalValueName,
			boolean parameterValue, boolean parameterDefaultValue) {
		
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");
		Objects.requireNonNull(parameterName, "Parameter name is required!");
		Objects.requireNonNull(conditionalValueName, "Conditional value name is required!");		

		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);
			
			Template template = firebaseRemoteConfig.getTemplate();
			
			Optional<ParameterGroup> parameterGroupOptional = getParameterGroup(template, parameterGroupName);
			ParameterGroup parameterGroup = new ParameterGroup();
			if (parameterGroupOptional.isPresent()) {
				parameterGroup = parameterGroupOptional.get();
			}
			
			Optional<Parameter> parameterOptional = getParameter(parameterGroup, parameterName);
			Parameter parameter = new Parameter();
			if (parameterOptional.isPresent()) {
				parameter = parameterOptional.get();
			}
			
			parameter = parameter.setDefaultValue(ParameterValue.of(String.valueOf(parameterDefaultValue)));
			
			Map<String, ParameterValue> parameterConditionalValues = new HashMap<String, ParameterValue>();
			parameterConditionalValues.put(conditionalValueName, ParameterValue.of(String.valueOf(parameterValue)));
			parameter.setConditionalValues(parameterConditionalValues);
			
			parameterGroup.getParameters().put(parameterName, parameter);
			
			template.getParameterGroups().put(parameterGroupName, parameterGroup);
			
			Template publishedTemplate = firebaseRemoteConfig.publishTemplate(template);
			if (publishedTemplate != null) {
				return true;
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}
		
		return false;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setStringParameterValue(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public boolean setStringParameterValue(String parameterGroupName, String parameterName, String conditionalValueName,
			String parameterValue, String parameterDefaultValue) {
		// TODO: implement when needed; see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setBooleanParameterValue(java.lang.String, java.lang.String, java.lang.String, boolean, boolean)
		return false;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setNumberParameterValue(java.lang.String, java.lang.String, java.lang.String, java.lang.Number, java.lang.Number)
	 */
	@Override
	public boolean setNumberParameterValue(String parameterGroupName, String parameterName, String conditionalValueName,
			Number parameterValue, Number parameterDefaultValue) {
		// TODO: implement when needed; see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setBooleanParameterValue(java.lang.String, java.lang.String, java.lang.String, boolean, boolean)
		return false;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setDoubleParameterValue(java.lang.String, java.lang.String, java.lang.Double)
	 */
	@Override
	public boolean setDoubleParameterValue(String parameterGroupName, String parameterName, Double parameterValue) {
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");
		Objects.requireNonNull(parameterName, "Parameter name is required!");
		Objects.requireNonNull(parameterValue, "Parameter value is required!");
		
		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);
			
			Template template = firebaseRemoteConfig.getTemplate();
			
			Optional<ParameterGroup> parameterGroupOptional = getParameterGroup(template, parameterGroupName);
			ParameterGroup parameterGroup = new ParameterGroup();
			if (parameterGroupOptional.isPresent()) {
				parameterGroup = parameterGroupOptional.get();
			}
			
			Optional<Parameter> parameterOptional = getParameter(parameterGroup, parameterName);
			Parameter parameter = new Parameter();
			if (parameterOptional.isPresent()) {
				parameter = parameterOptional.get();
			}
			
			parameter.setDefaultValue(ParameterValue.of(parameterValue.toString()));
			
			parameterGroup.getParameters().put(parameterName, parameter);
			
			template.getParameterGroups().put(parameterGroupName, parameterGroup);
			
			Template publishedTemplate = firebaseRemoteConfig.publishTemplate(template);
			if (publishedTemplate != null) {
				return true;
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}
		
		return false;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService#setDateTimeCondition(java.lang.String, java.lang.String)
	 */
	@Override
	public boolean setDateTimeCondition(String conditionName, String dateTime) {
		Objects.requireNonNull(conditionName, "Condition name is required!");
		Objects.requireNonNull(dateTime, "Date time is required!");

		// private static final String FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME = "before_date"; // pass as param

		if (!isDateTimeValid(dateTime)) {
			throw new IllegalArgumentException("Specify date time in 'DateTimeFormatter.ISO_OFFSET_DATE_TIME' format!");
		}

		String dateTimeZoneId = extractZoneID(dateTime);

		if (!isDateTimeZoneIdValid(dateTimeZoneId)) {
			throw new IllegalArgumentException("Specify valid time zone!");
		}

		try {

			FirebaseRemoteConfig firebaseRemoteConfig = FirebaseRemoteConfig.getInstance(firebaseApp);

			Template template = firebaseRemoteConfig.getTemplate();

			List<Condition> conditions = new ArrayList<Condition>();
			if (template.getConditions() != null && !template.getConditions().isEmpty()) {
				conditions = template.getConditions();
			}
			String conditionExpression = buildDateTimeExpression(dateTime, dateTimeZoneId);

			// @formatter:off
			Condition condition = conditions.stream()
					.filter(c -> conditionName.equalsIgnoreCase(c.getName()))
					.findFirst()
					.orElse(null);
			// @formatter:on

			if (condition != null) {
				condition.setExpression(conditionExpression);
			} else {
				condition = new Condition(conditionName, conditionExpression);
			}

			conditions.removeIf(c -> conditionName.equalsIgnoreCase(c.getName()));
			conditions.add(condition);
			template.setConditions(conditions);

			Template publishedTemplate = firebaseRemoteConfig.publishTemplate(template);
			if (publishedTemplate != null) {
				return true;
			}

		} catch (FirebaseRemoteConfigException e) {
			logger.error(e.getMessage());
			e.printStackTrace();
		}

		return false;
	}

	private Optional<ParameterGroup> getParameterGroup(Template template, String parameterGroupName) {
		Objects.requireNonNull(template, "Template is required!");
		Objects.requireNonNull(parameterGroupName, "Parameter group name is required!");

		Map<String, ParameterGroup> parameterGroups = template.getParameterGroups();

		if (parameterGroups.containsKey(parameterGroupName)) {
			return Optional.of(parameterGroups.get(parameterGroupName));
		}

		return Optional.empty();
	}

	private Optional<Parameter> getParameter(ParameterGroup parameterGroup, String parameterName) {
		Objects.requireNonNull(parameterGroup, "Parameter group is required!");
		Objects.requireNonNull(parameterName, "Parameter name is required!");

		Map<String, Parameter> parameterGroupParameters = parameterGroup.getParameters();
		if (!parameterGroupParameters.isEmpty()) {
			if (parameterGroupParameters.containsKey(parameterName)) {
				return Optional.of(parameterGroupParameters.get(parameterName));
			}
		}

		return Optional.empty();
	}

	private boolean isDateTimeValid(String dateTime) {
		Objects.requireNonNull(dateTime, "Date time is required!");

		try {

			ZonedDateTime.parse(dateTime, OFFSET_DATE_TIME_FORMATTER);
			return true;

		} catch (DateTimeParseException dtpe) {
			return false;
		}
	}

	private boolean isDateTimeZoneIdValid(String dateTimeZoneId) {
		Objects.requireNonNull(dateTimeZoneId, "Date time zone ID is required!");

		try {

			ZoneId.of(dateTimeZoneId);
			return true;

		} catch (DateTimeException dte) {
			return false;
		}
	}

	private String extractZoneID(String dateTime) {
		ZonedDateTime parsedDateTime = ZonedDateTime.parse(dateTime, OFFSET_DATE_TIME_FORMATTER);

		if ("Z".equalsIgnoreCase(parsedDateTime.getZone().getId())) {
			return DEFAULT_DATE_TIME_ZONE_ID.getId();
		}

		return parsedDateTime.getZone().getId();
	}

	private String buildDateTimeExpression(String dateTime, String dateTimeZoneId) {
		// e.g.: "dateTime < dateTime('2022-09-05T00:00:00', 'Etc/GMT')"
		StringBuilder expression = new StringBuilder();
		expression.append("dateTime");
		expression.append(" < ");
		expression.append("dateTime(");
		expression.append("'");
		expression.append(formatDateForDateTimeExpression(dateTime));
		expression.append("'");
		expression.append(", ");
		expression.append("'");
		expression.append(dateTimeZoneId);
		expression.append("')");
		return expression.toString();
	}

	private String formatDateForDateTimeExpression(String dateTime) {
		ZonedDateTime parsedOffsetDateTime = ZonedDateTime.parse(dateTime, OFFSET_DATE_TIME_FORMATTER);
		return parsedOffsetDateTime.format(LOCAL_DATE_TIME_FORMATTER);
	}

	private FirebaseApp initializeFirebase() throws IOException {
		Objects.requireNonNull(config.firebaseApplicationCredentialsPathEnv(),
				"Firebase application credentials path is required!");
		Objects.requireNonNull(config.firebaseProjectId(), "Firebase project ID is required!");
		Objects.requireNonNull(config.firebaseApplicationName(), "Firebase application name is required!");

		FirebaseOptions options = FirebaseOptions.builder()
				.setCredentials(GoogleCredentials.fromStream(new FileInputStream(
						getEnvironmentVariableValue(config.firebaseApplicationCredentialsPathEnv()))))
				.setProjectId(config.firebaseProjectId()).build();

		return FirebaseApp.initializeApp(options, config.firebaseApplicationName());
	}

	private String getEnvironmentVariableValue(String envVariableName) {
		Objects.requireNonNull(envVariableName, "Environment variable name is required!");

		String env = System.getenv(envVariableName);
		if (env == null) {
			env = System.getProperty(envVariableName);
		}

		if (env == null) {
			String errorMessage = "No value for environment variable named '" + envVariableName
					+ "' was found in neither 'System.env' nor 'System.properties'";
			logger.error(errorMessage);
			throw new IllegalStateException(errorMessage);
		}

		return env;
	}
}
