/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.config.firebaseremoteconfig.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.Optional;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.google.firebase.remoteconfig.Parameter;
import com.google.firebase.remoteconfig.ParameterGroup;
import com.google.firebase.remoteconfig.ParameterValue;
import com.playertour.backend.config.firebaseremoteconfig.service.api.FirebaseRemoteConfigService;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class FirebaseRemoteConfigIntegrationTest {

	private static final String FIREBASE_REMOTE_CONFIG_GROUP_NAME_PROMOTIONS = "Promotions";
	private static final String FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_PROMOTIONS = "promotions_doublingpoints_enabled";	
	private static final String FIREBASE_REMOTE_CONFIG_CONDITION_NAME_PROMOTIONS = "before_date";
	private static final String FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME_PROMOTIONS = FIREBASE_REMOTE_CONFIG_CONDITION_NAME_PROMOTIONS;
		
	private static final String FIREBASE_REMOTE_CONFIG_GROUP_NAME_CHALLENGES = "Challenges";
	private static final String FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_CHALLENGES = "challenges_commission";

	@InjectBundleContext
	BundleContext bundleContext;

	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		ServiceReference<FirebaseRemoteConfigService> firebaseRemoteConfigServiceReference = firebaseRemoteConfigServiceAware
				.getServiceReference();
		assertThat(firebaseRemoteConfigServiceReference).isNotNull();
	}

	@Test
	public void testGetParameterGroup(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();

		Optional<ParameterGroup> parameterGroupOptional = firebaseRemoteConfigService
				.getParameterGroup(FIREBASE_REMOTE_CONFIG_GROUP_NAME_PROMOTIONS);
		assertThat(parameterGroupOptional.isPresent()).isTrue();

		ParameterGroup parameterGroup = parameterGroupOptional.get();
		assertThat(parameterGroup.getParameters().containsKey(FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_PROMOTIONS)).isTrue();
	}
	
	@Test
	public void testGetParameter(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();

		Optional<Parameter> parameterOptional = firebaseRemoteConfigService
				.getParameter(FIREBASE_REMOTE_CONFIG_GROUP_NAME_PROMOTIONS, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_PROMOTIONS);
		assertThat(parameterOptional.isPresent()).isTrue();

		Parameter parameter = parameterOptional.get();
		assertThat(
				parameter.getConditionalValues().containsKey(FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME_PROMOTIONS))
						.isTrue();

		ParameterValue parameterValue = parameter.getConditionalValues()
				.get(FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME_PROMOTIONS);
		assertThat(parameterValue instanceof ParameterValue.Explicit).isTrue();
	}

	@Test
	public void testGetBooleanParameter(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();

		Optional<Boolean> booleanParameterValue = firebaseRemoteConfigService.getBooleanParameterValue(
				FIREBASE_REMOTE_CONFIG_GROUP_NAME_PROMOTIONS, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_PROMOTIONS,
				FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME_PROMOTIONS);
		assertThat(booleanParameterValue.isPresent()).isTrue();
	}
	
	@Test
	public void testGetDoubleParameter(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();

		Optional<Double> doubleParameterValue = firebaseRemoteConfigService.getDoubleParameterValue(
				FIREBASE_REMOTE_CONFIG_GROUP_NAME_CHALLENGES, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_CHALLENGES);
		assertThat(doubleParameterValue.isPresent()).isTrue();
	}
	
	@Disabled("Firebase Remote Configuration - used in front-end and Firebase services - should not be overwritten from integration tests; if re-enabling to test new features, make sure to restore to previous value(s) or store under a different / unused name")
	@Test
	public void testSetBooleanParameterValue(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();
		
		boolean parameterValue = true;
		boolean parameterDefaultValue = !parameterValue;
		
		boolean isSuccessfullySet = firebaseRemoteConfigService.setBooleanParameterValue(FIREBASE_REMOTE_CONFIG_GROUP_NAME_PROMOTIONS, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_PROMOTIONS, FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME_PROMOTIONS, parameterValue, parameterDefaultValue);
		assertThat(isSuccessfullySet).isTrue();
		
		Optional<Boolean> booleanParameterValue = firebaseRemoteConfigService.getBooleanParameterValue(
				FIREBASE_REMOTE_CONFIG_GROUP_NAME_PROMOTIONS, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_PROMOTIONS,
				FIREBASE_REMOTE_CONFIG_PARAMETER_CONDITIONAL_VALUE_NAME_PROMOTIONS);
		assertThat(booleanParameterValue.isPresent()).isTrue();
		assertThat(booleanParameterValue.get()).isEqualTo(parameterValue);
	}	
	
	@Disabled("Firebase Remote Configuration - used in front-end and Firebase services - should not be overwritten from integration tests; if re-enabling to test new features, make sure to restore to previous value(s) or store under a different / unused name")
	@Test
	public void testSetDateTimeCondition(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();
		
		String conditionDateTime = "2022-09-05T00:00:00+00:00";

		boolean isSuccessfullySet = firebaseRemoteConfigService.setDateTimeCondition(FIREBASE_REMOTE_CONFIG_CONDITION_NAME_PROMOTIONS, conditionDateTime);
		assertThat(isSuccessfullySet).isTrue();
	}
	
	@Disabled("Firebase Remote Configuration - used in front-end and Firebase services - should not be overwritten from integration tests; if re-enabling to test new features, make sure to restore to previous value(s) or store under a different / unused name")
	@Test
	public void testSetDoubleParameterValue(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseRemoteConfigService> firebaseRemoteConfigServiceAware) {

		assertThat(firebaseRemoteConfigServiceAware.getServices()).hasSize(1);
		FirebaseRemoteConfigService firebaseRemoteConfigService = firebaseRemoteConfigServiceAware.getService();
		assertThat(firebaseRemoteConfigService).isNotNull();
		
		Double parameterValue = Double.valueOf(0.10);
		
		boolean isSuccessfullySet = firebaseRemoteConfigService.setDoubleParameterValue(FIREBASE_REMOTE_CONFIG_GROUP_NAME_CHALLENGES, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_CHALLENGES, parameterValue);
		assertThat(isSuccessfullySet).isTrue();
				
		Optional<Double> doubleParameterValue = firebaseRemoteConfigService.getDoubleParameterValue(
				FIREBASE_REMOTE_CONFIG_GROUP_NAME_CHALLENGES, FIREBASE_REMOTE_CONFIG_PARAMETER_NAME_CHALLENGES);
		assertThat(doubleParameterValue.isPresent()).isTrue();
		assertThat(doubleParameterValue.get()).isEqualTo(parameterValue);
	}	
}
