/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.country.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.ArrayList;
import java.util.List;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.apis.country.CountryService;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.country.model.country.Country;
import com.playertour.backend.country.model.country.CountryPackage;
import com.playertour.backend.country.model.country.State;
import com.playertour.backend.igolf.model.igolf.CountryInfo;
import com.playertour.backend.igolf.model.igolf.IGolfFactory;
import com.playertour.backend.igolf.model.igolf.StateInfo;


/**
 * Tests the implementation of CountryService
 * @author ilenia
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class CountryIntegrationTest{
	
	@InjectBundleContext
	BundleContext bundleContext;
	
	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryService> countryAware) {
		
		assertThat(countryAware.getServices()).hasSize(1);	
		ServiceReference<CountryService> countryReference = countryAware.getServiceReference();
		assertThat(countryReference).isNotNull();	
	}
	
	@Test
	public void testSingleSave(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryService> countryAware) 
			throws UnknownTransformationException {
		
		assertThat(countryAware.getServices()).hasSize(1);	
		CountryService countryService = countryAware.getService();
		assertThat(countryService).isNotNull();		
		
		CountryInfo countryInfo = IGolfFactory.eINSTANCE.createCountryInfo();
		countryInfo.setCountryFull("Germany");
		countryInfo.setCountryShort("DE");
		countryInfo.setId_continent("EU");
		countryInfo.setId_country(3);
		
		Country country = countryService.saveCountry(countryInfo);
		
		assertThat(country).isNotNull();
		assertThat(country.getContinentId()).isEqualTo("EU");
		assertThat(country.getCountryId()).isEqualTo("3");
		assertThat(country.getFullName()).isEqualTo("Germany");
		assertThat(country.getShortName()).isEqualTo("DE");
	}
	
	@Test
	public void testMultiSave(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryService> countryAware) 
			throws UnknownTransformationException {
		
		assertThat(countryAware.getServices()).hasSize(1);	
		CountryService countryService = countryAware.getService();
		assertThat(countryService).isNotNull();		
		
		CountryInfo countryInfo1 = IGolfFactory.eINSTANCE.createCountryInfo();
		countryInfo1.setCountryFull("Germany");
		countryInfo1.setCountryShort("DE");
		countryInfo1.setId_continent("EU");
		countryInfo1.setId_country(3);
		
		CountryInfo countryInfo2 = IGolfFactory.eINSTANCE.createCountryInfo();
		countryInfo2.setCountryFull("Italy");
		countryInfo2.setCountryShort("IT");
		countryInfo2.setId_continent("EU");
		countryInfo2.setId_country(2);
		
		List<CountryInfo> countryInfos = new ArrayList<>(2);
		countryInfos.add(countryInfo1);
		countryInfos.add(countryInfo2);
		
		List<Country> countries = countryService.saveCountries(countryInfos);
		assertThat(countries).isNotNull();
		assertThat(countries).hasSize(2);
		
		Country country1 = countries.get(0);		
		assertThat(country1).isNotNull();
		assertThat(country1.getContinentId()).isEqualTo("EU");
		assertThat(country1.getCountryId()).isEqualTo("3");
		assertThat(country1.getFullName()).isEqualTo("Germany");
		assertThat(country1.getShortName()).isEqualTo("DE");
		
		Country country2 = countries.get(1);		
		assertThat(country2).isNotNull();
		assertThat(country2.getContinentId()).isEqualTo("EU");
		assertThat(country2.getCountryId()).isEqualTo("2");
		assertThat(country2.getFullName()).isEqualTo("Italy");
		assertThat(country2.getShortName()).isEqualTo("IT");
	}
	
	@Test
	public void testSaveStates(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryService> countryAware) 
			throws UnknownTransformationException, InterruptedException {
		
		assertThat(countryAware.getServices()).hasSize(1);	
		CountryService countryService = countryAware.getService();
		assertThat(countryService).isNotNull();		
		
		CountryInfo countryInfo = IGolfFactory.eINSTANCE.createCountryInfo();
		countryInfo.setCountryFull("Germany");
		countryInfo.setCountryShort("DE");
		countryInfo.setId_continent("EU");
		countryInfo.setId_country(3);
		
		Country country = countryService.saveCountry(countryInfo);
		
		Thread.sleep(2000);
		assertThat(country).isNotNull();
		
		StateInfo s1 = IGolfFactory.eINSTANCE.createStateInfo();
		s1.setId_country(3);
		s1.setId_state(6);
		s1.setStateFull("Thuringia");
		s1.setStateShort("TH");
		
		StateInfo s2 = IGolfFactory.eINSTANCE.createStateInfo();
		s2.setId_country(3);
		s2.setId_state(7);
		s2.setStateFull("Saxony");
		s2.setStateShort("SA");
		
		List<StateInfo> stateInfos = new ArrayList<>(2);
		stateInfos.add(s1);
		stateInfos.add(s2);
		List<State> states = countryService.saveStates(stateInfos, "3");
		
		Thread.sleep(2000);
		
		assertThat(states).isNotNull();
		assertThat(states).isNotEmpty();
		assertThat(states).hasSize(2);
		
		State state1 = null, state2 = null;
		for(State s : states) {
			if("6".equals(s.getId())) {
				state1 = s;
			}
			else if("7".equals(s.getId())) {
				state2 = s;
			}
		}
		assertThat(state1).isNotNull();
		assertThat(state2).isNotNull();		
	}
	
	
	@AfterEach
	public void doAfterEach(@InjectService(cardinality = 1, timeout = 500, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(CountryPackage.Literals.COUNTRY.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}
}
