/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.country;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.UriMapProvider;
import org.gecko.emf.pushstream.EPushStreamProvider;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.LuceneIndexService;
import org.gecko.util.pushstreams.GeckoPushbackPolicyOption;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.QueuePolicyOption;

import com.playertour.backend.apis.common.CommonIndexService;
import com.playertour.backend.apis.country.CountryIndexService;
import com.playertour.backend.country.helper.CountryIndexHelper;
import com.playertour.backend.country.model.country.Country;
import com.playertour.backend.country.model.country.CountryPackage;

/**
 * 
 * @author ilenia
 * @since Feb 17, 2021
 */
@Component(immediate = true, service = {CountryIndexService.class, CommonIndexService.class}, scope = ServiceScope.SINGLETON, reference = {
		@Reference(name = "mongoCondition", service = UriMapProvider.class, target = "(uri.map.src=mongodb://playertour/)", cardinality = ReferenceCardinality.MANDATORY),
		@Reference(name = "modelCondition", service = ResourceSetFactory.class, target = "(emf.model.name=country)", cardinality = ReferenceCardinality.MANDATORY)
})
public class CountryIndexServiceImpl implements CountryIndexService, CommonIndexService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(target = "(id=country)")
	private LuceneIndexService countryIndex;
	
	@Reference(service=LoggerFactory.class)
	private Logger logger;
	
	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));

	
	@Activate
	public void activate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(100, TimeUnit.MILLISECONDS);
			initializeIndex();
			return true;
		}).onFailure(t -> t.printStackTrace());	
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.country.CountryIndexService#indexCountry(com.playertour.backend.country.model.country.Country, org.gecko.search.api.IndexActionType)
	 */
	@Override
	public void indexCountry(Country country, IndexActionType actionType) {
		DocumentIndexContextObject context = CountryIndexHelper.mapCountry(country, actionType, null);			
		countryIndex.handleContextSync(context);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.common.CommonIndexService#initializeIndex()
	 */
	@Override
	public Promise<Void> initializeIndex() {
		EMFRepository repository = repositoryServiceObjects.getService();
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		IQuery query = queryRepo.createQueryBuilder().allQuery().build();
		logger.info("Starting index Build for country");

		resetIndex();

		EPushStreamProvider psp = queryRepo.getEObjectByQuery(CountryPackage.Literals.COUNTRY, query, CommonIndexService.getLoadOptions());
		if(psp == null) {
			return null;
		}
		
		PushStream<EObject> indexNew = psp.createPushStreamBuilder()
				.withPushbackPolicy(GeckoPushbackPolicyOption.LINEAR_AFTER_THRESHOLD.getPolicy(650))
				.withQueuePolicy(QueuePolicyOption.BLOCK)
				.withExecutor(Executors.newSingleThreadExecutor())
				.withBuffer(new ArrayBlockingQueue<PushEvent<? extends EObject>>(1200))
				.build();

		return indexNew
				.map(eo -> (Country) eo)
				.map(CountryIndexHelper::mapCountryNew)
				.forEach(d -> {
					repository.detach(d.getObject());
					countryIndex.handleContext(d);
				}).onFailure(t -> repositoryServiceObjects.ungetService(repository))
				.thenAccept(result -> logger.info("finished index build for country"))
				.thenAccept(result -> repositoryServiceObjects.ungetService(repository));
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.common.CommonIndexService#resetIndex()
	 */
	@Override
	public void resetIndex() {
		try {
			countryIndex.getIndexWriter().deleteAll();
			countryIndex.commit();
		} catch (IOException e) {
			logger.error("Could not delete country index", e);
		}			
	}

}
