/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.country;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.api.IndexActionType;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.apis.country.CountryIndexService;
import com.playertour.backend.apis.country.CountrySearchService;
import com.playertour.backend.apis.country.CountryService;
import com.playertour.backend.apis.mmt.common.PlayertourModelTransformator;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.country.model.country.Country;
import com.playertour.backend.country.model.country.CountryPackage;
import com.playertour.backend.country.model.country.State;
import com.playertour.backend.igolf.model.igolf.CountryInfo;
import com.playertour.backend.igolf.model.igolf.IGolfPackage;
import com.playertour.backend.igolf.model.igolf.StateInfo;

@Component(scope = ServiceScope.PROTOTYPE)
public class CountryServiceImpl implements CountryService {

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	EMFRepository repository;

	@Reference(target = ("(component.name=BEModelTransformator)"))
	PlayertourModelTransformator transformator;

	@Reference
	CountryIndexService countryIndexService;

	@Reference
	CountrySearchService countrySearchService;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.country.CountryService#saveCountry(com.playertour.backend.igolf.model.igolf.CountryInfo)
	 */
	@Override
	public Country saveCountry(CountryInfo iGolfCountry) throws UnknownTransformationException {

		if(iGolfCountry == null) {
			throw new IllegalArgumentException("Cannot save null CountryInfo object");
		}
		Country country = transformator.transform(iGolfCountry, IGolfPackage.Literals.COUNTRY_INFO, CountryPackage.Literals.COUNTRY);
		boolean isFirstSave = isFirstSave(country);
		if(isFirstSave) {
			repository.attach(country);
		}
		repository.save(country);
		countryIndexService.indexCountry(country, isFirstSave ? IndexActionType.ADD : IndexActionType.MODIFY);
		return country;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.country.CountryService#saveCountries(java.util.List)
	 */
	@Override
	public List<Country> saveCountries(List<CountryInfo> iGolfCountries) throws UnknownTransformationException {

		if(iGolfCountries == null) {
			throw new IllegalArgumentException("Cannot save null CountryInfo object");
		}
		List<Country> countries = new ArrayList<Country>(iGolfCountries.size());
		countries = transformator.transform(iGolfCountries, CountryPackage.Literals.COUNTRY);
		
		Map<Country, Boolean> isFirstSaveMap = new HashMap<>();
		for(Country country : countries) {
			boolean isFirstSave = isFirstSave(country);
			if(isFirstSave) {
				repository.attach(country);
			}
			isFirstSaveMap.put(country, isFirstSave);
		}
		save(countries);
		isFirstSaveMap.entrySet().stream().forEach(e-> {
			countryIndexService.indexCountry(e.getKey(), e.getValue() ? IndexActionType.ADD : IndexActionType.MODIFY);
		});
		return countries;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.country.CountryService#saveStates(java.util.List, java.lang.String)
	 */
	@Override
	public List<State> saveStates(List<StateInfo> iGolfStates, String countryId) throws UnknownTransformationException {
		if(iGolfStates == null) {
			throw new IllegalArgumentException("Cannot save null StateInfo object");
		}
		Country country = countrySearchService.getCountryByCountryId(countryId);
		if(country == null) {
			throw new IllegalArgumentException("Cannot save states for non-existing country");
		}
		List<State> states = new ArrayList<State>(iGolfStates.size());
		states = transformator.transform(iGolfStates, CountryPackage.Literals.STATE);
		country.getStates().clear();
		country.getStates().addAll(states);
		repository.save(country);
		countryIndexService.indexCountry(country, IndexActionType.MODIFY);
		return states;
	}

	@SuppressWarnings("unchecked")
	private void save(Collection<? extends EObject> objects) {
		repository.save((Collection<EObject>)  objects);
	}

	private boolean isFirstSave(Country country) {
		boolean isFirstSave = true;
		Country oldCountry = countrySearchService.getCountryByCountryId(country.getCountryId());
		if(oldCountry != null) {
			isFirstSave = false;
			country.setId(oldCountry.getId());
		}
		return isFirstSave;
	}

	
}
