/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.gamesimulator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.apache.commons.lang3.RandomStringUtils;
import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.api.Stroke;
import com.playertour.backend.api.StrokeType;
import com.playertour.backend.api.TeePreferenceType;
import com.playertour.backend.apis.gamesim.GolfGameSimulationService;
import com.playertour.backend.apis.mmt.common.PlayertourModelTransformator;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.apis.player.exceptions.PlayerUsernameValidationException;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.golfcourse.model.golfcourse.Hole;
import com.playertour.backend.golfcourse.model.golfcourse.Tee;

@Component(name = "GolfGameSimulationService", scope = ServiceScope.PROTOTYPE)
public class GolfGameSimulationServiceImpl implements GolfGameSimulationService {

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerService playerService;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerScorecardService playerScorecardService;

	@Reference(target = ("(component.name=APIModelTransformator)"))
	private PlayertourModelTransformator playertourModelTransformator;

	private List<GolfCourse> golfCourses;

	private List<PlayerProfile> playerProfiles;

	private int numberOfPlayers;

	private boolean skipHoles;

	private int skipHolesMod;

	private int extraStrokes;
	
	private Map<String, String> playerLoginIdByLoginNameMap; // FIXME: again, we go back to issues we've had where not all information is available at player profile level... so this is a (temporary?) workaround...

	private static final ObjectMapper OBJECT_MAPPER;
	static {
		OBJECT_MAPPER = new ObjectMapper();
		OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		OBJECT_MAPPER.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		OBJECT_MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		OBJECT_MAPPER.configure(SerializationFeature.INDENT_OUTPUT, false);
		OBJECT_MAPPER.setSerializationInclusion(Include.NON_NULL);
	}

	@Activate
	public void activate() {
		// FIXME: again, we go back to issues we've had where not all information is available at player profile level... so this is a (temporary?) workaround...
		this.playerLoginIdByLoginNameMap = new HashMap<String, String>();
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.gamesim.GolfGameSimulationService#playGolfGame(java.util.List, boolean, boolean, int, boolean, int, int, java.lang.String[])
	 */
	@Override
	public void playGolfGame(List<String> golfCoursesIds, boolean outputGameInfo, boolean generatePlayers,
			int numberOfPlayersToGenerate, boolean skipHoles, int skipHolesMod, int extraStrokes,
			String... existingPlayersLoginIds) {
		playGolfGame(golfCoursesIds, outputGameInfo, generatePlayers,
				numberOfPlayersToGenerate, skipHoles, skipHolesMod, extraStrokes, false,
				existingPlayersLoginIds);
	}	
	
	// play golf game on one or more golf courses and generate data as a side-effect
	@Override
	public void playGolfGame(List<String> golfCoursesIds, boolean outputGameInfo, boolean generatePlayers,
			int numberOfPlayersToGenerate, boolean skipHoles, int skipHolesMod, int extraStrokes, boolean skipStroke,
			String... existingPlayersLoginIds) {

		if (generatePlayers && (numberOfPlayersToGenerate > 0)) {
			this.playerProfiles = generatePlayers(numberOfPlayersToGenerate);
			this.numberOfPlayers = numberOfPlayersToGenerate;

		} else if (Objects.nonNull(existingPlayersLoginIds) && (existingPlayersLoginIds.length > 0)) {
			this.playerProfiles = fetchExistingPlayers(existingPlayersLoginIds);
			this.numberOfPlayers = existingPlayersLoginIds.length;

		} else {
			throw new IllegalArgumentException(
					"Required input parameters are invalid! To use the 'generate players' mode, set the flag and specify number of players to generate, otherwise to use 'fetch existing' mode, specify login IDs of existing players!");
		}

		this.skipHoles = skipHoles;
		this.skipHolesMod = skipHolesMod;

		this.extraStrokes = extraStrokes;

		// remap list of golf course ids to list of golf courses
		this.golfCourses = getCoursesByIds(golfCoursesIds);

		if (outputGameInfo) {
			outputGameInfo();
		}

		System.out.println("---------------------START--------------------------------------------");
		System.out.println(String.format("Starting game of %d player(s) on %d golf courses", this.numberOfPlayers,
				this.golfCourses.size()));

		System.out.println("----------------------------------------------------------------------");
		System.out.println(String.format("There are %d player(s)", this.playerProfiles.size()));
		for (PlayerProfile playerProfile : this.playerProfiles) {
			System.out.println(convertToJSONString(playerProfile));
		}

		this.playerProfiles.forEach(playerProfile -> {

			String playerProfileId = playerProfile.getId(); // id used internally
			System.out.println(String.format("Player profile ID: %s", playerProfileId));
			
			String playerProfileLoginName = playerProfile.getLoginName();
			System.out.println(String.format("Player profile login name: %s", playerProfileLoginName));
			
			// FIXME: again, we go back to issues we've had where not all information is available at player profile level... so this is a (temporary?) workaround...
			String playerLoginId = playerLoginIdByLoginNameMap.get(playerProfileLoginName);
			System.out.println(String.format("Player profile login ID: %s", playerLoginId));

			double playerProfilePTA = playerProfile.getPlayerTourAbility();
			System.out.println(String.format("Player profile PTA: %.2f", playerProfilePTA));

			this.golfCourses.forEach(gc -> {

				String courseId = gc.getId();
				String courseName = gc.getCourseDetails().getCourseName();

				Objects.requireNonNull(courseId, "Golf course ID is required!");
				Objects.requireNonNull(courseName, "Golf course name is required!");

				Objects.requireNonNull(gc.getCourseGPSVector(), "Golf course GPS Vector is required!");
				Objects.requireNonNull(gc.getCourseGPSVector().getHoles(), "Golf course requires holes!");

				System.out.println("----------------------------------------------------------------------");
				System.out.println(String.format("Player '%s' (ID: %s) starting at golf course '%s'",
						playerProfileLoginName, playerProfileId, courseName));

				System.out.println("----------------------------------------------------------------------");
				
				PlayerScorecard playerScorecard = playerScorecardService.openScorecard(playerLoginId, courseId);
				
				Objects.requireNonNull(playerScorecard.getCourseScorecardInfo(), "Golf course score card is required!");
				
				List<Integer> scorecardPars = playerScorecard.getCourseScorecardInfo().getParHole();
				System.out.println("PARs: " + Arrays.toString(scorecardPars.toArray()));

				System.out.println(String.format("Opened score card (ID: %s) for player '%s' (ID: %s)",
						playerScorecard.getId(), playerProfileLoginName, playerProfileId));

				System.out.println("----------------------------------------------------------------------");
				int holesCount = gc.getCourseGPSVector().getHoles().size();
				int skippedHolesCount = 0;
				for (Hole hole : gc.getCourseGPSVector().getHoles()) {

					int holeNumber = hole.getHoleNumber();

					if (this.skipHoles && this.skipHolesMod > 0 && this.skipHolesMod < holesCount) {
						if (holeNumber % this.skipHolesMod == 0) {
							System.out.println(String.format("Player '%s' (ID: %s) skips hole number %d",
									playerProfileLoginName, playerProfileId, holeNumber));
							skippedHolesCount++;
							continue;
						}
					}

					System.out.println(String.format("Player '%s' (ID: %s) proceeds to hole number %d",
							playerProfileLoginName, playerProfileId, holeNumber));

					Integer holePar = scorecardPars.get((holeNumber - (1 + skippedHolesCount)));
					System.out.println(String.format("Par for hole number %d is %d", holeNumber, holePar));
					
					//int holeStrokes = skipStroke ? (holePar - 2) : holePar;
					int holeStrokes = skipStroke ? (holePar - 1) : holePar;
					
					for (int i = 1; i <= holeStrokes; i++) {
						
						System.out
						.println(String.format("Player '%s' (ID: %s) submits stroke no. %d for hole number %d",
								playerProfileLoginName, playerProfileId, i, holeNumber));

						Stroke stroke = PlayerApiFactory.eINSTANCE.createStroke();
						stroke.setType(StrokeType.DEFAULT);
						stroke.setHoleNumber(holeNumber);

						playerScorecard = playerScorecardService.submitStroke(playerLoginId, playerScorecard.getId(), stroke);
					}					
					
					/*
					for (int i = 1; i <= holePar; i++) {

						System.out
						.println(String.format("Player '%s' (ID: %s) submits stroke no. %d for hole number %d",
								playerProfileLoginName, playerProfileId, i, holeNumber));

						Stroke stroke = PlayerApiFactory.eINSTANCE.createStroke();
						stroke.setType(StrokeType.DEFAULT);
						stroke.setHoleNumber(holeNumber);

						playerScorecard = playerScorecardService.submitStroke(playerLoginId, playerScorecard.getId(), stroke);
					}
					*/

					// add extra strokes
					if (this.extraStrokes > 0 && (holeNumber % 3 == 0)) { // extra strokes for holes: 1, 2

						for (int strokeNumber = 1; strokeNumber <= this.extraStrokes; strokeNumber++) {

							System.out.println(
									String.format("Player '%s' (ID: %s) submits extra stroke no. %d for hole number %d",
											playerProfileLoginName, playerProfileId, strokeNumber, holeNumber));

							Stroke extraStroke = PlayerApiFactory.eINSTANCE.createStroke();
							extraStroke.setType(StrokeType.DEFAULT);
							extraStroke.setHoleNumber(holeNumber);

							playerScorecard = playerScorecardService.submitStroke(playerLoginId, playerScorecard.getId(), extraStroke);
						}
					}

					// close hole
					System.out.println(String.format("Player '%s' (ID: %s) closes hole number %d",
							playerProfileLoginName, playerProfileId, holeNumber));

					System.out.println(String.format("HoleStats (size): %d", playerScorecard.getHoleStats().size()));

					HoleScore holeScore = playerScorecard.getHoleStats().get((holeNumber - (1 + skippedHolesCount)));
					System.out.println("HoleScore: " + convertToJSONString(holeScore));

				//	playerScorecard = playerScorecardService.closeHole(playerLoginId, playerScorecard.getId(), holeScore);
				}

				System.out.println("----------------------------------------------------------------------");
				System.out.println(String.format("Player '%s' (ID: %s) finished at golf course '%s'",
						playerProfileLoginName, playerProfileId, courseName));

				playerScorecard = playerScorecardService.closeScorecard(playerLoginId, playerScorecard.getId());

				System.out.println("----------------------------------------------------------------------");
				System.out.println(String.format("Closed score card (ID: %s) for player '%s' (ID: %s)",
						playerScorecard.getId(), playerProfileLoginName, playerProfileId));

				////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// PAR: the sum of all hole PARs added together as defined by Scorecard
				System.out.println("----------------------------------------------------------------------");
				int gcPAR = playerScorecardService.calculatePAR(scorecardPars);
				System.out.println(String.format("PAR is %d", gcPAR));

				////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// PTS (Player Tour Score)
				System.out.println("----------------------------------------------------------------------");
				for (HoleScore holeScore : playerScorecard.getHoleStats()) {
					System.out.println(String.format("PTS for hole number %d is %d", holeScore.getHoleNumber(),
							holeScore.getPTScore()));
				}

				////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// PTR (Player Tour Result)
				System.out.println("----------------------------------------------------------------------");
				int gcPTR = playerScorecardService.calculatePTR(playerScorecard);
				System.out.println(String.format("PTR is %d", gcPTR));

				////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// PTB (Player Tour Balance)
				int gcPTB = playerScorecardService.calculatePTB(gcPAR, gcPTR);
				System.out.println(String.format("PTB is %d", gcPTB));

				// playerScorecardService.saveScorecard(playerProfileLoginName, playerScorecard);
				playerScorecardService.saveScorecard(playerLoginId, playerScorecard);

				////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// PTA (Player Tour Ability): The Playertour Handicap - the Average PTB
				PlayerProfile playerProfilePTAUpdated = playerScorecardService.updatePTA(playerLoginId);
				double gcPTA = playerProfilePTAUpdated.getPlayerTourAbility();
				System.out.println(String.format("PTA is %.2f", gcPTA));
				
				////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// Leader-board points			
				PlayerProfile playerProfileLeaderBoardPointsUpdated = playerScorecardService.updateLeaderBoardPoints(playerLoginId);
				int leaderBoardPoints = playerProfileLeaderBoardPointsUpdated.getLeaderBoardPoints();
				System.out.println(String.format("Leader-board points: %d", leaderBoardPoints));
			});

			System.out.println(
					String.format("Player '%s' (ID: %s) finished game", playerProfileLoginName, playerProfileId));

		});

		System.out.println(String.format("Finished game of %d player(s) on %d golf courses", this.numberOfPlayers,
				this.golfCourses.size()));
		System.out.println("---------------------END----------------------------------------------");

	}

	// calculate PTAs based on data collected
	@Override
	public void calculatePlayersPTAs(String... existingPlayersLoginIds) {

		this.playerProfiles = fetchExistingPlayers(existingPlayersLoginIds);

		System.out.println("----------------------------------------------------------------------");
		System.out.println(String.format("There are %d player(s)", this.playerProfiles.size()));
		for (PlayerProfile playerProfile : this.playerProfiles) {
			System.out.println(convertToJSONString(playerProfile));
		}

		System.out.println("---------------------START--------------------------------------------");
		System.out.println(String.format("Starting calculation of PTAs for %d player(s)", this.playerProfiles.size()));

		this.playerProfiles.forEach(playerProfile -> {

			String playerProfileLoginName = playerProfile.getLoginName();
			
			// FIXME: again, we go back to issues we've had where not all information is available at player profile level... so this is a (temporary?) workaround...
			String playerLoginId = playerLoginIdByLoginNameMap.get(playerProfileLoginName);

			double playerPTA = playerScorecardService.calculatePTA(playerLoginId);

			System.out.println(String.format("PTA for player %s is %.2f", playerProfileLoginName, playerPTA));

		});

		System.out.println(String.format("Finished calculation of PTAs for %d player(s)", this.playerProfiles.size()));
		System.out.println("---------------------END----------------------------------------------");
	}

	private void outputGameInfo() {
		System.out.println("---------------------START--------------------------------------------");
		System.out.println("Started listing game info");

		System.out.println(String.format("There are %d player(s)", this.playerProfiles.size()));
		for (PlayerProfile playerProfile : this.playerProfiles) {
			System.out.println(convertToJSONString(playerProfile));
		}

		System.out.println(String.format("There are %d golf course(s)", this.golfCourses.size()));

		this.golfCourses.forEach(gc -> {

			String id = gc.getId();
			String courseId = gc.getCourseId();
			String courseName = gc.getCourseDetails().getCourseName();

			System.out.println("----------------------------------------------------------------------");
			System.out.println(String.format("Info for golf course '%s' ..", courseName));
			System.out.println(String.format("ID: '%s'", id));
			System.out.println(String.format("Course ID: '%s'", courseId));

			// score cards
			if (Objects.nonNull(gc.getScorecards())) {

				System.out.println(String.format("Golf course '%s' has score card(s)", courseName));
				System.out.println(
						String.format("There are %d men scorecard(s)", gc.getScorecards().getMenScorecard().size()));
				System.out.println(
						String.format("There are %d women scorecard(s)", gc.getScorecards().getWmnScorecard().size()));

				System.out.println(convertToJSONString(gc.getScorecards()));

			} else {
				System.out.println(String.format("Golf course '%s' has NO score card(s)!", courseName));
			}

			// holes
			if (Objects.nonNull(gc.getCourseGPSVector())) {

				System.out.println(String.format("Hole count: %d", gc.getCourseGPSVector().getHoleCount()));
				System.out.println(String.format("Holes: %d", gc.getCourseGPSVector().getHoles().size()));

				List<Integer> holeNumbers = new ArrayList<Integer>(gc.getCourseGPSVector().getHoles().size());
				for (Hole hole : gc.getCourseGPSVector().getHoles()) {
					holeNumbers.add(hole.getHoleNumber());
				}
				System.out.println(Arrays.toString(holeNumbers.toArray()));

				// enable when needed - outputs lots of data
				/*
				for(Hole hole : gc.getCourseGPSVector().getHoles()) {
					System.out.println(convertToJSONString(hole)); 
				}
				 */

			} else {
				System.out.println(String.format("Golf course '%s' has NO GPS vector!", courseName));
			}

			// tee(s)
			if (Objects.nonNull(gc.getTee())) {
				System.out.println(String.format("Tee(s): %d", gc.getTee().size()));

				for (Tee tee : gc.getTee()) {
					System.out.println(convertToJSONString(tee));
				}

			} else {
				System.out.println(String.format("Golf course '%s' has NO tee(s)!", courseName));
			}

			System.out.println("----------------------------------------------------------------------");

		});

		System.out.println("Finished listing game info");
		System.out.println("---------------------END----------------------------------------------");
	}	

	private List<PlayerProfile> generatePlayers(int numberOfPlayers) {
		List<PlayerProfile> playerProfiles = new ArrayList<PlayerProfile>();

		IntStream.range(0, numberOfPlayers).forEach(playerNumber -> {
			String loginId = RandomStringUtils.randomAlphanumeric(10);
			String loginName = RandomStringUtils.randomAlphanumeric(10);
			PlayerProfile playerProfile = playerService.getPlayerProfile(loginId, loginName);			
			playerProfile.setTeePreference(TeePreferenceType.BLACK);
			try {
				playerProfile = playerService.saveProfile(playerProfile, loginId);
			} catch (PlayerUsernameValidationException e) {
				e.printStackTrace();
			}
			playerProfiles.add(playerProfile);
			
			// FIXME: again, we go back to issues we've had where not all information is available at player profile level... so this is a (temporary?) workaround...
			playerLoginIdByLoginNameMap.put(loginName, loginId);
		});

		return playerProfiles;
	}

	private List<PlayerProfile> fetchExistingPlayers(String... playerLoginIds) {
		return Arrays.stream(playerLoginIds).map(playerLoginId -> fetchExistingPlayer(playerLoginId))
				.collect(Collectors.toList());
	}

	private PlayerProfile fetchExistingPlayer(String playerLoginId) {
		PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(playerLoginId);
		
		// FIXME: again, we go back to issues we've had where not all information is available at player profile level... so this is a (temporary?) workaround...
		playerLoginIdByLoginNameMap.put(playerProfile.getLoginName(), playerLoginId);
		return playerProfile;	
	}

	private List<GolfCourse> getCoursesByIds(List<String> courseIds) {
		return courseIds.stream()
				.map(courseId -> (GolfCourse) repository.getEObject(GolfCoursePackage.Literals.GOLF_COURSE, courseId))
				.collect(Collectors.toList());
	}

	private String convertToJSONString(Object obj) {
		try {
			return OBJECT_MAPPER.writeValueAsString(obj);
		} catch (JsonProcessingException e) {
			System.err.println(e.getMessage());
		}

		return null;
	}
}

