/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.geoapify;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.emfjson.jackson.databind.EMFContext;
import org.gecko.emf.osgi.EMFUriHandlerConstants;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.geoapify.GeoapifyAPIConfigService;
import com.playertour.backend.apis.geoapify.GeoapifyService;
import com.playertour.backend.geoapify.model.geoapify.FeatureCollection;
import com.playertour.backend.geoapify.model.geoapify.GeoapifyPackage;


@Component(name = "GeoapifyService")
public class GeoapifyServiceImpl implements GeoapifyService{
	
	@Reference
	private GeoapifyAPIConfigService apiConfigService;
	
	@Reference
	private ComponentServiceObjects<ResourceSet> rsFactory;
	
	@Reference(service=LoggerFactory.class)
	private Logger logger;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.geoapify.GeoapifyService#getAddressByAutocompletion(java.lang.String)
	 */
	@Override
	public FeatureCollection getAddressByAutocompletion(String query) {
		if(query == null || query.isEmpty()) {
			throw new IllegalArgumentException("Cannot search for address autocompletion with null or empty suggestion!");
		}
		ResourceSet set = null;
		try {
			set = rsFactory.getService();
			String strURI = createStrURI(GeoapifyAPIConstants.GEOAPIFY_AUTOCOMPLETE_PATH, 
					Collections.singletonMap(GeoapifyAPIConstants.GEOAPIFY_AUTOCOMPLETE_TEXT_PARAM, query));
			URI uri = URI.createURI(strURI);
			Resource responseRes = sendRequest(uri, "GET", GeoapifyPackage.Literals.FEATURE_COLLECTION, set);
			if(isValidResponse(responseRes)) {
				FeatureCollection geoJSON = (FeatureCollection) responseRes.getContents().get(0);
				return geoJSON;
			}
			return null;			
			
		} finally {
			rsFactory.ungetService(set);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.geoapify.GeoapifyService#getReverseGeoCoding(double, double)
	 */
	@Override
	public FeatureCollection getReverseGeoCoding(double latitude, double longitude) {
		ResourceSet set = null;
		try {
			set = rsFactory.getService();
			Map<String, String> parameterMap = new HashMap<String, String>();
			parameterMap.put(GeoapifyAPIConstants.GEOAPIFY_REVERSE_GEOCODING_LAT_PARAM, String.valueOf(latitude));
			parameterMap.put(GeoapifyAPIConstants.GEOAPIFY_REVERSE_GEOCODING_LON_PARAM, String.valueOf(longitude));

			String strURI = createStrURI(GeoapifyAPIConstants.GEOAPIFY_REVERSE_GEOCODING_PATH, parameterMap);
			URI uri = URI.createURI(strURI);
			Resource responseRes = sendRequest(uri, "GET", GeoapifyPackage.Literals.FEATURE_COLLECTION, set);
			if(isValidResponse(responseRes)) {
				FeatureCollection geoJSON = (FeatureCollection) responseRes.getContents().get(0);
				return geoJSON;
			}
			return null;				
		} finally {
			rsFactory.ungetService(set);
		}
	}
	
	
	private String createStrURI(String endpoint, Map<String, String> parameterMap) {
		StringBuilder sb = new StringBuilder();
		sb.append(apiConfigService.getAPIBasePath());
		sb.append("/");
		sb.append(endpoint);
		sb.append("?");
		parameterMap.entrySet().forEach(e-> {
			sb.append(e.getKey()+"="+e.getValue());
			sb.append("&");
		});
		sb.append("apiKey="+apiConfigService.getAPIKey());
		return sb.toString();
	}
	
	private Resource sendRequest(URI uri, String method, EClass responseClass, ResourceSet set) {
		
		try {
			Map<Object, Object> options = new HashMap<>();
			Resource requestRes = set.createResource(uri, "application/json");			

			options.put(EMFUriHandlerConstants.OPTION_HTTP_METHOD, method);
			
			Map<String, Object> headers = new HashMap<>();
			headers.put("Accept", "application/json");
			headers.put("Method", "GET");
			options.put(EMFUriHandlerConstants.OPTION_HTTP_HEADERS, headers);
			options.put(EMFContext.Attributes.ROOT_ELEMENT, responseClass);
			
			requestRes.load(options);
			return requestRes;
			
		} catch (IOException e) {
			e.printStackTrace();
		} 		
		return null;
	}

	private boolean isValidResponse(Resource responseRes) {
		if(responseRes == null) {
			logger.warn("Response Resource is null!");
			return false;
		}
		if(responseRes.getContents() == null || responseRes.getContents().isEmpty()) {
			logger.warn("Response Resource does not have any content!");
			return false;
		}
		return true;
	}

}
