/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse.api.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.List;
import java.util.UUID;

import org.gecko.search.api.IndexActionType;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.apis.common.CommonIndexService;
import com.playertour.backend.apis.course.CourseIndexService;
import com.playertour.backend.apis.course.CourseSearchService;
import com.playertour.backend.apis.player.APIGolfCourseService;
import com.playertour.backend.golfcourse.model.golfcourse.CourseDetails;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.Location;

/**
 * Tests the implementation of the API GolfCourseService
 * @author ilenia
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class APIGolfCourseIntegrationTest {

	@InjectBundleContext
	BundleContext bundleContext;
	
	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIGolfCourseService> apiGolfCourseAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseIndexService> indexAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseSearchService> searchAware) {

		assertThat(apiGolfCourseAware.getServices()).hasSize(1);	
		ServiceReference<APIGolfCourseService> apiCourseReference = apiGolfCourseAware.getServiceReference();
		assertThat(apiCourseReference).isNotNull();	
		
		assertThat(indexAware.getServices()).hasSize(1);	
		ServiceReference<CourseIndexService> indexReference = indexAware.getServiceReference();
		assertThat(indexReference).isNotNull();		
		
		assertThat(searchAware.getServices()).hasSize(1);	
		ServiceReference<CourseSearchService> searchReference = searchAware.getServiceReference();
		assertThat(searchReference).isNotNull();
	}
	
	@Test
	public void testCoursesNearMe(@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIGolfCourseService> apiGolfCourseAware,
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseIndexService> indexAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseSearchService> searchAware) throws Exception {

		assertThat(apiGolfCourseAware.getServices()).hasSize(1);	
		APIGolfCourseService apiCourseService = apiGolfCourseAware.getService();
		assertThat(apiCourseService).isNotNull();	
		
		assertThat(indexAware.getServices()).hasSize(1);	
		CourseIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();		
		
		assertThat(searchAware.getServices()).hasSize(1);	
		CourseSearchService searchService = searchAware.getService();
		assertThat(searchService).isNotNull();
		
		GolfCourse golfCourse = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse.setId(UUID.randomUUID().toString());
		CourseDetails details = GolfCourseFactory.eINSTANCE.createCourseDetails();
		details.setCourseName("Jena Golf Club");
		details.setImageUrl("www.golfclubjena.com/image.png");
		Location loc = GolfCourseFactory.eINSTANCE.createLocation();
		loc.setLatitude(50.9234);
		loc.setLongitude(11.5087);
		details.setLocation(loc);
		golfCourse.setCourseDetails(details);
		
		CourseGPSVector gpsVector = GolfCourseFactory.eINSTANCE.createCourseGPSVector();
		gpsVector.setHoleCount(9);
		golfCourse.setCourseGPSVector(gpsVector);
		
		indexService.indexGolfCourse(golfCourse, IndexActionType.ADD);
		
		Thread.sleep(2000);
		
		List<GolfCourseSearchResult> results = apiCourseService.findCoursesNearMe("50.9233999857679", "11.508699981495738", 2);
		assertThat(results).isNotNull();
		assertThat(results).isNotEmpty();
		assertThat(results).hasSize(1);
		assertThat(results.get(0).getCourse().getId()).isEqualTo(golfCourse.getId());
		assertThat(results.get(0).getDistance()).isNotNull();
		
		
	}
	
	@Test
	public void testNoCoursesNearMe(@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIGolfCourseService> apiGolfCourseAware,
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseIndexService> indexAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseSearchService> searchAware) throws Exception {

		assertThat(apiGolfCourseAware.getServices()).hasSize(1);	
		APIGolfCourseService apiCourseService = apiGolfCourseAware.getService();
		assertThat(apiCourseService).isNotNull();	
		
		assertThat(indexAware.getServices()).hasSize(1);	
		CourseIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();		
		
		assertThat(searchAware.getServices()).hasSize(1);	
		CourseSearchService searchService = searchAware.getService();
		assertThat(searchService).isNotNull();
		
		GolfCourse golfCourse = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse.setId(UUID.randomUUID().toString());
		CourseDetails details = GolfCourseFactory.eINSTANCE.createCourseDetails();
		details.setCourseName("Jena Golf Club");
		details.setImageUrl("www.golfclubjena.com/image.png");
		Location loc = GolfCourseFactory.eINSTANCE.createLocation();
		loc.setLatitude(50.9234);
		loc.setLongitude(11.5087);
		details.setLocation(loc);
		golfCourse.setCourseDetails(details);
		
		CourseGPSVector gpsVector = GolfCourseFactory.eINSTANCE.createCourseGPSVector();
		gpsVector.setHoleCount(9);
		golfCourse.setCourseGPSVector(gpsVector);
		
		indexService.indexGolfCourse(golfCourse, IndexActionType.ADD);
		
		Thread.sleep(2000);
		
		List<GolfCourseSearchResult> results = apiCourseService.findCoursesNearMe("51.93", "12.27", 2);
		assertThat(results).isNotNull();
		assertThat(results).isEmpty();	
	}
	
	@Test
	public void testCoursesByName(@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIGolfCourseService> apiGolfCourseAware,
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseIndexService> indexAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseSearchService> searchAware) throws Exception {

		assertThat(apiGolfCourseAware.getServices()).hasSize(1);	
		APIGolfCourseService apiCourseService = apiGolfCourseAware.getService();
		assertThat(apiCourseService).isNotNull();	
		
		assertThat(indexAware.getServices()).hasSize(1);	
		CourseIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();		
		
		assertThat(searchAware.getServices()).hasSize(1);	
		CourseSearchService searchService = searchAware.getService();
		assertThat(searchService).isNotNull();
		
		GolfCourse golfCourse1 = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse1.setId(UUID.randomUUID().toString());
		CourseDetails details = GolfCourseFactory.eINSTANCE.createCourseDetails();
		details.setCourseName("Jena Golf Club");
		details.setImageUrl("www.golfclubjena.com/image.png");
		Location loc = GolfCourseFactory.eINSTANCE.createLocation();
		loc.setLatitude(50.9234);
		loc.setLongitude(11.5087);
		details.setLocation(loc);
		golfCourse1.setCourseDetails(details);
		
		CourseGPSVector gpsVector = GolfCourseFactory.eINSTANCE.createCourseGPSVector();
		gpsVector.setHoleCount(9);
		golfCourse1.setCourseGPSVector(gpsVector);
		
		indexService.indexGolfCourse(golfCourse1, IndexActionType.ADD);
		
		GolfCourse golfCourse2 = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse2.setId(UUID.randomUUID().toString());
		CourseDetails details2 = GolfCourseFactory.eINSTANCE.createCourseDetails();
		details2.setCourseName("Weimar Golf Club");
		details2.setImageUrl("www.golfclubweimar.com/image.png");
		Location loc2 = GolfCourseFactory.eINSTANCE.createLocation();
		loc2.setLatitude(50.97891367631334);
		loc2.setLongitude(11.319484030925306);
		details2.setLocation(loc);
		golfCourse2.setCourseDetails(details2);
		
		CourseGPSVector gpsVector2 = GolfCourseFactory.eINSTANCE.createCourseGPSVector();
		gpsVector2.setHoleCount(9);
		golfCourse2.setCourseGPSVector(gpsVector2);
		
		indexService.indexGolfCourse(golfCourse2, IndexActionType.ADD);
		
		Thread.sleep(2000);
		
		List<GolfCourseSearchResult> results = apiCourseService.findCourses("club", 10);
		assertThat(results).isNotNull();
		assertThat(results).isNotEmpty();
		assertThat(results).hasSize(2);	
		
		results = apiCourseService.findCourses("weim", 10);
		assertThat(results).isNotNull();
		assertThat(results).isNotEmpty();
		assertThat(results).hasSize(1);
		
		results = apiCourseService.findCourses("berlin", 10);
		assertThat(results).isNotNull();
		assertThat(results).isEmpty();		
	}

	
	@AfterEach
	public void doAfterEach(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CommonIndexService> indexAware) {
		
		assertThat(indexAware.getServices()).hasSize(1);	
		CommonIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();		
		
		indexService.resetIndex();
	}
	
}
