/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse.featuresadjacency.service.api;

import static java.util.Comparator.comparing;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

// TODO: decide if this should be extracted to interface and implemented separately
public class GolfCourseFeaturesAdjacencyGraph {
	final Map<GolfCourseFeature, List<GolfCourseFeatureConnection>> _connections;

	public GolfCourseFeaturesAdjacencyGraph() {
		this._connections = new HashMap<GolfCourseFeature, List<GolfCourseFeatureConnection>>();
	}

	public Iterable<GolfCourseFeature> getFeatures() {
		return _connections.keySet();
	}

	public GolfCourseFeature createFeature(GolfCourseFeatureType featureType, GeometryType geometryType,
			org.locationtech.spatial4j.shape.Shape shape) {
		Optional<GolfCourseFeature> match = getFeature(featureType);
		if (match.isPresent()) {
			return match.get();
		}

		GolfCourseFeature feature = new GolfCourseFeature(featureType, geometryType, shape);
		_connections.put(feature, new ArrayList<GolfCourseFeatureConnection>());
		return feature;
	}

	public void addFeature(GolfCourseFeature feature) {
		Optional<GolfCourseFeature> match = getFeature(feature.getFeatureType());
		if (match.isEmpty()) {
			_connections.put(feature, new ArrayList<GolfCourseFeatureConnection>());
		}
	}

	public GolfCourseHoleFeature createHoleFeature(GolfCourseFeatureType featureType, GeometryType geometryType,
			org.locationtech.spatial4j.shape.Shape shape, Integer holeNumber) {
		Optional<GolfCourseHoleFeature> match = getHoleFeature(featureType, holeNumber);
		if (match.isPresent()) {
			return match.get();
		}

		GolfCourseHoleFeature feature = new GolfCourseHoleFeature(featureType, geometryType, shape, holeNumber);
		_connections.put(feature, new ArrayList<GolfCourseFeatureConnection>());
		return feature;
	}

	public void addHoleFeature(GolfCourseHoleFeature feature) {
		Optional<GolfCourseHoleFeature> match = getHoleFeature(feature.getFeatureType(), feature.getHoleNumber());
		if (match.isEmpty()) {
			_connections.put(feature, new ArrayList<GolfCourseFeatureConnection>());
		}
	}

	public Optional<GolfCourseFeature> getFeature(GolfCourseFeatureType type) {
		return _connections.keySet().stream().filter(gcFeature -> gcFeature.getFeatureType() == type).findFirst();
	}

	public Optional<GolfCourseHoleFeature> getHoleFeature(GolfCourseFeatureType type, Integer holeNumber) {
		// @formatter:off
		return _connections.keySet().stream()
			.filter(obj -> obj instanceof GolfCourseHoleFeature)
			.map(obj -> (GolfCourseHoleFeature) obj)
			.filter(gcFeature -> gcFeature.getFeatureType() == type && gcFeature.getHoleNumber() == holeNumber)
			.findFirst();
		// @formatter:on
	}

	public boolean featureExists(GolfCourseFeature feature) {
		return _connections.keySet().stream().filter(gcFeature -> gcFeature == feature).findFirst().isPresent();
	}

	public boolean featureExists(GolfCourseHoleFeature feature) {
		return _connections.keySet().stream().filter(gcFeature -> gcFeature == feature).findFirst().isPresent();
	}

	public boolean featureExists(GolfCourseFeatureType type) {
		return getFeature(type).isPresent();
	}

	public boolean featureExists(GolfCourseFeatureType type, Integer holeNumber) {
		return getHoleFeature(type, holeNumber).isPresent();
	}

	public void addConnection(GolfCourseFeature source, GolfCourseFeature destination, double distance) {
		if (connectionExists(source, destination)) {
			return;
		}

		addConnectionInternal(source, destination, distance);

		if (connectionExists(destination, source)) {
			return;
		}

		addConnectionInternal(destination, source, distance);
	}

	private void addConnectionInternal(GolfCourseFeature from, GolfCourseFeature to, double distance) {
		List<GolfCourseFeatureConnection> connections = new ArrayList<GolfCourseFeatureConnection>();
		if (_connections.containsKey(from)) {
			connections = _connections.get(from);
		}

		connections.add(new GolfCourseFeatureConnection(from, to, distance));
		_connections.put(from, connections);
	}

	public boolean connectionExists(GolfCourseFeature source, GolfCourseFeature destination) {
		// @formatter:off
		Optional<GolfCourseFeatureConnection> match = connections(source)
				.stream()
				.filter(edge -> edge.getDestination() == destination)
				.findAny();
		// @formatter:on

		return (match.isPresent());
	}

	public List<GolfCourseFeatureConnection> connections(GolfCourseFeature source) {
		if (_connections.containsKey(source)) {
			return _connections.get(source);
		} else {
			return Collections.emptyList();
		}
	}

	public List<GolfCourseFeatureConnection> sortedConnections(GolfCourseFeature source) {
		// @formatter:off
		return connections(source)
				.stream()
				.sorted(comparing(GolfCourseFeatureConnection::getDistance))
				.collect(Collectors.toList());
		// @formatter:on
	}

	public Optional<Double> distance(GolfCourseFeature source, GolfCourseFeature destination) {
		// @formatter:off
		Optional<GolfCourseFeatureConnection> match = connections(source)
				.stream()
				.filter(edge -> edge.getDestination() == destination)
				.findAny();
		// @formatter:on
		if (match.isPresent()) {
			return Optional.of(match.get().getDistance());
		} else {
			return Optional.empty();
		}
	}

	@Override
	public String toString() {
		StringBuilder result = new StringBuilder();

		_connections.forEach((feature, featureConnections) -> {

			// @formatter:off
			List<String> destinations = featureConnections.stream()
					.sorted(comparing(GolfCourseFeatureConnection::getDistance))
					.map((featureConnection) -> {
				return String.valueOf(featureConnection.getDestination() + " (" + featureConnection.getDistance() + ")");
			}).collect(Collectors.toList());
			// @formatter:on

			result.append(feature);
			result.append("--> ");
			if (!destinations.isEmpty()) {
				result.append(String.join(", ", destinations));
			}
			result.append("\n");
		});

		return result.toString();
	}
}
