/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse.featuresadjacency.service.api;

import java.io.InputStream;
import java.io.Writer;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.javatuples.Pair;
import org.osgi.annotation.versioning.ProviderType;

@ProviderType
public interface GolfCourseFeaturesAdjacencyService {

	/**
	 * Find closest golf course feature.
	 * 
	 * @param golfCourseId
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestGolfCourseFeature(String golfCourseId, double longitude, double latitude);

	/**
	 * Find closest golf course feature.
	 * 
	 * @param golfCourseId
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestGolfCourseFeature(String golfCourseId,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude);

	/**
	 * Find closest golf course feature.
	 * 
	 * @param golfCourseGPSVector
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestGolfCourseFeature(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, double longitude,
			double latitude);

	/**
	 * Find closest golf course feature.
	 * 
	 * @param golfCourseGPSVector
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestGolfCourseFeature(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude);

	/**
	 * Find closest golf course feature.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestGolfCourseFeature(InputStream golfCourseGPSVectorInputStream, double longitude, double latitude);

	/**
	 * Find closest golf course feature.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestGolfCourseFeature(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude);

	/**
	 * List adjacent golf course features' based on given position.
	 * 
	 * @param golfCourseId
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId, double longitude, double latitude);
	
	/**
	 * List adjacent golf course features' based on given position.
	 * 
	 * @param golfCourseId
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId, 
			Set<GolfCourseFeatureType> golfCourseFeatureTypes,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes, 
			double longitude, 
			double latitude);

	/**
	 * List adjacent golf course features' based on given position.
	 * 
	 * @param golfCourseGPSVector
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, double longitude,
			double latitude);

	/**
	 * List adjacent golf course features' based on given position.
	 * 
	 * @param golfCourseGPSVector
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude);	

	/**
	 * List adjacent golf course features' based on given position.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream, double longitude, double latitude);

	/**
	 * List adjacent golf course features' based on given position.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude);
	
	/**
	 * List adjacent golf course features' based on given position and optional hole number.
	 * 
	 * @param golfCourseId
	 * @param longitude
	 * @param latitude
	 * @param holeNumberOptional
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId, double longitude,
			double latitude, Optional<Integer> holeNumberOptional);

	/**
	 * List adjacent golf course features' based on given position and optional hole number.
	 * 
	 * @param golfCourseId
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @param holeNumberOptional
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude, Optional<Integer> holeNumberOptional);

	/**
	 * List adjacent golf course features' based on given position and optional hole number.
	 * 
	 * @param golfCourseGPSVector
	 * @param longitude
	 * @param latitude
	 * @param holeNumberOptional
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, double longitude,
			double latitude, Optional<Integer> holeNumberOptional);

	/**
	 * List adjacent golf course features' based on given position and optional hole number.
	 * 
	 * @param golfCourseGPSVector
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @param holeNumberOptional
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude, Optional<Integer> holeNumberOptional);

	/**
	 * List adjacent golf course features' based on given position and optional hole number.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param longitude
	 * @param latitude
	 * @param holeNumberOptional
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream,
			double longitude, double latitude, Optional<Integer> holeNumberOptional);

	/**
	 * List adjacent golf course features' based on given position and optional hole number.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param longitude
	 * @param latitude
	 * @param holeNumberOptional
	 * @return
	 */
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude, Optional<Integer> holeNumberOptional);
	
	/**
	 * Check if given position is within golf course perimeter or within pre-defined
	 * distance from golf course perimeter.
	 * 
	 * @param golfCourseId
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	boolean isWithinGolfCoursePerimeter(String golfCourseId, double longitude, double latitude);

	/**
	 * Check if given position is within golf course perimeter or within pre-defined
	 * distance from golf course perimeter.
	 * 
	 * @param golfCourseGPSVector
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	boolean isWithinGolfCoursePerimeter(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, double longitude,
			double latitude);

	/**
	 * Check if given position is within golf course perimeter or within pre-defined
	 * distance from golf course perimeter.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	boolean isWithinGolfCoursePerimeter(InputStream golfCourseGPSVectorInputStream, double longitude, double latitude);

	/**
	 * Find nearest hole.
	 * 
	 * @param golfCourseId
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<Optional<GolfCourseFeature>, Boolean> findNearestHole(String golfCourseId, double longitude, double latitude);

	/**
	 * Find nearest hole.
	 * 
	 * @param golfCourseGPSVector
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<Optional<GolfCourseFeature>, Boolean> findNearestHole(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, double longitude, double latitude);

	/**
	 * Find nearest hole.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Pair<Optional<GolfCourseFeature>, Boolean> findNearestHole(InputStream golfCourseGPSVectorInputStream, double longitude, double latitude);
	
	/**
	 * Check if given position is within perimeter of given golf course hole.
	 * 
	 * @param golfCourseId
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	boolean isWithinHolePerimeter(String golfCourseId, Integer holeNumber, double longitude, double latitude);

	/**
	 * Check if given position is within perimeter of given golf course hole.
	 * 
	 * @param golfCourseGPSVector
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	boolean isWithinHolePerimeter(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, Integer holeNumber, double longitude, double latitude);

	/**
	 * Check if given position is within perimeter of given golf course hole.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	boolean isWithinHolePerimeter(InputStream golfCourseGPSVectorInputStream, Integer holeNumber, double longitude, double latitude);
	
	/**
	 * Find closest hole feature.
	 * 
	 * Be aware that cross-cutting features (spanning entire golf course or several holes' perimeters), such as water hazards (ocean, creek, pond, lake, etc.), sand, rock, lava - are excluded when searching only on hole-level; use 'findClosestGolfCourseFeature' instead.
	 * 
	 * @param golfCourseId
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestHoleFeature(String golfCourseId, Integer holeNumber, double longitude, double latitude);

	/**
	 * Find closest hole feature.
	 * 
	 * Be aware that cross-cutting features (spanning entire golf course or several holes' perimeters), such as water hazards (ocean, creek, pond, lake, etc.), sand, rock, lava - are excluded when searching only on hole-level; use 'findClosestGolfCourseFeature' instead.
	 * 
	 * @param golfCourseGPSVector
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestHoleFeature(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, Integer holeNumber, double longitude, double latitude);

	/**
	 * Find closest hole feature.
	 * 
	 * Be aware that cross-cutting features (spanning entire golf course or several holes' perimeters), such as water hazards (ocean, creek, pond, lake, etc.), sand, rock, lava - are excluded when searching only on hole-level; use 'findClosestGolfCourseFeature' instead.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	Optional<GolfCourseFeature> findClosestHoleFeature(InputStream golfCourseGPSVectorInputStream, Integer holeNumber, double longitude, double latitude);

	/**
	 * List adjacent hole features' for given hole number based on given position.
	 * 
	 * Be aware that cross-cutting features (spanning entire golf course or several holes' perimeters), such as water hazards (ocean, creek, pond, lake, etc.), sand, rock, lava - are excluded when searching only on hole-level; use 'findClosestGolfCourseFeature' instead.
	 * 
	 * @param golfCourseId
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	@Deprecated(forRemoval = true)
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentHoleFeatures(String golfCourseId, Integer holeNumber, double longitude, double latitude);

	/**
	 * List adjacent hole features' for given hole number based on given position.
	 * 
	 * Be aware that cross-cutting features (spanning entire golf course or several holes' perimeters), such as water hazards (ocean, creek, pond, lake, etc.), sand, rock, lava - are excluded when searching only on hole-level; use 'findClosestGolfCourseFeature' instead.
	 * 
	 * @param golfCourseGPSVector
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	@Deprecated(forRemoval = true)
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentHoleFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, Integer holeNumber, double longitude, double latitude);

	/**
	 * List adjacent hole features' for given hole number based on given position.
	 * 
	 * Be aware that cross-cutting features (spanning entire golf course or several holes' perimeters), such as water hazards (ocean, creek, pond, lake, etc.), sand, rock, lava - are excluded when searching only on hole-level; use 'findClosestGolfCourseFeature' instead.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param holeNumber
	 * @param longitude
	 * @param latitude
	 * @return
	 */
	@Deprecated(forRemoval = true)
	Pair<List<GolfCourseFeature>, Boolean> listAdjacentHoleFeatures(InputStream golfCourseGPSVectorInputStream, Integer holeNumber, double longitude, double latitude);
	
	/**
	 * Visualize adjacency graph as GeoJson.
	 * 
	 * @param golfCourseId
	 * @param writer
	 */
	void visualizeAdjacencyViaGeoJson(String golfCourseId, Writer writer);

	/**
	 * Visualize adjacency graph as GeoJson.
	 * 
	 * @param golfCourseId
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param writer
	 */
	void visualizeAdjacencyViaGeoJson(String golfCourseId, 
			Set<GolfCourseFeatureType> golfCourseFeatureTypes,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes, 
			Writer writer);

	/**
	 * Visualize adjacency graph as GeoJson.
	 * 
	 * @param golfCourseGPSVector
	 * @param writer	 
	 */
	void visualizeAdjacencyViaGeoJson(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector, 
			Writer writer);

	/**
	 * Visualize adjacency graph as GeoJson.
	 * 
	 * @param golfCourseGPSVector
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param writer
	 */
	void visualizeAdjacencyViaGeoJson(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes, 
			Writer writer);

	/**
	 * Visualize adjacency graph as GeoJson.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param writer
	 */
	void visualizeAdjacencyViaGeoJson(InputStream golfCourseGPSVectorInputStream, Writer writer);

	/**
	 * Visualize adjacency graph as GeoJson.
	 * 
	 * @param golfCourseGPSVectorInputStream
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param writer
	 */
	void visualizeAdjacencyViaGeoJson(InputStream golfCourseGPSVectorInputStream, 
			Set<GolfCourseFeatureType> golfCourseFeatureTypes,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes, 
			Writer writer);
	
	/**
	 * Repackage to EMF.
	 * 
	 * @param isOnGolfCourse
	 * @return
	 */
	com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.IsOnGolfCourseResult createIsOnGolfCourseResult(
			boolean isOnGolfCourse);
	
	/**
	 * Repackage to EMF.
	 * 
	 * @param adjacentFeaturesResult
	 * @return
	 */
	com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.AdjacentGolfCourseFeaturesResult createAdjacentGolfCourseFeaturesResult(
			Pair<List<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature>, Boolean> adjacentFeaturesResult);
	
	/**
	 * Repackage to EMF.
	 * 
	 * @param nearestHoleResult
	 * @return
	 */
	com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.NearestHoleResult createNearestHoleResult(
			Pair<Optional<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature>, Boolean> nearestHoleResult);

	/**
	 * Repackage to EMF.
	 * 
	 * @param isWithinHolePerimeter
	 * @param holeNumber
	 * @return
	 */
	com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.IsWithinHolePerimeterResult createIsWithinHolePerimeterResult(
			boolean isWithinHolePerimeter, Integer holeNumber);
	
}
