/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse.featuresadjacency.service.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.geojson.Feature;
import org.geojson.FeatureCollection;
import org.javatuples.Pair;
import org.locationtech.jts.operation.distance.DistanceOp;
import org.locationtech.spatial4j.context.SpatialContext;
import org.locationtech.spatial4j.context.jts.JtsSpatialContext;
import org.locationtech.spatial4j.context.jts.JtsSpatialContextFactory;
import org.locationtech.spatial4j.distance.DistanceUtils;
import org.locationtech.spatial4j.exception.InvalidShapeException;
import org.locationtech.spatial4j.io.GeoJSONReader;
import org.locationtech.spatial4j.shape.jts.JtsShapeFactory;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.playertour.backend.apis.course.CourseGPSService;
import com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.AdjacentGolfCourseFeaturesResult;
import com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.GCFeaturesAdjacencyFactory;
import com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.IsOnGolfCourseResult;
import com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.IsWithinHolePerimeterResult;
import com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.NearestHoleResult;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GeometryType;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeatureConnection;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeatureType;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyGraph;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseHoleFeature;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;

@Component(name = "GolfCourseFeaturesAdjacencyService", scope = ServiceScope.PROTOTYPE, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "GolfCourseFeaturesAdjacencyService")
@Designate(ocd = GolfCourseFeaturesAdjacencyServiceImpl.Config.class)
public class GolfCourseFeaturesAdjacencyServiceImpl implements GolfCourseFeaturesAdjacencyService {

	/**************************/
	/** Services' references **/
	@Reference(service=LoggerFactory.class)
	private Logger logger;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private CourseGPSService courseGPSService;
	
	/********************************/
	/** Externalized configuration **/	
	private Config config;
	
	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {

		// Minimum distance between golf course features to be considered connected in meters
		int featureConnectionMinDistance();
		
		// Minimum distance to golf course feature to be considered close enough in meters
		int featureClosestMinDistance();
		
		// Maximum distance to golf course feature above which it is considered too far in meters
		int featureClosestMaxDistance();
		
		// Maximum acceptable distance from golf course in meters
		int withinGolfcourseDistance();
		
		// Maximum acceptable distance from hole perimeter in meters
		int withinHolePerimeterDistance();
		
		// Whether check of maximum acceptable distance from golf course should be done internally
		boolean withinGolfcourseCheck();
		
	}
	
	/*****************************/
	/** JTS / Spatial4J related **/	
	private static final JtsSpatialContext JTS_SPATIAL_CONTEXT = JtsSpatialContext.GEO;
	private static final JtsShapeFactory JTS_SHAPE_FACTORY = JTS_SPATIAL_CONTEXT.getShapeFactory();
	// TODO: perhaps there's a better way to correlate with JTS_SPATIAL_CONTEXT ? 
	private static final JtsSpatialContextFactory JTS_SPATIAL_CONTEXT_FACTORY = new JtsSpatialContextFactory();
	private static final GeoJSONReader GEOJSON_READER = new GeoJSONReader(JTS_SPATIAL_CONTEXT, JTS_SPATIAL_CONTEXT_FACTORY); /** GeoJSON reader */
	
	/*********************/
	/** JACKSON related **/
	private static final ObjectMapper OBJECT_MAPPER;
	static {
		OBJECT_MAPPER = new ObjectMapper();
		OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		OBJECT_MAPPER.configure(SerializationFeature.WRAP_ROOT_VALUE, false);
		OBJECT_MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		OBJECT_MAPPER.configure(SerializationFeature.INDENT_OUTPUT, false);
		OBJECT_MAPPER.setSerializationInclusion(Include.NON_NULL);
	}
	
	/*******************************/
	/** Golf Course Feature types **/
	
	/** Golf Course level feature types */
	private static final Set<GolfCourseFeatureType> GOLFCOURSE_FEATURE_TYPES = EnumSet.noneOf(GolfCourseFeatureType.class);
	static {
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.background);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.ocean);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.bridge);
	//	GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.vegetation);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.creek);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.pond);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.sand);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.rock);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.lake);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.water);
	//	GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.tree);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.clubHouse);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.lava);
		GOLFCOURSE_FEATURE_TYPES.add(GolfCourseFeatureType.path);
	}
	
	/** Golf Course hole level feature types */
	private static final Set<GolfCourseFeatureType> GOLFCOURSE_HOLE_FEATURE_TYPES = EnumSet.noneOf(GolfCourseFeatureType.class);
	static {
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.green);
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.greenCenter);
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.bunker);
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.centralPath);
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.fairway);
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.perimeter);
		GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.teeBox);
	//	GOLFCOURSE_HOLE_FEATURE_TYPES.add(GolfCourseFeatureType.teeBoxCenter);
	}
		
	@Activate
	public void activate(Config config) {
		this.config = config;
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestGolfCourseFeature(java.lang.String, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestGolfCourseFeature(String golfCourseId, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		
		return findClosestGolfCourseFeature(golfCourseId, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestGolfCourseFeature(java.lang.String, java.util.Set, java.util.Set, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestGolfCourseFeature(String golfCourseId,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, 
			double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(golfCourseFeatureTypes, golfCourseHoleFeatureTypes);
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));
		
		return findClosestGolfCourseFeature(golfCourseGPSVector, 
				golfCourseFeatureTypes, 
				golfCourseHoleFeatureTypes,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestGolfCourseFeature(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestGolfCourseFeature(CourseGPSVector golfCourseGPSVector, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		
		return findClosestGolfCourseFeature(golfCourseGPSVector, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestGolfCourseFeature(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.util.Set, java.util.Set, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestGolfCourseFeature(CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, 
			double latitude) {
		
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		
		List<GolfCourseFeature> golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVector,
				golfCourseFeatureTypes, 
				golfCourseHoleFeatureTypes,
				EnumSet.of(GeometryType.Polygon, GeometryType.Point));
		      //EnumSet.of(GeometryType.Polygon));
		
		Pair<Optional<GolfCourseFeature>, Boolean> closestFeatureResult = findClosestFeature(golfCourseFeatures, longitude, latitude, false);
		return closestFeatureResult.getValue0();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestGolfCourseFeature(java.io.InputStream, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestGolfCourseFeature(InputStream golfCourseGPSVectorInputStream, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		
		return findClosestGolfCourseFeature(golfCourseGPSVectorInputStream, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestGolfCourseFeature(java.io.InputStream, java.util.Set, java.util.Set, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestGolfCourseFeature(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, 
			double latitude) {
		
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		
		List<GolfCourseFeature> golfCourseFeatures = Collections.emptyList();
		
		try {
			golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVectorInputStream,
					golfCourseFeatureTypes, 
					golfCourseHoleFeatureTypes,
					EnumSet.of(GeometryType.Polygon, GeometryType.Point));
			      //EnumSet.of(GeometryType.Polygon));
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		Pair<Optional<GolfCourseFeature>, Boolean> closestFeatureResult = findClosestFeature(golfCourseFeatures, longitude, latitude, false);
		return closestFeatureResult.getValue0();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.lang.String, double, double)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		
		return listAdjacentGolfCourseFeatures(golfCourseId,
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.lang.String, java.util.Set, java.util.Set, double, double)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, 
			double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");		

		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(golfCourseFeatureTypes, golfCourseHoleFeatureTypes);
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));
		
		return listAdjacentGolfCourseFeatures(golfCourseGPSVector,
				golfCourseFeatureTypes, 
				golfCourseHoleFeatureTypes,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, double, double)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(CourseGPSVector golfCourseGPSVector, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		
		return listAdjacentGolfCourseFeatures(golfCourseGPSVector,
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.util.Set, java.util.Set, double, double)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, 
			double latitude) {
		
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		
		List<GolfCourseFeature> golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVector,
				golfCourseFeatureTypes, 
				golfCourseHoleFeatureTypes,
				EnumSet.of(GeometryType.Polygon, GeometryType.Point));
		      //EnumSet.of(GeometryType.Polygon));
		
		return listAdjacentFeatures(golfCourseFeatures, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.io.InputStream, double, double)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		
		return listAdjacentGolfCourseFeatures(golfCourseGPSVectorInputStream, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude);	
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.io.InputStream, java.util.Set, java.util.Set, double, double)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude) {
		
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		
		List<GolfCourseFeature> golfCourseFeatures = Collections.emptyList();
		
		try {
			golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVectorInputStream,
					golfCourseFeatureTypes, 
					golfCourseHoleFeatureTypes,
					EnumSet.of(GeometryType.Polygon, GeometryType.Point));
			      //EnumSet.of(GeometryType.Polygon));
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		return listAdjacentFeatures(golfCourseFeatures, longitude, latitude);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.lang.String, double, double, java.util.Optional)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId, double longitude,
			double latitude, Optional<Integer> holeNumberOptional) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		
		return listAdjacentGolfCourseFeatures(golfCourseId,
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude, 
				holeNumberOptional);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.lang.String, java.util.Set, java.util.Set, double, double, java.util.Optional)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(String golfCourseId,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude, Optional<Integer> holeNumberOptional) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");

		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(golfCourseFeatureTypes,
				golfCourseHoleFeatureTypes);

		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService
				.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));

		return listAdjacentGolfCourseFeatures(golfCourseGPSVector, golfCourseFeatureTypes, golfCourseHoleFeatureTypes,
				longitude, latitude, holeNumberOptional);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, double, double, java.util.Optional)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(CourseGPSVector golfCourseGPSVector,
			double longitude, 
			double latitude, 
			Optional<Integer> holeNumberOptional) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		
		return listAdjacentGolfCourseFeatures(golfCourseGPSVector,
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude, 
				holeNumberOptional);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.util.Set, java.util.Set, double, double, java.util.Optional)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			double longitude, double latitude, Optional<Integer> holeNumberOptional) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");

		List<GolfCourseFeature> golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVector,
				golfCourseFeatureTypes, golfCourseHoleFeatureTypes,
				EnumSet.of(GeometryType.Polygon, GeometryType.Point));
		// EnumSet.of(GeometryType.Polygon));

		return listAdjacentFeatures(golfCourseFeatures, longitude, latitude, holeNumberOptional);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.io.InputStream, double, double, java.util.Optional)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(
			InputStream golfCourseGPSVectorInputStream, double longitude, double latitude,
			Optional<Integer> holeNumberOptional) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		
		return listAdjacentGolfCourseFeatures(golfCourseGPSVectorInputStream, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				longitude, 
				latitude, 
				holeNumberOptional);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentGolfCourseFeatures(java.io.InputStream, java.util.Set, java.util.Set, double, double, java.util.Optional)
	 */
	@Override
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentGolfCourseFeatures(
			InputStream golfCourseGPSVectorInputStream, Set<GolfCourseFeatureType> golfCourseFeatureTypes,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes, double longitude, double latitude,
			Optional<Integer> holeNumberOptional) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		
		List<GolfCourseFeature> golfCourseFeatures = Collections.emptyList();
		
		try {
			golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVectorInputStream,
					golfCourseFeatureTypes, 
					golfCourseHoleFeatureTypes,
					EnumSet.of(GeometryType.Polygon, GeometryType.Point));
			      //EnumSet.of(GeometryType.Polygon));
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		return listAdjacentFeatures(golfCourseFeatures, longitude, latitude, holeNumberOptional);
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#isWithinGolfCoursePerimeter(java.lang.String, double, double)
	 */
	@Override
	public boolean isWithinGolfCoursePerimeter(String golfCourseId, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(Set.of(GolfCourseFeatureType.background));
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));
		
		return isWithinGolfCoursePerimeter(golfCourseGPSVector, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#isWithinGolfCoursePerimeter(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, double, double)
	 */
	@Override
	public boolean isWithinGolfCoursePerimeter(CourseGPSVector golfCourseGPSVector, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		
		List<GolfCourseFeature> golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVector,
				EnumSet.of(GolfCourseFeatureType.background), 
				EnumSet.noneOf(GolfCourseFeatureType.class),
				EnumSet.of(GeometryType.Polygon));
		
		return isWithinGolfCoursePerimeter(golfCourseFeatures, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#isWithinGolfCoursePerimeter(java.io.InputStream, double, double)
	 */
	@Override
	public boolean isWithinGolfCoursePerimeter(InputStream golfCourseGPSVectorInputStream, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		
		List<GolfCourseFeature> golfCourseFeatures = Collections.emptyList();
		
		try {
			golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVectorInputStream,
					EnumSet.of(GolfCourseFeatureType.background), 
					EnumSet.noneOf(GolfCourseFeatureType.class),
					EnumSet.of(GeometryType.Polygon));
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		return isWithinGolfCoursePerimeter(golfCourseFeatures, longitude, latitude);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findNearestHole(java.lang.String, double, double)
	 */
	@Override
	public Pair<Optional<GolfCourseFeature>, Boolean> findNearestHole(String golfCourseId, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(Set.of(GolfCourseFeatureType.background, GolfCourseFeatureType.perimeter));
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));		
		
		return findNearestHole(golfCourseGPSVector, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findNearestHole(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, double, double)
	 */
	@Override
	public Pair<Optional<GolfCourseFeature>, Boolean> findNearestHole(CourseGPSVector golfCourseGPSVector, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		
		List<GolfCourseFeature> golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVector,
				EnumSet.of(GolfCourseFeatureType.background),
				EnumSet.of(GolfCourseFeatureType.perimeter),
				EnumSet.of(GeometryType.Polygon));
		
		return findClosestFeature(golfCourseFeatures, longitude, latitude, true);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findNearestHole(java.io.InputStream, double, double)
	 */
	@Override
	public Pair<Optional<GolfCourseFeature>, Boolean> findNearestHole(InputStream golfCourseGPSVectorInputStream,
			double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		
		List<GolfCourseFeature> golfCourseFeatures = Collections.emptyList();
		
		try {
			golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVectorInputStream,
					EnumSet.of(GolfCourseFeatureType.background),
					EnumSet.of(GolfCourseFeatureType.perimeter),
					EnumSet.of(GeometryType.Polygon));
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		return findClosestFeature(golfCourseFeatures, longitude, latitude, true);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#isWithinHolePerimeter(java.lang.String, java.lang.Integer, double, double)
	 */
	@Override
	public boolean isWithinHolePerimeter(String golfCourseId, Integer holeNumber, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(Set.of(GolfCourseFeatureType.background, GolfCourseFeatureType.perimeter));
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));		
		
		return isWithinHolePerimeter(golfCourseGPSVector, holeNumber, longitude, latitude);		
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#isWithinHolePerimeter(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.lang.Integer, double, double)
	 */
	@Override
	public boolean isWithinHolePerimeter(CourseGPSVector golfCourseGPSVector, Integer holeNumber, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<GolfCourseFeature> golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVector,
				EnumSet.of(GolfCourseFeatureType.background),
				EnumSet.of(GolfCourseFeatureType.perimeter),
				EnumSet.of(GeometryType.Polygon));
		
		return isWithinHolePerimeter(golfCourseFeatures, holeNumber, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#isWithinHolePerimeter(java.io.InputStream, java.lang.Integer, double, double)
	 */
	@Override
	public boolean isWithinHolePerimeter(InputStream golfCourseGPSVectorInputStream, Integer holeNumber,
			double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<GolfCourseFeature> golfCourseFeatures = Collections.emptyList();
		
		try {
			golfCourseFeatures = extractGolfCourseFeatures(golfCourseGPSVectorInputStream,
					EnumSet.of(GolfCourseFeatureType.background),
					EnumSet.of(GolfCourseFeatureType.perimeter),
					EnumSet.of(GeometryType.Polygon));
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		return isWithinHolePerimeter(golfCourseFeatures, holeNumber, longitude, latitude);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestHoleFeature(java.lang.String, java.lang.Integer, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestHoleFeature(String golfCourseId, Integer holeNumber, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(
				Set.of(GolfCourseFeatureType.background), 
				GOLFCOURSE_HOLE_FEATURE_TYPES);
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));		
		
		return findClosestHoleFeature(golfCourseGPSVector, holeNumber, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestHoleFeature(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.lang.Integer, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestHoleFeature(CourseGPSVector golfCourseGPSVector, Integer holeNumber,
			double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<GolfCourseFeature> holeFeatures = extractHoleFeatures(golfCourseGPSVector,
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				EnumSet.of(GeometryType.Polygon), 
				holeNumber);
			
		Pair<Optional<GolfCourseFeature>, Boolean> closestFeatureResult = findClosestFeature(holeFeatures, longitude, latitude, false);
		return closestFeatureResult.getValue0();
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#findClosestHoleFeature(java.io.InputStream, java.lang.Integer, double, double)
	 */
	@Override
	public Optional<GolfCourseFeature> findClosestHoleFeature(InputStream golfCourseGPSVectorInputStream,
			Integer holeNumber, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<GolfCourseFeature> holeFeatures = Collections.emptyList();
		
		try {
			
			holeFeatures = extractHoleFeatures(golfCourseGPSVectorInputStream,
					GOLFCOURSE_HOLE_FEATURE_TYPES,
					EnumSet.of(GeometryType.Polygon), 
					holeNumber);
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}		
		
		Pair<Optional<GolfCourseFeature>, Boolean> closestFeatureResult = findClosestFeature(holeFeatures, longitude, latitude, false);
		return closestFeatureResult.getValue0();		
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentHoleFeatures(java.lang.String, java.lang.Integer, double, double)
	 */
	@Override
	@Deprecated(forRemoval = true)
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentHoleFeatures(String golfCourseId, Integer holeNumber,
			double longitude, double latitude) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(
				Set.of(GolfCourseFeatureType.background), 
				GOLFCOURSE_HOLE_FEATURE_TYPES);
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));		
		
		return listAdjacentHoleFeatures(golfCourseGPSVector, holeNumber, longitude, latitude);		
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentHoleFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.lang.Integer, double, double)
	 */
	@Override
	@Deprecated(forRemoval = true)
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentHoleFeatures(CourseGPSVector golfCourseGPSVector,
			Integer holeNumber, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");
		
		List<GolfCourseFeature> holeFeatures = extractHoleFeatures(golfCourseGPSVector,
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				EnumSet.of(GeometryType.Polygon), 
				holeNumber);
		
		return listAdjacentFeatures(holeFeatures, longitude, latitude);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#listAdjacentHoleFeatures(java.io.InputStream, java.lang.Integer, double, double)
	 */
	@Override
	@Deprecated(forRemoval = true)
	public Pair<List<GolfCourseFeature>, Boolean> listAdjacentHoleFeatures(InputStream golfCourseGPSVectorInputStream,
			Integer holeNumber, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		Objects.requireNonNull(holeNumber, "Hole number is required!");		
		
		List<GolfCourseFeature> holeFeatures = Collections.emptyList();
		
		try {
			holeFeatures = extractHoleFeatures(golfCourseGPSVectorInputStream,
					GOLFCOURSE_HOLE_FEATURE_TYPES,
					EnumSet.of(GeometryType.Polygon), 
					holeNumber);
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}
		
		return listAdjacentFeatures(holeFeatures, longitude, latitude);		
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#visualizeAdjacencyViaGeoJson(java.lang.String, java.io.Writer)
	 */
	@Override
	public void visualizeAdjacencyViaGeoJson(String golfCourseId, Writer writer) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(writer, "Writer is required!");
		
		visualizeAdjacencyViaGeoJson(golfCourseId, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				writer);		
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#visualizeAdjacencyViaGeoJson(java.lang.String, java.util.Set, java.util.Set, java.io.Writer)
	 */
	@Override
	public void visualizeAdjacencyViaGeoJson(String golfCourseId, 
			Set<GolfCourseFeatureType> golfCourseFeatureTypes,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes, 
			Writer writer) {
		Objects.requireNonNull(golfCourseId, "Golf course ID is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		Objects.requireNonNull(writer, "Writer is required!");
		
		List<String> projections = mapGolfCourseFeatureTypeToCourseGPSServiceProjection(golfCourseFeatureTypes, golfCourseHoleFeatureTypes);
		
		com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector golfCourseGPSVector = courseGPSService.getCourseGPSVector(golfCourseId, projections.toArray(new String[projections.size()]));
		
		visualizeAdjacencyViaGeoJson(golfCourseGPSVector,
				golfCourseFeatureTypes, 
				golfCourseHoleFeatureTypes,
				writer);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#visualizeAdjacencyViaGeoJson(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.io.Writer)
	 */
	@Override
	public void visualizeAdjacencyViaGeoJson(CourseGPSVector golfCourseGPSVector, Writer writer) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(writer, "Writer is required!");
		
		visualizeAdjacencyViaGeoJson(golfCourseGPSVector,
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				writer);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#visualizeAdjacencyViaGeoJson(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector, java.util.Set, java.util.Set, java.io.Writer)
	 */
	@Override
	public void visualizeAdjacencyViaGeoJson(CourseGPSVector golfCourseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Writer writer) {
		Objects.requireNonNull(golfCourseGPSVector, "Golf course GPS vector is required!");
		Objects.requireNonNull(golfCourseFeatureTypes, "Golf Course level feature types are required!");
		Objects.requireNonNull(golfCourseHoleFeatureTypes, "Golf Course hole level feature types are required!");
		Objects.requireNonNull(writer, "Writer is required!");
		
		Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple = mapAsGeoJSONFeatures(golfCourseGPSVector,
				golfCourseFeatureTypes, 
				golfCourseHoleFeatureTypes,
				EnumSet.of(GeometryType.Polygon));
		
		visualizeAdjacencyViaGeoJson(mappedFeaturesTuple, writer);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#visualizeAdjacencyViaGeoJson(java.io.InputStream, java.io.Writer)
	 */
	@Override
	public void visualizeAdjacencyViaGeoJson(InputStream golfCourseGPSVectorInputStream, Writer writer) {
		Objects.requireNonNull(golfCourseGPSVectorInputStream, "Golf course GPS vector input stream is required!");
		
		visualizeAdjacencyViaGeoJson(golfCourseGPSVectorInputStream, 
				GOLFCOURSE_FEATURE_TYPES, 
				GOLFCOURSE_HOLE_FEATURE_TYPES,
				writer);
	}
		
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#visualizeAdjacencyViaGeoJson(java.io.InputStream, java.util.Set, java.util.Set, java.io.Writer)
	 */
	@Override
	public void visualizeAdjacencyViaGeoJson(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Writer writer) {
		
		try {
			
			Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple = mapAsGeoJSONFeatures(golfCourseGPSVectorInputStream,
					golfCourseFeatureTypes, 
					golfCourseHoleFeatureTypes,
					EnumSet.of(GeometryType.Polygon));
			
			visualizeAdjacencyViaGeoJson(mappedFeaturesTuple, writer);
			
		} catch (InvalidShapeException | IOException | ParseException e) {
			logger.error(e.getMessage());
		}		
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#createIsOnGolfCourseResult(boolean)
	 */
	@Override
	public IsOnGolfCourseResult createIsOnGolfCourseResult(boolean isOnGolfCourse) {
		com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.IsOnGolfCourseResult result = GCFeaturesAdjacencyFactory.eINSTANCE.createIsOnGolfCourseResult();
		result.setIsOnGolfCourse(isOnGolfCourse);
		return result;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#createAdjacentGolfCourseFeaturesResult(org.javatuples.Pair)
	 */
	@Override
	public AdjacentGolfCourseFeaturesResult createAdjacentGolfCourseFeaturesResult(
			Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult) {
		com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.AdjacentGolfCourseFeaturesResult result = GCFeaturesAdjacencyFactory.eINSTANCE
				.createAdjacentGolfCourseFeaturesResult();
		result.getFeatures().addAll(mapToEMFGolfCourseFeatures(adjacentFeaturesResult.getValue0()));
		result.setIsOnGolfCourse(adjacentFeaturesResult.getValue1());
		return result;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#createNearestHoleResult(org.javatuples.Pair)
	 */
	@Override
	public NearestHoleResult createNearestHoleResult(Pair<Optional<GolfCourseFeature>, Boolean> nearestHoleResult) {
		com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.NearestHoleResult result = GCFeaturesAdjacencyFactory.eINSTANCE
				.createNearestHoleResult();
		if (nearestHoleResult.getValue0().isPresent()) {
			result.setNearestHole(mapToEMFGolfCourseFeature(nearestHoleResult.getValue0().get()));
		}

		result.setIsOnGolfCourse(nearestHoleResult.getValue1());

		return result;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService#createIsWithinHolePerimeterResult(boolean, java.lang.Integer)
	 */
	@Override
	public IsWithinHolePerimeterResult createIsWithinHolePerimeterResult(boolean isWithinHolePerimeter,
			Integer holeNumber) {
		com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.IsWithinHolePerimeterResult result = GCFeaturesAdjacencyFactory.eINSTANCE.createIsWithinHolePerimeterResult();
		result.setIsWithinHolePerimeter(isWithinHolePerimeter);
		result.setHoleNumber(holeNumber.intValue());
		return result;
	}
	
	@SafeVarargs
	private List<String> mapGolfCourseFeatureTypeToCourseGPSServiceProjection(Set<GolfCourseFeatureType>... featureTypes) {
		// @formatter:off
		return Stream.of(featureTypes)
				.flatMap(Set::stream)
                .collect(Collectors.toSet())
                .stream()
                .map(ft -> ft.getLabel())
                .collect(Collectors.toList());
		// @formatter:on
	}
	
	private List<GolfCourseFeature> extractGolfCourseFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector courseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Set<GeometryType> geometryTypes) {
		
		List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();
		
		// Course level features
		for (GolfCourseFeatureType golfCourseFeatureType : golfCourseFeatureTypes) {
			
			Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> golfCourseFeatureShapeObjectOptional = extractShapeObjectForGolfCourseFeatureType(
					courseGPSVector, golfCourseFeatureType);

			if (golfCourseFeatureShapeObjectOptional.isPresent()) {
				logger.debug("Found " + golfCourseFeatureShapeObjectOptional.get().getShapeCount() + " shapes for: " + golfCourseFeatureType.name());

				golfCourseFeatures.addAll(extractGolfCourseFeatures(golfCourseFeatureShapeObjectOptional.get(), golfCourseFeatureType,
						geometryTypes, Optional.empty()));
			}
		}

		// Course hole level features
		if (courseGPSVector.getHoleCount() > 0 && courseGPSVector.getHoles() != null) {

			for (com.playertour.backend.golfcourse.model.golfcourse.Hole courseHole : courseGPSVector.getHoles()) {

				if (courseHole.getHoleNumber() > 0) {

					logger.debug("Processing hole number: " + courseHole.getHoleNumber());

					for (GolfCourseFeatureType golfCourseHoleFeatureType : golfCourseHoleFeatureTypes) {
						
						Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> golfCourseHoleFeatureShapeObjectOptional = extractShapeObjectForGolfCourseHoleFeatureType(
								courseHole, golfCourseHoleFeatureType);
						
						if (golfCourseHoleFeatureShapeObjectOptional.isPresent()) {
							logger.debug("Found " + golfCourseHoleFeatureShapeObjectOptional.get().getShapeCount() + " shapes for: " + golfCourseHoleFeatureType.name());

							golfCourseFeatures.addAll(
									extractGolfCourseFeatures(golfCourseHoleFeatureShapeObjectOptional.get(), golfCourseHoleFeatureType,
											geometryTypes, Optional.of(courseHole.getHoleNumber())));
						}
					}
				}
			}
		}

		return golfCourseFeatures;
	}
	
	private List<GolfCourseFeature> extractGolfCourseFeatures(com.playertour.backend.golfcourse.model.golfcourse.ShapeObject golfCourseFeatureShapeObject,
			GolfCourseFeatureType golfCourseFeatureType, Set<GeometryType> geometryTypes, Optional<Integer> holeNumber) {
		
		if (golfCourseFeatureShapeObject.getFeatureCollection() != null 
				&& golfCourseFeatureShapeObject.getFeatureCollection().getFeatures() != null 
				&& golfCourseFeatureShapeObject.getFeatureCollection().getFeatures().size() > 0) {
			
			List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();
			
			logger.debug("Feature collection for '" + golfCourseFeatureType.getInternalName() + "' feature type present!");
			
			for (com.playertour.backend.geojson.model.geojson.Feature geoJsonFeature : golfCourseFeatureShapeObject.getFeatureCollection().getFeatures()) {
				
				if (geoJsonFeature.getGeometry() != null 
						&& geoJsonFeature.getGeometry().getType() != null 
						&& geometryTypes.contains(GeometryType.valueOfOrNull(geoJsonFeature.getGeometry().getType()))) {
					
					GeometryType geometryType = GeometryType.valueOf(geoJsonFeature.getGeometry().getType());
					
					logger.debug("'" + geometryType.name() + "' geometry type present! Extracting...");
					
					/*
					if (geoJsonFeature.getProperties() != null) {
						// TODO: ? extract properties ? 
					}
					*/
					
					org.locationtech.spatial4j.shape.Shape shape = mapAsShape(
							geoJsonFeature,
							geometryType);
					
					if (shape != null) {
						logger.debug(shape.toString());
						
						addFeatureOrShape(golfCourseFeatures, 
								golfCourseFeatureType, 
								geometryType, 
								shape, 
								holeNumber);
					}
					
				} else {
					logger.debug("'" + golfCourseFeatureType.getInternalName() + "' projection is not of type '" + Arrays.toString(geometryTypes.toArray()) + "'! Skipping..");
				}
			}
			
			return golfCourseFeatures;
		}

		return Collections.emptyList();
	}
	
	/**
	 * Extracts features from course GPS vector GeoJSON.
	 * 
	 * Examples of feature types - see 'GolfCourseFeatureType' for full list:
	 *  
	 *  * Perimeter of golf course
	 *  
	 *  * Cross-cutting features - polygon geometry type only 
	 *     => i.e. features cross-cutting entire golf course, such as water hazards ("water", "lake", "ocean", "creek", "pond") and sand hazard(s) ("sand")
   	 * 
   	 *  * Features within perimeter of each hole - polygons only (!?)
   	 *   * Tee box(es)
   	 *   * Fairway
   	 *   * Rough
   	 *   * Bunker(s)
   	 *   * Green
	 * 
	 * @param courseGPSVectorFile
	 * @param golfCourseFeatureTypes
	 * @param golfCourseHoleFeatureTypes
	 * @param geometryTypes
	 * @param mapAsGeoJSON
	 * @return
	 * @throws IOException
	 * @throws InvalidShapeException
	 * @throws ParseException
	 */	
	private List<GolfCourseFeature> extractGolfCourseFeatures(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Set<GeometryType> geometryTypes)
			throws IOException, InvalidShapeException, ParseException {
		
		List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();

		JsonNode golfCourseGPSVectorJsonNode = OBJECT_MAPPER.readTree(golfCourseGPSVectorInputStream);

		// Course level features
		for (GolfCourseFeatureType golfCourseFeatureType : golfCourseFeatureTypes) {
			
			JsonNode shapeCountJsonNode = golfCourseGPSVectorJsonNode
					.at("/" + golfCourseFeatureType.getInternalName() + "/shapeCount");
			if (!shapeCountJsonNode.isMissingNode() && shapeCountJsonNode.isInt() && shapeCountJsonNode.asInt() > 1) {
				logger.debug("Found " + shapeCountJsonNode.asInt() + " shapes for: " + golfCourseFeatureType.name());
			}

			golfCourseFeatures.addAll(extractGolfCourseFeatures(golfCourseGPSVectorJsonNode, golfCourseFeatureType,
					geometryTypes, Optional.empty()));
		}

		// Course hole level features
		if (golfCourseGPSVectorJsonNode.has("holes")) {

			JsonNode golfCourseHolesGPSVectorJsonNode = golfCourseGPSVectorJsonNode.at("/holes");
			if (!golfCourseHolesGPSVectorJsonNode.isMissingNode() && golfCourseHolesGPSVectorJsonNode.isArray()) {

				for (JsonNode golfCourseHoleGPSVectorJsonNode : golfCourseHolesGPSVectorJsonNode) {

					if (golfCourseHoleGPSVectorJsonNode.has("holeNumber")) {
						int holeNumber = golfCourseHoleGPSVectorJsonNode.get("holeNumber").asInt();

						logger.debug("Processing hole number: " + holeNumber);

						for (GolfCourseFeatureType golfCourseHoleFeatureType : golfCourseHoleFeatureTypes) {
							
							JsonNode shapeCountJsonNode = golfCourseHoleGPSVectorJsonNode
									.at("/" + golfCourseHoleFeatureType.getInternalName() + "/shapeCount");
							if (!shapeCountJsonNode.isMissingNode() && shapeCountJsonNode.isInt() && shapeCountJsonNode.asInt() > 1) {
								logger.debug("Found " + shapeCountJsonNode.asInt() + " shapes for: " + golfCourseHoleFeatureType.name());
							}

							golfCourseFeatures.addAll(extractGolfCourseFeatures(golfCourseHoleGPSVectorJsonNode,
									golfCourseHoleFeatureType, geometryTypes, Optional.of(holeNumber)));
						}
					}
				}
			}
		}

		return golfCourseFeatures;
	}
	
	/**
	 * Extracts features of given type from course GPS vector GeoJSON.
	 * 
	 * @param gpsVectorJsonNode
	 * @param golfCourseFeatureType
	 * @param geometryTypes
	 * @param holeNumber
	 * @param mapAsGeoJSON
	 * @return
	 * @throws InvalidShapeException
	 * @throws IOException
	 * @throws ParseException
	 */
	private List<GolfCourseFeature> extractGolfCourseFeatures(JsonNode gpsVectorJsonNode,
			GolfCourseFeatureType golfCourseFeatureType, 
			Set<GeometryType> geometryTypes, 
			Optional<Integer> holeNumber) throws InvalidShapeException, IOException, ParseException {

		if (gpsVectorJsonNode.has(golfCourseFeatureType.getInternalName())) {
			List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();

			JsonNode geoJsonFeaturesJsonNode = gpsVectorJsonNode
					.at("/" + golfCourseFeatureType.getInternalName() + "/featureCollection/features");

			if (!geoJsonFeaturesJsonNode.isMissingNode() && geoJsonFeaturesJsonNode.isArray()) {

				logger.debug("'" + golfCourseFeatureType.getInternalName() + "' feature type present!");

				for (JsonNode geoJsonFeatureNode : geoJsonFeaturesJsonNode) {

					JsonNode geoJsonFeatureGeometryTypeNode = geoJsonFeatureNode.at("/geometry/type");

					if (!geoJsonFeatureGeometryTypeNode.isMissingNode() && geometryTypes
							.contains(GeometryType.valueOfOrNull(geoJsonFeatureGeometryTypeNode.asText()))) {

						GeometryType geometryType = GeometryType.valueOf(geoJsonFeatureGeometryTypeNode.asText());

						logger.debug("'" + geometryType.name() + "' geometry type present! Extracting...");

						JsonNode geoJsonFeatureGeometryJsonNode = geoJsonFeatureNode.at("/geometry");

						// remove 'eClass'
						if (!geoJsonFeatureGeometryJsonNode.at("/eClass").isMissingNode()) {
							((ObjectNode) geoJsonFeatureGeometryJsonNode).remove("eClass");
						}

						// TODO: extract properties
						if (geoJsonFeatureNode.has("properties")) {
							JsonNode geoJsonFeaturePropertiesJsonNode = geoJsonFeatureNode.at("/properties");

							logger.debug("'properties' present!");
						}

						org.locationtech.spatial4j.shape.Shape shape = GEOJSON_READER
								.read(geoJsonFeatureGeometryJsonNode);
						
						addFeatureOrShape(golfCourseFeatures, 
								golfCourseFeatureType, 
								geometryType, 
								shape, 
								holeNumber);

					} else {
						logger.debug("'" + golfCourseFeatureType.getInternalName() + "' projection is not of type '"
										+ Arrays.toString(geometryTypes.toArray()) + "'! Skipping..");
					}
				}
			}

			return golfCourseFeatures;
		}

		return Collections.emptyList();
	}

	// Golf Course level feature types
	private Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> extractShapeObjectForGolfCourseFeatureType(
			com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector courseGPSVector,
			GolfCourseFeatureType golfCourseFeatureType) {

		switch (golfCourseFeatureType) {

		case background:
			return Optional.ofNullable(courseGPSVector.getBackgrund());

		case bridge:
			return Optional.ofNullable(courseGPSVector.getBridge());

		case clubHouse:
			return Optional.ofNullable(courseGPSVector.getClubhouse());

		case creek:
			return Optional.ofNullable(courseGPSVector.getCreek());

		case lake:
			return Optional.ofNullable(courseGPSVector.getLake());

		case lava:
			return Optional.ofNullable(courseGPSVector.getLava());

		case ocean:
			return Optional.ofNullable(courseGPSVector.getOcean());

		case path:
			return Optional.ofNullable(courseGPSVector.getPath());

		case pond:
			return Optional.ofNullable(courseGPSVector.getPond());

		case rock:
			return Optional.ofNullable(courseGPSVector.getRock());

		case sand:
			return Optional.ofNullable(courseGPSVector.getSand());

		case tree:
			return Optional.ofNullable(courseGPSVector.getTree());

		case vegetation:
			return Optional.ofNullable(courseGPSVector.getVegetation());

		case water:
			return Optional.ofNullable(courseGPSVector.getWater());

		default:
			throw new IllegalArgumentException(golfCourseFeatureType.name() + " is not supported!");
		}
	}
	
	// Golf Course hole level feature types
	private Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> extractShapeObjectForGolfCourseHoleFeatureType(
			com.playertour.backend.golfcourse.model.golfcourse.Hole courseHoleGPSVector,
			GolfCourseFeatureType golfCourseHoleFeatureType) {

		switch (golfCourseHoleFeatureType) {

		case bunker:
			return Optional.ofNullable(courseHoleGPSVector.getBunker());

		case centralPath:
			return Optional.ofNullable(courseHoleGPSVector.getCentralPath());

		case fairway:
			return Optional.ofNullable(courseHoleGPSVector.getFairway());

		case green:
			return Optional.ofNullable(courseHoleGPSVector.getGreen());

		case greenCenter:
			return Optional.ofNullable(courseHoleGPSVector.getGreenCenter());

		case perimeter:
			return Optional.ofNullable(courseHoleGPSVector.getPerimeter());

		case teeBox:
			return Optional.ofNullable(courseHoleGPSVector.getTeebox());

		case teeBoxCenter:
			return Optional.ofNullable(courseHoleGPSVector.getTeeboxCenter());

		default:
			throw new IllegalArgumentException(golfCourseHoleFeatureType.name() + " is not supported!");
		}
	}
	
	private org.locationtech.spatial4j.shape.Shape mapAsShape(
			com.playertour.backend.geojson.model.geojson.Feature geoJsonFeature, GeometryType geometryType) {

		switch (geometryType) {
		case Polygon:
			return mapAsPolygonShape(geoJsonFeature.getGeometry());
		case LineString:
			return mapAsLineStringShape(geoJsonFeature.getGeometry());
		case Point:
			return mapAsPointShape(geoJsonFeature.getGeometry());
		default:
			throw new IllegalArgumentException(geometryType.name() + " is not supported!");
		}
	}
	
	private org.locationtech.spatial4j.shape.Shape mapAsPolygonShape(
			com.playertour.backend.geojson.model.geojson.AbstractGeometry geoJsonGeometry) {
		
		try {
			com.playertour.backend.geojson.model.geojson.Polygon polygonGeoJsonGeometry = (com.playertour.backend.geojson.model.geojson.Polygon) geoJsonGeometry;

			org.locationtech.spatial4j.shape.ShapeFactory.PolygonBuilder polygonShapeBuilder = JTS_SHAPE_FACTORY.polygon();

			for (Double[][] coordinates : polygonGeoJsonGeometry.getCoordinates()) {
				for (Double[] coordinate : coordinates) {
					polygonShapeBuilder = polygonShapeBuilder.pointLatLon(coordinate[1], coordinate[0]);
				}
			}

			return polygonShapeBuilder.build();
			
		} catch (Throwable t) {
			logger.warn(t.getMessage());
			
			return null;
		}
	}

	private org.locationtech.spatial4j.shape.Shape mapAsLineStringShape(
			com.playertour.backend.geojson.model.geojson.AbstractGeometry geoJsonGeometry) {
		
		try {
			com.playertour.backend.geojson.model.geojson.LineString lineStringGeoJsonGeometry = (com.playertour.backend.geojson.model.geojson.LineString) geoJsonGeometry;
			
			org.locationtech.spatial4j.shape.ShapeFactory.LineStringBuilder lineStringShapeBuilder = SpatialContext.GEO
					.getShapeFactory().lineString();

			for (Double[] coordinate : lineStringGeoJsonGeometry.getCoordinates()) {
				lineStringShapeBuilder = lineStringShapeBuilder.pointLatLon(coordinate[1], coordinate[0]);
			}

			return lineStringShapeBuilder.build();

		} catch (Throwable t) {
			logger.warn(t.getMessage());

			return null;
		}
	}

	private org.locationtech.spatial4j.shape.Shape mapAsPointShape(
			com.playertour.backend.geojson.model.geojson.AbstractGeometry geoJsonGeometry) {

		try {
			com.playertour.backend.geojson.model.geojson.Point pointGeoJsonGeometry = (com.playertour.backend.geojson.model.geojson.Point) geoJsonGeometry;

			return SpatialContext.GEO.getShapeFactory().pointLatLon(pointGeoJsonGeometry.getCoordinates().get(1),
					pointGeoJsonGeometry.getCoordinates().get(0));

		} catch (Throwable t) {
			logger.warn(t.getMessage());

			return null;
		}
	}
	
	/**
	 * Builds golf course features' adjacency graph by measuring distance between features and adding connections where distance is below threshold.
	 * 
	 * @param golfCourseFeatures
	 * @param excludeBackground
	 * @return
	 */	
	private GolfCourseFeaturesAdjacencyGraph buildAdjacencyGraph(List<GolfCourseFeature> golfCourseFeatures,
			boolean excludeBackground) {
		
		logger.debug("------------------------");
		logger.debug("Building golf course features' adjacency graph based on " + golfCourseFeatures.size() + " golf course features ...");
		
		GolfCourseFeaturesAdjacencyGraph adjacencyGraph = new GolfCourseFeaturesAdjacencyGraph();
		
		for (GolfCourseFeature golfCourseFeature1 : golfCourseFeatures) {
			
			if (excludeBackground && golfCourseFeature1.getFeatureType() == GolfCourseFeatureType.background) {
				continue;
			}
			
			adjacencyGraph.addFeature(golfCourseFeature1);
			
			for (GolfCourseFeature golfCourseFeature2 : golfCourseFeatures) {
				
				if (excludeBackground && golfCourseFeature2.getFeatureType() == GolfCourseFeatureType.background) {
					continue;
				}				
				
				if (golfCourseFeature1.hashCode() != golfCourseFeature2.hashCode()) {
					
					adjacencyGraph.addFeature(golfCourseFeature2);
					
					if (!adjacencyGraph.connectionExists(golfCourseFeature1, golfCourseFeature2)) {
						
						double distanceMeters = calculateDistanceInMeters(golfCourseFeature1, golfCourseFeature2);
						
						if (distanceMeters <= config.featureConnectionMinDistance()) {
							logger.debug("Adding connection between '" + golfCourseFeature1 + "' and '" + golfCourseFeature2 + "'");
							logger.debug("Distance: " + distanceMeters + " m");
							
							adjacencyGraph.addConnection(golfCourseFeature1, golfCourseFeature2, distanceMeters);
							
						} else {
							logger.debug("Distance between '" + golfCourseFeature1 + "' and '" + golfCourseFeature2 + "' is too large (" + distanceMeters + ")");
						}
						
						logger.debug("------------------------\n");
					}
				}
			}
		}
		
		return adjacencyGraph;
	}
	
	private GolfCourseFeaturesAdjacencyGraph buildAdjacencyGraph(List<GolfCourseFeature> golfCourseFeatures,
			boolean excludeBackground, Optional<Integer> holeNumberOptional) {
		
		logger.debug("------------------------");
		logger.debug("Building golf course features' adjacency graph based on " + golfCourseFeatures.size() + " golf course features " + (holeNumberOptional.isPresent() ? ("for hole number " + holeNumberOptional.get()) : "")  + " ...");
		
		GolfCourseFeaturesAdjacencyGraph adjacencyGraph = new GolfCourseFeaturesAdjacencyGraph();
		
		for (GolfCourseFeature golfCourseFeature1 : golfCourseFeatures) {
			
			if (excludeBackground && golfCourseFeature1.getFeatureType() == GolfCourseFeatureType.background) {
				continue;
			}
			
			if (holeNumberOptional.isPresent() && (golfCourseFeature1 instanceof GolfCourseHoleFeature
					&& ((GolfCourseHoleFeature)golfCourseFeature1).getHoleNumber().intValue() != holeNumberOptional.get().intValue())) {
				continue;
			}
			
			adjacencyGraph.addFeature(golfCourseFeature1);
			
			for (GolfCourseFeature golfCourseFeature2 : golfCourseFeatures) {
				
				if (excludeBackground && golfCourseFeature2.getFeatureType() == GolfCourseFeatureType.background) {
					continue;
				}
				
				if (holeNumberOptional.isPresent() && (golfCourseFeature2 instanceof GolfCourseHoleFeature
						&& ((GolfCourseHoleFeature)golfCourseFeature2).getHoleNumber().intValue() != holeNumberOptional.get().intValue())) {
					continue;
				}
				
				if (golfCourseFeature1.hashCode() != golfCourseFeature2.hashCode()) {
					
					adjacencyGraph.addFeature(golfCourseFeature2);
					
					if (!adjacencyGraph.connectionExists(golfCourseFeature1, golfCourseFeature2)) {
						
						double distanceMeters = calculateDistanceInMeters(golfCourseFeature1, golfCourseFeature2);
						
						if (distanceMeters <= config.featureConnectionMinDistance()) {
							logger.debug("Adding connection between '" + golfCourseFeature1 + "' and '" + golfCourseFeature2 + "'");
							logger.debug("Distance: " + distanceMeters + " m");
							
							adjacencyGraph.addConnection(golfCourseFeature1, golfCourseFeature2, distanceMeters);
							
						} else {
							logger.debug("Distance between '" + golfCourseFeature1 + "' and '" + golfCourseFeature2 + "' is too large (" + distanceMeters + ")");
						}
						
						logger.debug("------------------------\n");
					}
				}
			}
		}
		
		return adjacencyGraph;
	}	
	
	private double calculateDistanceInMeters(GolfCourseFeature golfCourseFeature1, GolfCourseFeature golfCourseFeature2) {		
		if (golfCourseFeature1.getShapeCount() > 1 || golfCourseFeature2.getShapeCount() > 1) {
			
			List<Double> distancesInMeters = new ArrayList<Double>();
			
			for (org.locationtech.spatial4j.shape.Shape golfCourseFeature1Shape : golfCourseFeature1.getShapes()) {
				
				for (org.locationtech.spatial4j.shape.Shape golfCourseFeature2Shape : golfCourseFeature2.getShapes()) {
					
					// @formatter:off
					double distanceInMeters = calculateDistanceInMeters(golfCourseFeature1, 
							golfCourseFeature1Shape,
							golfCourseFeature2, 
							golfCourseFeature2Shape);
					// @formatter:on
					
					distancesInMeters.add(Double.valueOf(distanceInMeters));
				}
			}
			
			Collections.sort(distancesInMeters);
			
			logger.debug("Distances in meters: " + Arrays.toString(distancesInMeters.toArray()));
			
			return distancesInMeters.get(0);
			
		} else {
			
			// @formatter:off
			return calculateDistanceInMeters(golfCourseFeature1, 
					golfCourseFeature1.getShape(), 
					golfCourseFeature2, 
					golfCourseFeature2.getShape());
			// @formatter:on
		}
	}
	
	private double calculateDistanceInMeters(GolfCourseFeature golfCourseFeature1,
			org.locationtech.spatial4j.shape.Shape golfCourseFeature1Shape, GolfCourseFeature golfCourseFeature2,
			org.locationtech.spatial4j.shape.Shape golfCourseFeature2Shape) {

		org.locationtech.jts.geom.Geometry shape1Geometry = JTS_SPATIAL_CONTEXT.getShapeFactory()
				.getGeometryFrom(golfCourseFeature1Shape);

		org.locationtech.jts.geom.Geometry shape2Geometry = JTS_SPATIAL_CONTEXT.getShapeFactory()
				.getGeometryFrom(golfCourseFeature2Shape);

		if ((golfCourseFeature1 instanceof GolfCourseHoleFeature
				&& golfCourseFeature1.getFeatureType() == GolfCourseFeatureType.perimeter
				&& golfCourseFeature2 instanceof GolfCourseHoleFeature
				&& GOLFCOURSE_HOLE_FEATURE_TYPES.contains(golfCourseFeature2.getFeatureType())
		// && ((GolfCourseHoleFeature)golfCourseFeature1).getHoleNumber() == ((GolfCourseHoleFeature)golfCourseFeature2).getHoleNumber()
		) || (golfCourseFeature2 instanceof GolfCourseHoleFeature
				&& golfCourseFeature2.getFeatureType() == GolfCourseFeatureType.perimeter
				&& golfCourseFeature1 instanceof GolfCourseHoleFeature
				&& GOLFCOURSE_HOLE_FEATURE_TYPES.contains(golfCourseFeature1.getFeatureType())
		// && ((GolfCourseHoleFeature)golfCourseFeature2).getHoleNumber() == ((GolfCourseHoleFeature)golfCourseFeature1).getHoleNumber()
		)) {
			return calculateDistanceInMeters(shape1Geometry.getBoundary(), shape2Geometry.getBoundary());
		} else {
			return calculateDistanceInMeters(shape1Geometry, shape2Geometry);
		}
	}
	
	private double calculateDistanceInMeters(org.locationtech.jts.geom.Geometry shape1Geometry, 
			org.locationtech.jts.geom.Geometry shape2Geometry) {
		return ((shape1Geometry.distance(shape2Geometry) * DistanceUtils.DEG_TO_KM) * 1000);
	}
	
	private double calculateDistanceInMeters(org.locationtech.spatial4j.shape.Point point, double longitude,
			double latitude) {
		return ((JTS_SPATIAL_CONTEXT.calcDistance(point, longitude, latitude) * DistanceUtils.DEG_TO_KM) * 1000);
	}
	
	private Pair<List<GolfCourseFeature>, Boolean> listAdjacentFeatures(List<GolfCourseFeature> golfCourseFeatures, double longitude, double latitude) {
		if (config.withinGolfcourseCheck() 
				&& !isWithinGolfCoursePerimeter(golfCourseFeatures, longitude, latitude)) {
			return Pair.with(Collections.emptyList(), Boolean.FALSE);
		}
		
		///////////////////////////////////////////////////////////////////////////////////////////////////////
		// TODO: optimize by storing entire pre-processed graph and shapes' spatial index for faster searching
		
		GolfCourseFeaturesAdjacencyGraph adjacencyGraph = buildAdjacencyGraph(golfCourseFeatures, true);
		logger.debug(adjacencyGraph.toString());
		
		// TODO: when optimized, all of the above, intermediate steps (including earlier step of extracting golf course features), 
		// 	would be replaced by retrieving pre-processed graph and/or shapes' spatial index and finding closest feature based on that
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		
		Optional<GolfCourseFeature> gcFeatureOptional = findClosestFeature(adjacencyGraph, longitude, latitude, false);
		if (gcFeatureOptional.isPresent()) { 
			List<GolfCourseFeature> adjacentFeatures = breadthFirstSearch(adjacencyGraph, gcFeatureOptional.get());
			return Pair.with(adjacentFeatures, Boolean.TRUE);
		} else {
			return Pair.with(Collections.emptyList(), Boolean.TRUE);
		}
	}
	
	private Pair<List<GolfCourseFeature>, Boolean> listAdjacentFeatures(List<GolfCourseFeature> golfCourseFeatures, double longitude, double latitude, Optional<Integer> holeNumberOptional) {
		if (config.withinGolfcourseCheck() 
				&& !isWithinGolfCoursePerimeter(golfCourseFeatures, longitude, latitude)) {
			return Pair.with(Collections.emptyList(), Boolean.FALSE);
		}
		
		///////////////////////////////////////////////////////////////////////////////////////////////////////
		// TODO: optimize by storing entire pre-processed graph and shapes' spatial index for faster searching
		
		GolfCourseFeaturesAdjacencyGraph adjacencyGraph = buildAdjacencyGraph(golfCourseFeatures, true, holeNumberOptional);
		logger.debug(adjacencyGraph.toString());
		
		// TODO: when optimized, all of the above, intermediate steps (including earlier step of extracting golf course features), 
		// 	would be replaced by retrieving pre-processed graph and/or shapes' spatial index and finding closest feature based on that
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		
		Optional<GolfCourseFeature> gcFeatureOptional = findClosestFeature(adjacencyGraph, longitude, latitude, false, holeNumberOptional);
		if (gcFeatureOptional.isPresent()) { 
			List<GolfCourseFeature> adjacentFeatures = breadthFirstSearch(adjacencyGraph, gcFeatureOptional.get());
			return Pair.with(adjacentFeatures, Boolean.TRUE);
		} else {
			return Pair.with(Collections.emptyList(), Boolean.TRUE);
		}
	}	
	
	private Pair<Optional<GolfCourseFeature>, Boolean> findClosestFeature(List<GolfCourseFeature> golfCourseFeatures, double longitude, double latitude, boolean noMaxDistance) {
		if (config.withinGolfcourseCheck() 
				&& !isWithinGolfCoursePerimeter(golfCourseFeatures, longitude, latitude)) {
			return Pair.with(Optional.empty(), Boolean.FALSE);
		}		
		
		///////////////////////////////////////////////////////////////////////////////////////////////////////
		// TODO: optimize by storing entire pre-processed graph and shapes' spatial index for faster searching
		
		GolfCourseFeaturesAdjacencyGraph adjacencyGraph = buildAdjacencyGraph(golfCourseFeatures, true);
		
		// TODO: when optimized, all of the above, intermediate steps (including earlier step of extracting golf course features), 
		// 	would be replaced by retrieving pre-processed graph and/or shapes' spatial index and finding closest feature based on that
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		
		Optional<GolfCourseFeature> closestFeatureOptional = findClosestFeature(adjacencyGraph, longitude, latitude, noMaxDistance);
		return Pair.with(closestFeatureOptional, Boolean.TRUE);
	}
	
	private Optional<GolfCourseFeature> findClosestFeature(GolfCourseFeaturesAdjacencyGraph graph, double longitude, double latitude, boolean noMaxDistance) {
		SortedMap<Double, List<GolfCourseFeature>> perFeatureDistancesInMeters = new TreeMap<Double, List<GolfCourseFeature>>();
		
		for (GolfCourseFeature golfCourseFeature : graph.getFeatures()) {
			for (org.locationtech.spatial4j.shape.Shape golfCourseFeatureShape : golfCourseFeature.getShapes()) {
				
				GeometryType geometryType = golfCourseFeature.getGeometryType();
				
				double distanceInMeters = calculateDistanceInMeters(golfCourseFeatureShape, geometryType, longitude, latitude);
				
				if (distanceInMeters <= config.featureClosestMinDistance()) {
					logger.debug("------------------------------");
					logger.debug("Minimum distance criterion met");
					logger.debug("Distance: " + distanceInMeters);
					logger.debug(golfCourseFeature.toString());
					logger.debug("------------------------------");
					return Optional.of(golfCourseFeature);
				} else if (!noMaxDistance && (distanceInMeters >= config.featureClosestMaxDistance())) {
					continue;
					//break;
				}
				
				if (perFeatureDistancesInMeters.containsKey(Double.valueOf(distanceInMeters))) {
					perFeatureDistancesInMeters.get(Double.valueOf(distanceInMeters)).add(golfCourseFeature);
				} else {
					perFeatureDistancesInMeters.put(Double.valueOf(distanceInMeters), new ArrayList<GolfCourseFeature>(List.of(golfCourseFeature)));
				}
			}
		}
		
		perFeatureDistancesInMeters.forEach((key, value) -> {
			logger.debug("-------------------------");
			logger.debug(key.toString());
			logger.debug("=");
			logger.debug(value.toString());
			logger.debug("-------------------------");
		});
		
		if (!perFeatureDistancesInMeters.isEmpty()) {
			return Optional.of(perFeatureDistancesInMeters.get(perFeatureDistancesInMeters.firstKey()).get(0));
		}
		
		return Optional.empty();
	}
	
	private Optional<GolfCourseFeature> findClosestFeature(GolfCourseFeaturesAdjacencyGraph graph, double longitude, double latitude, boolean noMaxDistance, Optional<Integer> holeNumberOptional) {		
		SortedMap<Double, List<GolfCourseFeature>> perFeatureDistancesInMeters = new TreeMap<Double, List<GolfCourseFeature>>();
		
		FEATURES:
		for (GolfCourseFeature golfCourseFeature : graph.getFeatures()) {
			
			if (holeNumberOptional.isPresent() && (golfCourseFeature instanceof GolfCourseHoleFeature
					&& ((GolfCourseHoleFeature)golfCourseFeature).getHoleNumber().intValue() != holeNumberOptional.get().intValue())) {
				continue;				
			}
			
			FEATURE_SHAPES:
			for (org.locationtech.spatial4j.shape.Shape golfCourseFeatureShape : golfCourseFeature.getShapes()) {
				
				GeometryType geometryType = golfCourseFeature.getGeometryType();
				
				double distanceInMeters = calculateDistanceInMeters(golfCourseFeatureShape, geometryType, longitude, latitude);
				
				if (distanceInMeters <= config.featureClosestMinDistance()) {
					logger.debug("------------------------------");
					logger.debug("Minimum distance criterion met");
					logger.debug("Distance: " + distanceInMeters);
					logger.debug(golfCourseFeature.toString());
					logger.debug("------------------------------");
					return Optional.of(golfCourseFeature);
				} else if (distanceInMeters >= 500) { // TODO: parameterize this
					break FEATURE_SHAPES;					
				} else if (!noMaxDistance && (distanceInMeters >= config.featureClosestMaxDistance())) {
					continue;
					//break;
				}				
								
				if (perFeatureDistancesInMeters.containsKey(Double.valueOf(distanceInMeters))) {
					perFeatureDistancesInMeters.get(Double.valueOf(distanceInMeters)).add(golfCourseFeature);
				} else {
					perFeatureDistancesInMeters.put(Double.valueOf(distanceInMeters), new ArrayList<GolfCourseFeature>(List.of(golfCourseFeature)));
				}
				
				if (perFeatureDistancesInMeters.size() >= 5) { // TODO: parameterize this
					break FEATURES;
				}
			}
		}
		
		perFeatureDistancesInMeters.forEach((key, value) -> {
			logger.debug("-------------------------");
			logger.debug(key.toString());
			logger.debug("=");
			logger.debug(value.toString());
			logger.debug("-------------------------");
		});
		
		if (!perFeatureDistancesInMeters.isEmpty()) {
			return Optional.of(perFeatureDistancesInMeters.get(perFeatureDistancesInMeters.firstKey()).get(0));
		}
		
		return Optional.empty();
	}	
	
	private double calculateDistanceInMeters(org.locationtech.spatial4j.shape.Shape golfCourseFeatureShape, 
			GeometryType geometryType,
			double longitude, 
			double latitude) {
		
		if (geometryType == GeometryType.Point) {
			
			return calculateDistanceInMeters(((org.locationtech.spatial4j.shape.Point) golfCourseFeatureShape), longitude, latitude);
			
		} else {
			org.locationtech.spatial4j.shape.Point currentLocationPointShape = SpatialContext.GEO.getShapeFactory().pointLatLon(latitude, longitude);
			
			org.locationtech.jts.geom.Geometry currentLocationPointShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory().getGeometryFrom(currentLocationPointShape);
			
			org.locationtech.jts.geom.Geometry golfCourseFeatureShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory().getGeometryFrom(golfCourseFeatureShape);
			
			return calculateDistanceInMeters(golfCourseFeatureShapeGeometry.getBoundary(), currentLocationPointShapeGeometry);			
		}
	}
	
	/**
	 * Adds new feature or adds new shape to existing feature - as there can be multiple shapes per feature.
	 * 
	 * @param golfCourseFeatures
	 * @param golfCourseFeatureType
	 * @param geometryType
	 * @param shape
	 * @param holeNumber
	 */
	private void addFeatureOrShape(List<GolfCourseFeature> golfCourseFeatures, 
			GolfCourseFeatureType golfCourseFeatureType, 
			GeometryType geometryType, 
			org.locationtech.spatial4j.shape.Shape shape, 
			Optional<Integer> holeNumber) {
		
		if (holeNumber.isPresent()) {
			GolfCourseHoleFeature feature = new GolfCourseHoleFeature(golfCourseFeatureType,
					geometryType, shape, holeNumber.get());
			
			if (golfCourseFeatures.contains(feature)) {
				feature = (GolfCourseHoleFeature) golfCourseFeatures.get(golfCourseFeatures.indexOf(feature));
				feature.addShape(shape);
				
				logger.debug("Added new shape for: " + feature);
				
			} else {
				golfCourseFeatures.add(feature);
			}
			
		} else {
			GolfCourseFeature feature = new GolfCourseFeature(golfCourseFeatureType, geometryType,
					shape);
			
			if (golfCourseFeatures.contains(feature)) {
				feature = golfCourseFeatures.get(golfCourseFeatures.indexOf(feature));
				feature.addShape(shape);
				
				logger.debug("Added new shape for: " + feature);
				
			} else {
				golfCourseFeatures.add(feature);
			}
		}
	}
	
	private <E> List<GolfCourseFeature> breadthFirstSearch(GolfCourseFeaturesAdjacencyGraph graph, GolfCourseFeature source) {
		Queue<GolfCourseFeature> queue = new LinkedList<GolfCourseFeature>();
		Set<GolfCourseFeature> enqueued = new HashSet<GolfCourseFeature>();
		List<GolfCourseFeature> visited = new ArrayList<GolfCourseFeature>();

		queue.add(source);

		enqueued.add(source);

		while (!queue.isEmpty()) {
			GolfCourseFeature gcFeature = queue.remove();
			visited.add(gcFeature);

			List<GolfCourseFeatureConnection> neighborConnections = graph.sortedConnections(gcFeature);
			for (GolfCourseFeatureConnection connection : neighborConnections) {
				if (!enqueued.contains(connection.getDestination())) {
					queue.add(connection.getDestination());
					enqueued.add(connection.getDestination());
				}
			}
		}

		return visited;
	}
	
	private Optional<GolfCourseFeature> getBackgroundFeatureType(List<GolfCourseFeature> golfCourseFeatures) {
		// @formatter:off
		return golfCourseFeatures
				.stream()
				.filter(gcFeature -> gcFeature.getFeatureType() == GolfCourseFeatureType.background)
				.findFirst();
		// @formatter:on
	}

	private boolean isWithinGolfCoursePerimeter(List<GolfCourseFeature> golfCourseFeatures, double longitude,
			double latitude) {
		Optional<GolfCourseFeature> gcBackgroundFeatureOptional = getBackgroundFeatureType(golfCourseFeatures);
		if (gcBackgroundFeatureOptional.isPresent()) {
			return isWithinGolfCoursePerimeter(gcBackgroundFeatureOptional.get(), longitude, latitude);
		} else {
			return false;
		}
	}

	private boolean isWithinGolfCoursePerimeter(GolfCourseFeature golfCourseBackgroundFeature, double longitude,
			double latitude) {
		Objects.requireNonNull(golfCourseBackgroundFeature, "Golf course background feature is required!");

		org.locationtech.spatial4j.shape.Point pointShape = SpatialContext.GEO.getShapeFactory().pointLatLon(latitude,
				longitude);

		org.locationtech.jts.geom.Geometry pointShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory()
				.getGeometryFrom(pointShape);

		org.locationtech.jts.geom.Geometry backgroundShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory()
				.getGeometryFrom(golfCourseBackgroundFeature.getShape());
		
		return pointShapeGeometry.isWithinDistance(backgroundShapeGeometry,
				((config.withinGolfcourseDistance() / 1000) * DistanceUtils.KM_TO_DEG));
	}
	
	private boolean isWithinHolePerimeter(List<GolfCourseFeature> golfCourseFeatures, Integer holeNumber, double longitude,
			double latitude) {
		
		Optional<GolfCourseFeature> gcHoleFeatureOptional = getHolePerimeterFeatureType(golfCourseFeatures, holeNumber);
		if (gcHoleFeatureOptional.isPresent()) {
			return isWithinHolePerimeter((GolfCourseHoleFeature)gcHoleFeatureOptional.get(), longitude, latitude);
		} else {
			return false;
		}
	}
	
	private Optional<GolfCourseFeature> getHolePerimeterFeatureType(List<GolfCourseFeature> golfCourseFeatures, Integer holeNumber) {
		// @formatter:off
		return golfCourseFeatures
				.stream()
				.filter(gcFeature -> (gcFeature instanceof GolfCourseHoleFeature
										&& gcFeature.getFeatureType() == GolfCourseFeatureType.perimeter
										&& ((GolfCourseHoleFeature)gcFeature).getHoleNumber().intValue() == holeNumber.intValue()))
				.findFirst();
		// @formatter:on
	}
	
	private boolean isWithinHolePerimeter(GolfCourseHoleFeature golfCourseHoleFeature, double longitude, double latitude) {
		Objects.requireNonNull(golfCourseHoleFeature, "Golf course hole feature is required!");

		org.locationtech.spatial4j.shape.Point pointShape = SpatialContext.GEO.getShapeFactory().pointLatLon(latitude,
				longitude);

		org.locationtech.jts.geom.Geometry pointShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory()
				.getGeometryFrom(pointShape);

		org.locationtech.jts.geom.Geometry holeShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory()
				.getGeometryFrom(golfCourseHoleFeature.getShape());
		
		return pointShapeGeometry.isWithinDistance(holeShapeGeometry, ((config.withinHolePerimeterDistance() / 1000) * DistanceUtils.KM_TO_DEG));
	}	
	
	private List<GolfCourseFeature> extractHoleFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector courseGPSVector,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Set<GeometryType> geometryTypes, 
			Integer holeNumber) {
		
		List<GolfCourseFeature> holeFeatures = new ArrayList<GolfCourseFeature>();
		
		// Course level features - 'background' only			
		Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> golfCourseFeatureShapeObjectOptional = extractShapeObjectForGolfCourseFeatureType(
				courseGPSVector, GolfCourseFeatureType.background);

		if (golfCourseFeatureShapeObjectOptional.isPresent()) {
			logger.debug("Found " + golfCourseFeatureShapeObjectOptional.get().getShapeCount() + " shapes for: " + GolfCourseFeatureType.background.name());

			holeFeatures.addAll(extractGolfCourseFeatures(golfCourseFeatureShapeObjectOptional.get(), GolfCourseFeatureType.background,
					geometryTypes, Optional.empty()));
		}

		// Course hole level features
		if (courseGPSVector.getHoleCount() > 0 && courseGPSVector.getHoles() != null) {

			for (com.playertour.backend.golfcourse.model.golfcourse.Hole courseHole : courseGPSVector.getHoles()) {

				if (holeNumber.intValue() == courseHole.getHoleNumber()) {

					logger.debug("Processing hole number: " + courseHole.getHoleNumber());

					for (GolfCourseFeatureType golfCourseHoleFeatureType : golfCourseHoleFeatureTypes) {
						
						Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> golfCourseHoleFeatureShapeObjectOptional = extractShapeObjectForGolfCourseHoleFeatureType(
								courseHole, golfCourseHoleFeatureType);
						
						if (golfCourseHoleFeatureShapeObjectOptional.isPresent()) {
							logger.debug("Found " + golfCourseHoleFeatureShapeObjectOptional.get().getShapeCount() + " shapes for: " + golfCourseHoleFeatureType.name());

							holeFeatures.addAll(
									extractGolfCourseFeatures(golfCourseHoleFeatureShapeObjectOptional.get(), golfCourseHoleFeatureType,
											geometryTypes, Optional.of(courseHole.getHoleNumber())));
						}
					}
				}
			}
		}

		return holeFeatures;		
	}
	
	private List<GolfCourseFeature> extractHoleFeatures(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Set<GeometryType> geometryTypes, 
			Integer holeNumber)
			throws IOException, InvalidShapeException, ParseException {
		
		List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();

		JsonNode golfCourseGPSVectorJsonNode = OBJECT_MAPPER.readTree(golfCourseGPSVectorInputStream);

		// Course level features - 'background' only
		JsonNode backgroundShapeCountJsonNode = golfCourseGPSVectorJsonNode.at("/" + GolfCourseFeatureType.background.getInternalName() + "/shapeCount");
		if (!backgroundShapeCountJsonNode.isMissingNode() && backgroundShapeCountJsonNode.isInt() && backgroundShapeCountJsonNode.asInt() > 1) {
			logger.debug("Found " + backgroundShapeCountJsonNode.asInt() + " shapes for: " + GolfCourseFeatureType.background.name());
		}

		golfCourseFeatures.addAll(extractGolfCourseFeatures(golfCourseGPSVectorJsonNode, GolfCourseFeatureType.background, geometryTypes, Optional.empty()));			

		// Course hole level features
		if (golfCourseGPSVectorJsonNode.has("holes")) {

			JsonNode golfCourseHolesGPSVectorJsonNode = golfCourseGPSVectorJsonNode.at("/holes");
			if (!golfCourseHolesGPSVectorJsonNode.isMissingNode() && golfCourseHolesGPSVectorJsonNode.isArray()) {

				for (JsonNode golfCourseHoleGPSVectorJsonNode : golfCourseHolesGPSVectorJsonNode) {

					if (golfCourseHoleGPSVectorJsonNode.has("holeNumber") && 
							(holeNumber.intValue() == golfCourseHoleGPSVectorJsonNode.get("holeNumber").asInt())) {

						logger.debug("Processing hole number: " + holeNumber);

						for (GolfCourseFeatureType golfCourseHoleFeatureType : golfCourseHoleFeatureTypes) {
							
							JsonNode shapeCountJsonNode = golfCourseHoleGPSVectorJsonNode
									.at("/" + golfCourseHoleFeatureType.getInternalName() + "/shapeCount");
							if (!shapeCountJsonNode.isMissingNode() && shapeCountJsonNode.isInt() && shapeCountJsonNode.asInt() > 1) {
								logger.debug("Found " + shapeCountJsonNode.asInt() + " shapes for: " + golfCourseHoleFeatureType.name());
							}

							golfCourseFeatures.addAll(extractGolfCourseFeatures(golfCourseHoleGPSVectorJsonNode,
									golfCourseHoleFeatureType, geometryTypes, Optional.of(holeNumber)));
						}
					}
				}
			}
		}

		return golfCourseFeatures;
	}	
	
	private Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mapAsGeoJSONFeatures(com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector courseGPSVector,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Set<GeometryType> geometryTypes) {
		
		List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();
		
		List<org.geojson.Feature> geoJSONFeatures = new ArrayList<org.geojson.Feature>();
		
		// Course level features
		for (GolfCourseFeatureType golfCourseFeatureType : golfCourseFeatureTypes) {
			
			Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> golfCourseFeatureShapeObjectOptional = extractShapeObjectForGolfCourseFeatureType(
					courseGPSVector, golfCourseFeatureType);

			if (golfCourseFeatureShapeObjectOptional.isPresent()) {
				logger.debug("Found " + golfCourseFeatureShapeObjectOptional.get().getShapeCount() + " shapes for: " + golfCourseFeatureType.name());
				
				Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple = mapAsGeoJSONFeatures(golfCourseFeatureShapeObjectOptional.get(), golfCourseFeatureType,
						geometryTypes, Optional.empty());
				
				golfCourseFeatures.addAll(mappedFeaturesTuple.getValue0());
				geoJSONFeatures.addAll(mappedFeaturesTuple.getValue1());
			}
		}

		// Course hole level features
		if (courseGPSVector.getHoleCount() > 0 && courseGPSVector.getHoles() != null) {

			for (com.playertour.backend.golfcourse.model.golfcourse.Hole courseHole : courseGPSVector.getHoles()) {

				if (courseHole.getHoleNumber() > 0) {

					logger.debug("Processing hole number: " + courseHole.getHoleNumber());

					for (GolfCourseFeatureType golfCourseHoleFeatureType : golfCourseHoleFeatureTypes) {
						
						Optional<com.playertour.backend.golfcourse.model.golfcourse.ShapeObject> golfCourseHoleFeatureShapeObjectOptional = extractShapeObjectForGolfCourseHoleFeatureType(
								courseHole, golfCourseHoleFeatureType);
						
						if (golfCourseHoleFeatureShapeObjectOptional.isPresent()) {
							logger.debug("Found " + golfCourseHoleFeatureShapeObjectOptional.get().getShapeCount() + " shapes for: " + golfCourseHoleFeatureType.name());
							
							Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple = mapAsGeoJSONFeatures(golfCourseHoleFeatureShapeObjectOptional.get(), golfCourseHoleFeatureType,
									geometryTypes, Optional.of(courseHole.getHoleNumber()));

							golfCourseFeatures.addAll(mappedFeaturesTuple.getValue0());
							geoJSONFeatures.addAll(mappedFeaturesTuple.getValue1());
						}
					}
				}
			}
		}
		
		return Pair.with(golfCourseFeatures, geoJSONFeatures);
	}
	
	private Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mapAsGeoJSONFeatures(com.playertour.backend.golfcourse.model.golfcourse.ShapeObject golfCourseFeatureShapeObject,
			GolfCourseFeatureType golfCourseFeatureType, 
			Set<GeometryType> geometryTypes, 
			Optional<Integer> holeNumberOptional) {
		
		if (golfCourseFeatureShapeObject.getFeatureCollection() != null 
				&& golfCourseFeatureShapeObject.getFeatureCollection().getFeatures() != null 
				&& golfCourseFeatureShapeObject.getFeatureCollection().getFeatures().size() > 0) {
			
			List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();
			
			List<org.geojson.Feature> geoJSONFeatures = new ArrayList<org.geojson.Feature>();
			
			logger.debug("Feature collection for '" + golfCourseFeatureType.getInternalName() + "' feature type present!");
			
			for (com.playertour.backend.geojson.model.geojson.Feature geoJsonFeature : golfCourseFeatureShapeObject.getFeatureCollection().getFeatures()) {
				
				if (geoJsonFeature.getGeometry() != null 
						&& geoJsonFeature.getGeometry().getType() != null 
						&& geometryTypes.contains(GeometryType.valueOfOrNull(geoJsonFeature.getGeometry().getType()))) {
					
					GeometryType geometryType = GeometryType.valueOf(geoJsonFeature.getGeometry().getType());
					
					logger.debug("'" + geometryType.name() + "' geometry type present! Extracting...");
					
					org.locationtech.spatial4j.shape.Shape shape = mapAsShape(
							geoJsonFeature,
							geometryType);
					
					if (shape != null) {
						logger.debug(shape.toString());
						
						addFeatureOrShape(golfCourseFeatures, 
								golfCourseFeatureType, 
								geometryType, 
								shape, 
								holeNumberOptional);
						
						mapAsGeoJSONFeature(geoJSONFeatures, geometryType, shape, 
								Optional.of(constructGeoJSONFeatureProperties(golfCourseFeatureType, holeNumberOptional)));
					}
					
				} else {
					logger.debug("'" + golfCourseFeatureType.getInternalName() + "' projection is not of type '" + Arrays.toString(geometryTypes.toArray()) + "'! Skipping..");
				}
			}
			
			return Pair.with(golfCourseFeatures, geoJSONFeatures);
		}
		
		return Pair.with(Collections.emptyList(), Collections.emptyList());
	}
	
	private Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mapAsGeoJSONFeatures(InputStream golfCourseGPSVectorInputStream,
			Set<GolfCourseFeatureType> golfCourseFeatureTypes, 
			Set<GolfCourseFeatureType> golfCourseHoleFeatureTypes,
			Set<GeometryType> geometryTypes)
			throws IOException, InvalidShapeException, ParseException {
		
		List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();
		
		List<org.geojson.Feature> geoJSONFeatures = new ArrayList<org.geojson.Feature>();

		JsonNode golfCourseGPSVectorJsonNode = OBJECT_MAPPER.readTree(golfCourseGPSVectorInputStream);

		// Course level features
		for (GolfCourseFeatureType golfCourseFeatureType : golfCourseFeatureTypes) {
			
			JsonNode shapeCountJsonNode = golfCourseGPSVectorJsonNode
					.at("/" + golfCourseFeatureType.getInternalName() + "/shapeCount");
			if (!shapeCountJsonNode.isMissingNode() && shapeCountJsonNode.isInt() && shapeCountJsonNode.asInt() > 1) {
				logger.debug("Found " + shapeCountJsonNode.asInt() + " shapes for: " + golfCourseFeatureType.name());
			}

			Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple = mapAsGeoJSONFeatures(golfCourseGPSVectorJsonNode, 
					golfCourseFeatureType,
					geometryTypes, 
					Optional.empty());
			
			golfCourseFeatures.addAll(mappedFeaturesTuple.getValue0());
			geoJSONFeatures.addAll(mappedFeaturesTuple.getValue1());
		}

		// Course hole level features
		if (golfCourseGPSVectorJsonNode.has("holes")) {

			JsonNode golfCourseHolesGPSVectorJsonNode = golfCourseGPSVectorJsonNode.at("/holes");
			if (!golfCourseHolesGPSVectorJsonNode.isMissingNode() && golfCourseHolesGPSVectorJsonNode.isArray()) {

				for (JsonNode golfCourseHoleGPSVectorJsonNode : golfCourseHolesGPSVectorJsonNode) {

					if (golfCourseHoleGPSVectorJsonNode.has("holeNumber")) {
						int holeNumber = golfCourseHoleGPSVectorJsonNode.get("holeNumber").asInt();

						logger.debug("Processing hole number: " + holeNumber);

						for (GolfCourseFeatureType golfCourseHoleFeatureType : golfCourseHoleFeatureTypes) {
							
							JsonNode shapeCountJsonNode = golfCourseHoleGPSVectorJsonNode
									.at("/" + golfCourseHoleFeatureType.getInternalName() + "/shapeCount");
							if (!shapeCountJsonNode.isMissingNode() && shapeCountJsonNode.isInt() && shapeCountJsonNode.asInt() > 1) {
								logger.debug("Found " + shapeCountJsonNode.asInt() + " shapes for: " + golfCourseHoleFeatureType.name());
							}
							
							Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple = mapAsGeoJSONFeatures(golfCourseHoleGPSVectorJsonNode, 
									golfCourseHoleFeatureType,
									geometryTypes, 
									Optional.of(holeNumber));

							golfCourseFeatures.addAll(mappedFeaturesTuple.getValue0());
							geoJSONFeatures.addAll(mappedFeaturesTuple.getValue1());
						}
					}
				}
			}
		}

		return Pair.with(golfCourseFeatures, geoJSONFeatures);
	}
	
	private Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mapAsGeoJSONFeatures(JsonNode gpsVectorJsonNode,
			GolfCourseFeatureType golfCourseFeatureType, 
			Set<GeometryType> geometryTypes, 
			Optional<Integer> holeNumberOptional) throws InvalidShapeException, IOException, ParseException {

		if (gpsVectorJsonNode.has(golfCourseFeatureType.getInternalName())) {
			List<GolfCourseFeature> golfCourseFeatures = new ArrayList<GolfCourseFeature>();
			
			List<org.geojson.Feature> geoJSONFeatures = new ArrayList<org.geojson.Feature>();

			JsonNode geoJsonFeaturesJsonNode = gpsVectorJsonNode
					.at("/" + golfCourseFeatureType.getInternalName() + "/featureCollection/features");

			if (!geoJsonFeaturesJsonNode.isMissingNode() && geoJsonFeaturesJsonNode.isArray()) {

				logger.debug("'" + golfCourseFeatureType.getInternalName() + "' feature type present!");

				for (JsonNode geoJsonFeatureNode : geoJsonFeaturesJsonNode) {

					JsonNode geoJsonFeatureGeometryTypeNode = geoJsonFeatureNode.at("/geometry/type");

					if (!geoJsonFeatureGeometryTypeNode.isMissingNode() && geometryTypes
							.contains(GeometryType.valueOfOrNull(geoJsonFeatureGeometryTypeNode.asText()))) {

						GeometryType geometryType = GeometryType.valueOf(geoJsonFeatureGeometryTypeNode.asText());

						logger.debug("'" + geometryType.name() + "' geometry type present! Extracting...");

						JsonNode geoJsonFeatureGeometryJsonNode = geoJsonFeatureNode.at("/geometry");

						// remove 'eClass'
						if (!geoJsonFeatureGeometryJsonNode.at("/eClass").isMissingNode()) {
							((ObjectNode) geoJsonFeatureGeometryJsonNode).remove("eClass");
						}

						org.locationtech.spatial4j.shape.Shape shape = GEOJSON_READER
								.read(geoJsonFeatureGeometryJsonNode);
						
						addFeatureOrShape(golfCourseFeatures, 
								golfCourseFeatureType, 
								geometryType, 
								shape, 
								holeNumberOptional);
						
						mapAsGeoJSONFeature(geoJSONFeatures, geometryType, shape, 
								Optional.of(constructGeoJSONFeatureProperties(golfCourseFeatureType, holeNumberOptional)));

					} else {
						logger.debug("'" + golfCourseFeatureType.getInternalName() + "' projection is not of type '"
										+ Arrays.toString(geometryTypes.toArray()) + "'! Skipping..");
					}
				}
			}

			return Pair.with(golfCourseFeatures, geoJSONFeatures);
		}

		return Pair.with(Collections.emptyList(), Collections.emptyList());
	}
	
	private Map<String, Object> constructGeoJSONFeatureProperties(GolfCourseFeatureType golfCourseFeatureType, Optional<Integer> holeNumberOptional) {
		Map<String, Object> properties = new HashMap<String, Object>();
		properties.put("golfCourseFeatureType", golfCourseFeatureType.getLabel());
		if (holeNumberOptional.isPresent()) {
			properties.put("holeNumber", holeNumberOptional.get());
		}
		
		return properties;
	}
	
	private List<org.geojson.Feature> constructConnectingShapes(Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple,
			boolean excludeBackground) {
		logger.debug("------------------------");
		logger.debug("Building golf course features' connections...");
		
		GolfCourseFeaturesAdjacencyGraph adjacencyGraph = new GolfCourseFeaturesAdjacencyGraph();
		
		List<org.geojson.Feature> geoJSONFeatures = new ArrayList<org.geojson.Feature>(List.copyOf(mappedFeaturesTuple.getValue1()));
		
		for (GolfCourseFeature golfCourseFeature1 : mappedFeaturesTuple.getValue0()) {
			
			if (excludeBackground && golfCourseFeature1.getFeatureType() == GolfCourseFeatureType.background) {
				continue;
			}
			
			adjacencyGraph.addFeature(golfCourseFeature1);
			
			for (GolfCourseFeature golfCourseFeature2 : mappedFeaturesTuple.getValue0()) {
				
				if (excludeBackground && golfCourseFeature2.getFeatureType() == GolfCourseFeatureType.background) {
					continue;
				}				
				
				if (golfCourseFeature1.hashCode() != golfCourseFeature2.hashCode()) {
					
					adjacencyGraph.addFeature(golfCourseFeature2);
					
					if (!adjacencyGraph.connectionExists(golfCourseFeature1, golfCourseFeature2)) {
						
						double distanceMeters = calculateDistanceInMeters(golfCourseFeature1, golfCourseFeature2);
						
						if (distanceMeters <= config.featureConnectionMinDistance()) {
							logger.debug("Adding connection between '" + golfCourseFeature1 + "' and '" + golfCourseFeature2 + "'");
							logger.debug("Distance: " + distanceMeters + " m");
							
							adjacencyGraph.addConnection(golfCourseFeature1, golfCourseFeature2, distanceMeters);
							
							// TODO: take into account features with more than one shape
							org.locationtech.jts.geom.Geometry golfCourseFeature1ShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory().getGeometryFrom(golfCourseFeature1.getShape());
							
							org.locationtech.jts.geom.Geometry golfCourseFeature2ShapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory().getGeometryFrom(golfCourseFeature2.getShape());
							
							org.locationtech.jts.geom.Coordinate[] nearestPoints = DistanceOp.nearestPoints(golfCourseFeature1ShapeGeometry, golfCourseFeature2ShapeGeometry);
							
							boolean nearestPointsSame = (nearestPoints.length >= 2 && (nearestPoints[0] == nearestPoints[1]));
							if (nearestPointsSame) {
								geoJSONFeatures.add(mapAsGeoJSONPointFeature(nearestPoints[0], Optional.empty()));
							} else {
								geoJSONFeatures.add(mapAsGeoJSONLineStringFeature(nearestPoints, Optional.empty()));
							}
							
						} else {
							logger.debug("Distance between '" + golfCourseFeature1 + "' and '" + golfCourseFeature2 + "' is too large (" + distanceMeters + ")");
						}
						
						logger.debug("------------------------\n");
					}
				}
			}
		}
		
		return geoJSONFeatures;
	}
	
	private void visualizeAdjacencyViaGeoJson(Pair<List<GolfCourseFeature>, List<org.geojson.Feature>> mappedFeaturesTuple, Writer writer) {
		List<org.geojson.Feature> geoJSONFeatures = constructConnectingShapes(mappedFeaturesTuple, true);
		
		try {
			visualizeAdjacencyViaGeoJson(geoJSONFeatures, writer);
		} catch (IOException e) {
			logger.error(e.getMessage());
		}
	}
	
	private void visualizeAdjacencyViaGeoJson(List<org.geojson.Feature> geoJSONFeatures, Writer writer) throws IOException {
		FeatureCollection geoJsonFeatureCollection = new FeatureCollection();
		geoJsonFeatureCollection.addAll(geoJSONFeatures);
		
		OBJECT_MAPPER.writerWithDefaultPrettyPrinter().writeValue(writer, geoJsonFeatureCollection);
	}
	
	private void mapAsGeoJSONFeature(List<org.geojson.Feature> geoJSONFeatures, GeometryType geometryType,
			org.locationtech.spatial4j.shape.Shape shape, Optional<Map<String, Object>> propertiesOptional) {

		org.locationtech.jts.geom.Geometry shapeGeometry = JTS_SPATIAL_CONTEXT.getShapeFactory().getGeometryFrom(shape);

		org.geojson.Feature geoJSONFeature;
		switch (geometryType) {
		case Point:
			geoJSONFeature = mapAsGeoJSONPointFeature(shapeGeometry.getCoordinate(), propertiesOptional);
			break;
		case LineString:
			geoJSONFeature = mapAsGeoJSONLineStringFeature(shapeGeometry.getCoordinates(), propertiesOptional);
			break;
		case Polygon:
			geoJSONFeature = mapAsGeoJSONPolygonFeature(shapeGeometry.getCoordinates(), propertiesOptional);
			break;
		default:
			throw new IllegalArgumentException(geometryType.name() + " is not supported!");
		}

		geoJSONFeatures.add(geoJSONFeature);
	}

	private org.geojson.Feature mapAsGeoJSONPolygonFeature(org.locationtech.jts.geom.Coordinate[] coordinates,
			Optional<Map<String, Object>> propertiesOptional) {
		Feature polygonFeature = new Feature();

		org.geojson.Polygon polygon = new org.geojson.Polygon(mapCoordinateToLngLatAlt(coordinates));
		polygonFeature.setGeometry(polygon);

		if (propertiesOptional.isPresent()) {
			polygonFeature.setProperties(propertiesOptional.get());
		}

		return polygonFeature;
	}

	private org.geojson.Feature mapAsGeoJSONLineStringFeature(org.locationtech.jts.geom.Coordinate[] coordinates,
			Optional<Map<String, Object>> propertiesOptional) {
		Feature lineStringFeature = new Feature();

		org.geojson.LineString lineString = new org.geojson.LineString(
				new org.geojson.LngLatAlt(coordinates[0].x, coordinates[0].y),
				new org.geojson.LngLatAlt(coordinates[1].x, coordinates[1].y));
		lineStringFeature.setGeometry(lineString);

		if (propertiesOptional.isPresent()) {
			lineStringFeature.setProperties(propertiesOptional.get());
		}

		return lineStringFeature;
	}

	private org.geojson.Feature mapAsGeoJSONPointFeature(org.locationtech.jts.geom.Coordinate coordinate,
			Optional<Map<String, Object>> propertiesOptional) {
		Feature pointFeature = new Feature();

		org.geojson.Point point = new org.geojson.Point(coordinate.x, coordinate.y);
		pointFeature.setGeometry(point);

		if (propertiesOptional.isPresent()) {
			pointFeature.setProperties(propertiesOptional.get());
		}

		return pointFeature;
	}
	
	private List<org.geojson.LngLatAlt> mapCoordinateToLngLatAlt(org.locationtech.jts.geom.Coordinate[] coordinates) {
		// @formatter:off
		return Arrays.stream(coordinates)
				.map(coordinate -> new org.geojson.LngLatAlt(coordinate.x, coordinate.y))
				.collect(Collectors.toList());
		// @formatter:on
	}
	
	private List<com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.GolfCourseFeature> mapToEMFGolfCourseFeatures(List<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature> adjacentFeaturesList) {
		// @formatter:off
		return adjacentFeaturesList
				.stream()
				.map(gcFeature -> mapToEMFGolfCourseFeature(gcFeature))
				.collect(Collectors.toList());
		// @formatter:on
	}
	
	private com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.GolfCourseFeature mapToEMFGolfCourseFeature(com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature gcFeature) {
		com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.GolfCourseFeature emfGcFeature = GCFeaturesAdjacencyFactory.eINSTANCE.createGolfCourseFeature();
		emfGcFeature.setFeatureType(mapToEMFGolfCourseFeatureType(gcFeature.getFeatureType()));
		if (gcFeature instanceof com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseHoleFeature) {
			emfGcFeature.setHoleNumber(((com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseHoleFeature)gcFeature).getHoleNumber());
		}		
		return emfGcFeature;
	}

	private com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.GolfCourseFeatureType mapToEMFGolfCourseFeatureType(com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeatureType gcFeatureType) {
		return com.playertour.backend.golfcourse.featuresadjacency.model.featuresadjacency.GolfCourseFeatureType.valueOf(gcFeatureType.name().toUpperCase());
	}
}
