/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse.featuresadjacency.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.time.Duration;
import java.time.Instant;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;

import org.javatuples.Pair;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.playertour.backend.apis.course.CourseGPSService;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeatureType;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeaturesAdjacencyService;
import com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseHoleFeature;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class GolfCourseFeaturesAdjacencyTest {
	
	@InjectBundleContext
	BundleContext bundleContext;
	
	/******************************************************************/
	/** Test data for "Golf Hukvaldy" (ID: 626fe9135fbc9fad135bf53c) **/
	private static final String TESTDATA_COURSE_ID = "626fe9135fbc9fad135bf53c";
	
	private static final String TESTDATA_COURSE_GPSVECTOR_FILENAME = "/data/CourseGPSVector.626fe9135fbc9fad135bf53c.json";
	
	private static final org.geojson.LngLatAlt TESTDATA_COURSE_LOCATION_CLUBHOUSE = new org.geojson.LngLatAlt(18.210453093052, 49.61804031805);
	private static final org.geojson.LngLatAlt TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY = new org.geojson.LngLatAlt(18.214723383566586, 49.61758974569394);
	private static final org.geojson.LngLatAlt TESTDATA_COURSE_LOCATION_HOLE_1_PERIMETER = new org.geojson.LngLatAlt(18.211200845278807, 49.618832634081116);
	private static final org.geojson.LngLatAlt TESTDATA_COURSE_LOCATION_HOLE_2_FAIRWAY_SHAPE_2 = new org.geojson.LngLatAlt(18.21591205894947, 49.616193706666735);
	private static final org.geojson.LngLatAlt TESTDATA_COURSE_LOCATION_WITHIN_DISTANCE = new org.geojson.LngLatAlt(18.226408321671585, 49.61725461615604);
	private static final org.geojson.LngLatAlt TESTDATA_COURSE_LOCATION_TOO_FAR = new org.geojson.LngLatAlt(18.223869200288075, 49.69877299772209);
	
	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseGPSService> courseGPSServiceAware) {

		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		ServiceReference<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceReference = courseFeaturesAdjacencyServiceAware
				.getServiceReference();
		assertThat(courseFeaturesAdjacencyServiceReference).isNotNull();

		assertThat(courseGPSServiceAware.getServices()).hasSize(1);
		ServiceReference<CourseGPSService> courseGPSServiceAwareReference = courseGPSServiceAware
				.getServiceReference();
		assertThat(courseGPSServiceAwareReference).isNotNull();
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testFindClosestGolfCourseFeatureCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		Optional<GolfCourseFeature> gcFeatureOptional = courseFeaturesAdjacencyService.findClosestGolfCourseFeature(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
		
		assertThat(gcFeatureOptional.isPresent()).isTrue();
		assertThat(gcFeatureOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
		assertThat(gcFeatureOptional.get() instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)gcFeatureOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(1));
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testFindClosestGolfCourseFeatureCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
						
			Optional<GolfCourseFeature> gcFeatureOptional = courseFeaturesAdjacencyService.findClosestGolfCourseFeature(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
			
			assertThat(gcFeatureOptional.isPresent()).isTrue();
			assertThat(gcFeatureOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
			assertThat(gcFeatureOptional.get() instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)gcFeatureOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(1));
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testListAdjacentGolfCourseFeaturesCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
		
		List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
		
		assertThat(adjacentFeaturesList).isNotNull();
		assertThat(adjacentFeaturesList).isNotEmpty();
		assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
		
		GolfCourseFeature closestGcFeature = adjacentFeaturesList.get(0);
		assertThat(closestGcFeature.getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
		assertThat(closestGcFeature instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)closestGcFeature).getHoleNumber()).isEqualTo(Integer.valueOf(1));
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testListAdjacentGolfCourseFeaturesCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
			
			List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
			
			assertThat(adjacentFeaturesList).isNotNull();
			assertThat(adjacentFeaturesList).isNotEmpty();
			assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
			
			GolfCourseFeature closestGcFeature = adjacentFeaturesList.get(0);
			assertThat(closestGcFeature.getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
			assertThat(closestGcFeature instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)closestGcFeature).getHoleNumber()).isEqualTo(Integer.valueOf(1));
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testListAdjacentGolfCourseFeaturesMultiShapeCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_HOLE_2_FAIRWAY_SHAPE_2.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_2_FAIRWAY_SHAPE_2.getLatitude(), 
				Optional.of(2));
		
		List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
		
		assertThat(adjacentFeaturesList).isNotNull();
		assertThat(adjacentFeaturesList).isNotEmpty();
		assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
		
		GolfCourseFeature closestGcFeature = adjacentFeaturesList.get(0);
		assertThat(closestGcFeature.getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
		assertThat(closestGcFeature instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)closestGcFeature).getHoleNumber()).isEqualTo(Integer.valueOf(2));
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testListAdjacentGolfCourseFeaturesMultiShapeCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_HOLE_2_FAIRWAY_SHAPE_2.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_2_FAIRWAY_SHAPE_2.getLatitude(), 
					Optional.of(2));
			
			List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
			
			assertThat(adjacentFeaturesList).isNotNull();
			assertThat(adjacentFeaturesList).isNotEmpty();
			assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
			
			GolfCourseFeature closestGcFeature = adjacentFeaturesList.get(0);
			assertThat(closestGcFeature.getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
			assertThat(closestGcFeature instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)closestGcFeature).getHoleNumber()).isEqualTo(Integer.valueOf(2));
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}	
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testIsWithinGolfCoursePerimeterCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		boolean isWithinGolfCoursePerimeter = courseFeaturesAdjacencyService.isWithinGolfCoursePerimeter(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
		
		assertThat(isWithinGolfCoursePerimeter).isTrue();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testIsWithinGolfCoursePerimeterCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			boolean isWithinGolfCoursePerimeter = courseFeaturesAdjacencyService.isWithinGolfCoursePerimeter(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
			
			assertThat(isWithinGolfCoursePerimeter).isTrue();
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}	
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testIsWithinDistanceFromGolfCoursePerimeterCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		boolean isWithinGolfCoursePerimeter = courseFeaturesAdjacencyService.isWithinGolfCoursePerimeter(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_WITHIN_DISTANCE.getLongitude(), 
				TESTDATA_COURSE_LOCATION_WITHIN_DISTANCE.getLatitude());
		
		assertThat(isWithinGolfCoursePerimeter).isTrue();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testIsWithinDistanceFromGolfCoursePerimeterCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			boolean isWithinGolfCoursePerimeter = courseFeaturesAdjacencyService.isWithinGolfCoursePerimeter(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_WITHIN_DISTANCE.getLongitude(), 
					TESTDATA_COURSE_LOCATION_WITHIN_DISTANCE.getLatitude());
			
			assertThat(isWithinGolfCoursePerimeter).isTrue();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testisTooFarFromGolfCoursePerimeterCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		boolean isWithinGolfCoursePerimeter = courseFeaturesAdjacencyService.isWithinGolfCoursePerimeter(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_TOO_FAR.getLongitude(), 
				TESTDATA_COURSE_LOCATION_TOO_FAR.getLatitude());
		
		assertThat(isWithinGolfCoursePerimeter).isFalse();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testisTooFarFromGolfCoursePerimeterCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			boolean isWithinGolfCoursePerimeter = courseFeaturesAdjacencyService.isWithinGolfCoursePerimeter(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_TOO_FAR.getLongitude(), 
					TESTDATA_COURSE_LOCATION_TOO_FAR.getLatitude());
			
			assertThat(isWithinGolfCoursePerimeter).isFalse();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testFindNearestHoleCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		Pair<Optional<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature>, Boolean> nearestHoleResult = courseFeaturesAdjacencyService
				.findNearestHole(TESTDATA_COURSE_ID, 
						TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
						TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
		
		Optional<GolfCourseFeature> nearestHoleOptional = nearestHoleResult.getValue0();
		boolean isWithinGolfCoursePerimeter = nearestHoleResult.getValue1().booleanValue();
		
		assertThat(nearestHoleOptional.isPresent()).isTrue();
		assertThat(nearestHoleOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.perimeter);
		assertThat(nearestHoleOptional.get() instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)nearestHoleOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(1));
		
		assertThat(isWithinGolfCoursePerimeter).isTrue();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testFindNearestHoleCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			Pair<Optional<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature>, Boolean> nearestHoleResult = courseFeaturesAdjacencyService
					.findNearestHole(golfCourseGPSVectorInputStream, 
							TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
							TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
			
			Optional<GolfCourseFeature> nearestHoleOptional = nearestHoleResult.getValue0();
			boolean isWithinGolfCoursePerimeter = nearestHoleResult.getValue1().booleanValue();
			
			assertThat(nearestHoleOptional.isPresent()).isTrue();
			assertThat(nearestHoleOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.perimeter);
			assertThat(nearestHoleOptional.get() instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)nearestHoleOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(1));
			
			assertThat(isWithinGolfCoursePerimeter).isTrue();
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testFindNearestHoleOutsidePerimeterCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		Pair<Optional<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature>, Boolean> nearestHoleResult = courseFeaturesAdjacencyService
				.findNearestHole(TESTDATA_COURSE_ID, 
						TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLongitude(), 
						TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLatitude());
		
		Optional<GolfCourseFeature> nearestHoleOptional = nearestHoleResult.getValue0();
		boolean isWithinGolfCoursePerimeter = nearestHoleResult.getValue1().booleanValue();
		
		assertThat(nearestHoleOptional.isPresent()).isTrue();
		assertThat(nearestHoleOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.perimeter);
		assertThat(nearestHoleOptional.get() instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)nearestHoleOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(9));
		
		assertThat(isWithinGolfCoursePerimeter).isTrue();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}	
	
	@Test
	public void testFindNearestHoleOutsidePerimeterFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			Pair<Optional<com.playertour.backend.golfcourse.featuresadjacency.service.api.GolfCourseFeature>, Boolean> nearestHoleResult = courseFeaturesAdjacencyService
					.findNearestHole(golfCourseGPSVectorInputStream, 
							TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLongitude(), 
							TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLatitude());
			
			Optional<GolfCourseFeature> nearestHoleOptional = nearestHoleResult.getValue0();
			boolean isWithinGolfCoursePerimeter = nearestHoleResult.getValue1().booleanValue();
			
			assertThat(nearestHoleOptional.isPresent()).isTrue();
			assertThat(nearestHoleOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.perimeter);
			assertThat(nearestHoleOptional.get() instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)nearestHoleOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(9));
			
			assertThat(isWithinGolfCoursePerimeter).isTrue();
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testIsWithinHolePerimeterCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		boolean isWithinHolePerimeter = courseFeaturesAdjacencyService.isWithinHolePerimeter(TESTDATA_COURSE_ID, 
				Integer.valueOf(1),
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
		
		assertThat(isWithinHolePerimeter).isTrue();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove			
	}
	
	@Test
	public void testIsWithinHolePerimeterCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			boolean isWithinHolePerimeter = courseFeaturesAdjacencyService.isWithinHolePerimeter(golfCourseGPSVectorInputStream, 
					Integer.valueOf(1),
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
			
			assertThat(isWithinHolePerimeter).isTrue();
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testIsOnHolePerimeterBorderCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		boolean isWithinHolePerimeter = courseFeaturesAdjacencyService.isWithinHolePerimeter(TESTDATA_COURSE_ID, 
				Integer.valueOf(1),
				TESTDATA_COURSE_LOCATION_HOLE_1_PERIMETER.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_PERIMETER.getLatitude());
		
		assertThat(isWithinHolePerimeter).isTrue();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove			
	}
	
	@Test
	public void testIsOnHolePerimeterBorderCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			boolean isWithinHolePerimeter = courseFeaturesAdjacencyService.isWithinHolePerimeter(golfCourseGPSVectorInputStream, 
					Integer.valueOf(1),
					TESTDATA_COURSE_LOCATION_HOLE_1_PERIMETER.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_PERIMETER.getLatitude());
			
			assertThat(isWithinHolePerimeter).isTrue();
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove
	}	
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testIsOutsideHolePerimeterCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		boolean isWithinHolePerimeter = courseFeaturesAdjacencyService.isWithinHolePerimeter(TESTDATA_COURSE_ID, 
				Integer.valueOf(1),
				TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLongitude(), 
				TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLatitude());
		
		assertThat(isWithinHolePerimeter).isFalse();
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove	
	}
	
	@Test
	public void testIsOutsideHolePerimeterCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			boolean isWithinHolePerimeter = courseFeaturesAdjacencyService.isWithinHolePerimeter(golfCourseGPSVectorInputStream, 
					Integer.valueOf(1),
					TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLongitude(), 
					TESTDATA_COURSE_LOCATION_CLUBHOUSE.getLatitude());
			
			assertThat(isWithinHolePerimeter).isFalse();
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testFindClosestHoleFeatureCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		Integer holeNumber = 1;
		
		Optional<GolfCourseFeature> gcFeatureOptional = courseFeaturesAdjacencyService.findClosestHoleFeature(TESTDATA_COURSE_ID, 
				holeNumber,
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
		
		assertThat(gcFeatureOptional.isPresent()).isTrue();
		assertThat(gcFeatureOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
		assertThat(gcFeatureOptional.get() instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)gcFeatureOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(holeNumber));
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testFindClosestHoleFeatureCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		Integer holeNumber = 1;
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
						
			Optional<GolfCourseFeature> gcFeatureOptional = courseFeaturesAdjacencyService.findClosestHoleFeature(golfCourseGPSVectorInputStream, 
					holeNumber,
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude());
			
			assertThat(gcFeatureOptional.isPresent()).isTrue();
			assertThat(gcFeatureOptional.get().getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
			assertThat(gcFeatureOptional.get() instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)gcFeatureOptional.get()).getHoleNumber()).isEqualTo(Integer.valueOf(holeNumber));
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testListAdjacentHoleFeaturesCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		Integer holeNumber = 1;
		
		Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(TESTDATA_COURSE_ID, 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
				TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude(), 
				Optional.of(holeNumber));
		
		List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
		
		assertThat(adjacentFeaturesList).isNotNull();
		assertThat(adjacentFeaturesList).isNotEmpty();
		assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
		
		GolfCourseFeature closestGcFeature = adjacentFeaturesList.get(0);
		assertThat(closestGcFeature.getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
		assertThat(closestGcFeature instanceof GolfCourseHoleFeature).isTrue();
		assertThat(((GolfCourseHoleFeature)closestGcFeature).getHoleNumber()).isEqualTo(Integer.valueOf(holeNumber));
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testListAdjacentHoleFeaturesCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		Integer holeNumber = 1;
		
		try (InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME)) {
			
			Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(golfCourseGPSVectorInputStream, 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLongitude(), 
					TESTDATA_COURSE_LOCATION_HOLE_1_FAIRWAY.getLatitude(), 
					Optional.of(holeNumber));
			
			List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
			
			assertThat(adjacentFeaturesList).isNotNull();
			assertThat(adjacentFeaturesList).isNotEmpty();
			assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
			
			GolfCourseFeature closestGcFeature = adjacentFeaturesList.get(0);
			assertThat(closestGcFeature.getFeatureType()).isEqualTo(GolfCourseFeatureType.fairway);
			assertThat(closestGcFeature instanceof GolfCourseHoleFeature).isTrue();
			assertThat(((GolfCourseHoleFeature)closestGcFeature).getHoleNumber()).isEqualTo(Integer.valueOf(holeNumber));
			
		} catch (IOException e) {
			e.printStackTrace();
		}		
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}	
		
	@Disabled("Integration tests are run on empty database; use version which reads from file instead; uncomment as needed only when testing locally")
	@Test
	public void testVisualizeAdjacencyViaGeoJsonCourseId(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();		
		
		try (Writer writer = new StringWriter()) {
			
			courseFeaturesAdjacencyService.visualizeAdjacencyViaGeoJson(TESTDATA_COURSE_ID, writer);
			
			String visualizeAdjacencyGeoJson = writer.toString();
			
			assertThat(visualizeAdjacencyGeoJson).isNotBlank();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Test
	public void testVisualizeAdjacencyViaGeoJsonCourseVectorFile(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		try (
				InputStream golfCourseGPSVectorInputStream = getGolfCourseGPSVectorInputStream(TESTDATA_COURSE_GPSVECTOR_FILENAME); 
				Writer writer = new StringWriter();
			) {
			
			courseFeaturesAdjacencyService.visualizeAdjacencyViaGeoJson(golfCourseGPSVectorInputStream, writer);
			
			String visualizeAdjacencyGeoJson = writer.toString();
			
			assertThat(visualizeAdjacencyGeoJson).isNotBlank();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}
	
	@Disabled
	@Test
	public void testVisualizeAdjacencyViaGeoJsonCourseIdSelfIntersectionIssue(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		String courseId = "626fe91d5fbc9fad135bf567"; // "New - Prosper Golf Club Celadna"
		
		try (Writer writer = new StringWriter()) {
			
			// @formatter:off
			courseFeaturesAdjacencyService.visualizeAdjacencyViaGeoJson(
					courseId, 
					EnumSet.noneOf(GolfCourseFeatureType.class),
					EnumSet.of(
							GolfCourseFeatureType.perimeter, 
							GolfCourseFeatureType.green, 
							GolfCourseFeatureType.greenCenter), 
					writer);
			// @formatter:on
			
			String visualizeAdjacencyGeoJson = writer.toString();
			
			assertThat(visualizeAdjacencyGeoJson).isNotBlank();
			
			System.out.println(visualizeAdjacencyGeoJson);
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove		
	}	
	
	@Disabled
	@Test
	public void testListAdjacentGolfCourseFeaturesCourseIdSelfIntersectionIssue(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<GolfCourseFeaturesAdjacencyService> courseFeaturesAdjacencyServiceAware) {
		
		Instant start = Instant.now();
		
		assertThat(courseFeaturesAdjacencyServiceAware.getServices()).hasSize(1);
		GolfCourseFeaturesAdjacencyService courseFeaturesAdjacencyService = courseFeaturesAdjacencyServiceAware
				.getService();
		assertThat(courseFeaturesAdjacencyService).isNotNull();
		
		String courseId = "626fe91d5fbc9fad135bf567"; // "New - Prosper Golf Club Celadna"
		double lng = 18.322640095383846;
		double lat = 49.533902570320734;
		
		Pair<List<GolfCourseFeature>, Boolean> adjacentFeaturesResult = courseFeaturesAdjacencyService.listAdjacentGolfCourseFeatures(courseId, 
				lng, 
				lat);
		
		List<GolfCourseFeature> adjacentFeaturesList = adjacentFeaturesResult.getValue0();
		
		assertThat(adjacentFeaturesList).isNotNull();
		assertThat(adjacentFeaturesList).isNotEmpty();
		assertThat(adjacentFeaturesList).hasSizeGreaterThanOrEqualTo(1);
		
		Instant finish = Instant.now();
		long timeElapsed = Duration.between(start, finish).toMillis();
		System.out.println("Processed in: " + timeElapsed + " milliseconds"); // TODO: log @ debug level or remove
	}	
	
	private InputStream getGolfCourseGPSVectorInputStream(String courseGPSVectorFileName) throws IOException {
		return FrameworkUtil.getBundle(getClass()).getEntry(courseGPSVectorFileName).openStream();
	}
}
