/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.UriMapProvider;
import org.gecko.emf.pushstream.EPushStreamProvider;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.LuceneIndexService;
import org.gecko.util.pushstreams.GeckoPushbackPolicyOption;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.QueuePolicyOption;

import com.playertour.backend.apis.common.CommonIndexService;
import com.playertour.backend.apis.course.CourseIndexService;
import com.playertour.backend.golfcourse.helper.CourseIndexHelper;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;

/**
 * 
 * @author ilenia
 * @since Feb 16, 2021
 */
@Component(immediate = true, service = {CourseIndexService.class, CommonIndexService.class}, scope = ServiceScope.SINGLETON, reference = {
		@Reference(name = "mongoCondition", service = UriMapProvider.class, target = "(uri.map.src=mongodb://playertour/)", cardinality = ReferenceCardinality.MANDATORY),
		@Reference(name = "modelCondition", service = ResourceSetFactory.class, target = "(emf.model.name=golfcourse)", cardinality = ReferenceCardinality.MANDATORY)
})
public class CourseIndexServiceImpl implements CourseIndexService, CommonIndexService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(target = "(id=course)")
	private LuceneIndexService courseIndex;
	
	@Reference(service=LoggerFactory.class)
	private Logger logger;
	
	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));

	
	@Activate
	public void activate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(100, TimeUnit.MILLISECONDS);
			initializeIndex();
			return true;
		}).onSuccess(t -> System.out.println("Finished Index Courses!"))
		.onFailure(t -> t.printStackTrace());	
	}
	
	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.course.CourseIndexService#indexGolfCourse(com.playertour.backend.golfcourse.model.golfcourse.GolfCourse, com.playertour.backend.apis.course.IndexActionType)
	 */
	@Override
	public void indexGolfCourse(GolfCourse golfCourse, IndexActionType actionType) {
		DocumentIndexContextObject context = CourseIndexHelper.mapCourse(golfCourse, actionType, null);			
		courseIndex.handleContextSync(context);
	}



	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.common.CommonIndexService#initializeIndex()
	 */
	@Override
	public Promise<Void> initializeIndex() {
		EMFRepository repository = repositoryServiceObjects.getService();
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		IQuery query = queryRepo.createQueryBuilder().allQuery().build();
		logger.info("Starting index Build for golf course");

		resetIndex();

		EPushStreamProvider psp = queryRepo.getEObjectByQuery(GolfCoursePackage.Literals.GOLF_COURSE, query, CommonIndexService.getLoadOptions());
		if(psp == null) {
			return null;
		}
		
		PushStream<EObject> indexNew = psp.createPushStreamBuilder()
				.withPushbackPolicy(GeckoPushbackPolicyOption.LINEAR_AFTER_THRESHOLD.getPolicy(650))
				.withQueuePolicy(QueuePolicyOption.BLOCK)
				.withExecutor(Executors.newSingleThreadExecutor())
				.withBuffer(new ArrayBlockingQueue<PushEvent<? extends EObject>>(1200))
				.build();

		return indexNew
				.map(eo -> (GolfCourse) eo)
				.map(CourseIndexHelper::mapCourseNew)
				.forEach(d -> {
					repository.detach(d.getObject());
					courseIndex.handleContext(d);
				}).onFailure(t -> repositoryServiceObjects.ungetService(repository))
				.thenAccept(result -> logger.info("finished index build for golf course"))
				.thenAccept(result -> repositoryServiceObjects.ungetService(repository));
	}



	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.common.CommonIndexService#resetIndex()
	 */
	@Override
	public void resetIndex() {
		try {
			courseIndex.getIndexWriter().deleteAll();
			courseIndex.commit();
		} catch (IOException e) {
			logger.error("Could not delete golf course index", e);
		}	
	}

}
