/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.api.IndexActionType;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.apis.course.CourseIndexService;
import com.playertour.backend.apis.course.CourseSearchService;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.apis.mmt.common.PlayertourModelTransformator;
import com.playertour.backend.apis.mmt.common.UnknownTransformationException;
import com.playertour.backend.golfcourse.model.golfcourse.Geometry;
import com.playertour.backend.golfcourse.model.golfcourse.GeometryType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.golfcourse.model.golfcourse.Location;
import com.playertour.backend.golfcourse.model.golfcourse.Shape;
import com.playertour.backend.golfcourse.model.golfcourse.ShapeObject;
import com.playertour.backend.golfcourse.model.golfcourse.ShapeWrapper;
import com.playertour.backend.igolf.model.igolf.CourseCompleteResponse;
import com.playertour.backend.igolf.model.igolf.IGolfPackage;

@Component(scope = ServiceScope.PROTOTYPE)
public class CourseServiceImpl implements CourseService {

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	EMFRepository repository;

	@Reference(target = ("(component.name=BEModelTransformator)"))
	PlayertourModelTransformator transformator;

	@Reference
	CourseIndexService courseIndexService;

	@Reference
	CourseSearchService courseSearchService;
	
	private static final List<String> POLYGONAL_SHAPES_NAMES = List.of(
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__BACKGRUND.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__CLUBHOUSE.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__LAVA.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__BRIDGE.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__VEGETATION.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__SAND.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__OCEAN.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__POND.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__LAKE.getName(),
			GolfCoursePackage.Literals.COURSE_GPS_VECTOR__WATER.getName(),
			GolfCoursePackage.Literals.HOLE__PERIMETER.getName(),
			GolfCoursePackage.Literals.HOLE__FAIRWAY.getName(),
			GolfCoursePackage.Literals.HOLE__TEEBOX.getName(),
			GolfCoursePackage.Literals.HOLE__GREEN.getName(),
			GolfCoursePackage.Literals.HOLE__BUNKER.getName());

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.course.CourseService#saveCourse(com.playertour.backend.igolf.model.igolf.CourseCompleteResponse)
	 */
	@Override
	public GolfCourse saveCourse(CourseCompleteResponse iGolfCourseResponse) throws UnknownTransformationException {

		if(iGolfCourseResponse == null) {
			throw new IllegalArgumentException("Cannot save null CourseCompleteResponse object");
		}	
		GolfCourse golfCourse = transformator.transform(iGolfCourseResponse, IGolfPackage.Literals.COURSE_COMPLETE_RESPONSE, 
				GolfCoursePackage.Literals.GOLF_COURSE);

		boolean isFirstSave = isFirstSave(golfCourse);
		if(isFirstSave) {
			repository.attach(golfCourse);
		}

		if(golfCourse.getCourseGPSVector() != null) {
			convertShapeObjects(golfCourse.getCourseGPSVector());
			golfCourse.getCourseGPSVector().getHoles().stream().forEach(h->convertShapeObjects(h));
		}
		repository.attach(golfCourse);
		repository.save(golfCourse);
		courseIndexService.indexGolfCourse(golfCourse, isFirstSave ? IndexActionType.ADD : IndexActionType.MODIFY);

		return golfCourse;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.course.CourseService#saveCourses(java.util.List)
	 */
	@Override
	public List<GolfCourse> saveCourses(List<CourseCompleteResponse> iGolfCourseResponses) throws UnknownTransformationException {

		if(iGolfCourseResponses == null) {
			throw new IllegalArgumentException("Cannot save null CourseCompleteResponse object");
		}
		List<GolfCourse> golfCourses = new ArrayList<GolfCourse>(iGolfCourseResponses.size());
		golfCourses = transformator.transform(iGolfCourseResponses, GolfCoursePackage.Literals.GOLF_COURSE);

		Map<GolfCourse, Boolean> isFirstSaveMap = new HashMap<>();
		for(GolfCourse golfCourse : golfCourses) {
			if(golfCourse.getCourseGPSVector() != null) {
				convertShapeObjects(golfCourse.getCourseGPSVector());
				golfCourse.getCourseGPSVector().getHoles().stream().forEach(h->convertShapeObjects(h));
			}			
			boolean isFirstSave = isFirstSave(golfCourse);
			if(isFirstSave) {
				repository.attach(golfCourse);
			}
			isFirstSaveMap.put(golfCourse, isFirstSave);
		}
		save(golfCourses);
		isFirstSaveMap.entrySet().stream().forEach(e-> {
			courseIndexService.indexGolfCourse(e.getKey(), e.getValue() ? IndexActionType.ADD : IndexActionType.MODIFY);
		});

		return golfCourses;
	}


	private boolean isFirstSave(GolfCourse golfCourse) {
		boolean isFirstSave = true;
		GolfCourse oldCourse = courseSearchService.getCourseByCourseId(golfCourse.getCourseId());
		if(oldCourse != null) {
			isFirstSave = false;
			golfCourse.setId(oldCourse.getId());
		}
		return isFirstSave;
	}
	
	private void convertShapeObjects(EObject eObject) {
		
		if(eObject == null) {
			return;
		}
		
		List<EReference> shapeObjRef = eObject.eClass().getEAllContainments().stream()
				.filter(c->GolfCoursePackage.Literals.SHAPE_OBJECT.equals(c.getEReferenceType()))
				.collect(Collectors.toList());

		for(EReference reference : shapeObjRef) {
			String refName = reference.getName();
			boolean isPolygonForSure = POLYGONAL_SHAPES_NAMES.contains(refName);
			ShapeObject shapeObj = (ShapeObject) eObject.eGet(reference);
			if(shapeObj == null) {
				continue;
			}
			for(Shape shape : shapeObj.getShapes()) {
				ShapeWrapper wrapper = shape.getWrapper();
				if(wrapper != null) {
					Geometry geometry = GolfCourseFactory.eINSTANCE.createGeometry();
					setGeometryType(geometry, wrapper, isPolygonForSure);
					Double[][] coord = new Double[wrapper.getPoints().size()][];
					int counter = 0;
					for(Location loc : wrapper.getPoints()) {
						coord[counter] = new Double[]{loc.getLatitude(), loc.getLongitude()};
						geometry.getCoordinates().add(coord);
						counter++;
					}
					shape.setGeometry(geometry);
				}				
			}
			eObject.eSet(reference, shapeObj);
		}		
	}

	private void setGeometryType(Geometry geometry, ShapeWrapper wrapper, boolean isPolygonForSure) {
		if(wrapper.getPoints().size() == 1) {
			geometry.setType(GeometryType.POINT);
		}
		else if(wrapper.getPoints().get(0).getLatitude() == wrapper.getPoints().get(wrapper.getPoints().size()-1).getLatitude()
				&& wrapper.getPoints().get(0).getLongitude() == wrapper.getPoints().get(wrapper.getPoints().size()-1).getLongitude()) {
			geometry.setType(GeometryType.POLYGON);
		}
		else if(isPolygonForSure) {
			geometry.setType(GeometryType.POLYGON);
			Location start = wrapper.getPoints().get(0);
			wrapper.getPoints().add(EcoreUtil.copy(start));
		}
		else {
			geometry.setType(GeometryType.LINE_STRING);
		}
	}

	@SuppressWarnings("unchecked")
	private void save(Collection<? extends EObject> objects) {
		repository.save((Collection<EObject>)  objects);
	}
}
