/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse.helper;

import java.util.Collections;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.LatLonDocValuesField;
import org.apache.lucene.document.LatLonPoint;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.DocumentIndexContextObjectImpl;
import org.osgi.util.promise.Deferred;

import com.playertour.backend.apis.common.DeferredCommitCallback;
import com.playertour.backend.apis.common.DocumentUtil;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;

/**
 * 
 * @author ilenia
 * @since Feb 16, 2021
 */
public class CourseIndexHelper {
	
	public static DocumentIndexContextObject mapCourseNew(GolfCourse golfCourse) {
		return mapCourse(golfCourse, IndexActionType.ADD, null);
	}

	public static DocumentIndexContextObject mapCourseNew(GolfCourse golfCourse, Deferred<Boolean> deferred) {
		return mapCourse(golfCourse, IndexActionType.ADD, deferred);
	}

	public static DocumentIndexContextObject mapCourseUpdate(GolfCourse golfCourse, Deferred<Boolean> deferred) {
		return mapCourse(golfCourse, IndexActionType.MODIFY, deferred);
	}

	public static DocumentIndexContextObject mapCourse(GolfCourse golfCourse, IndexActionType indexAction, Deferred<Boolean> deferred) {
		Document doc = new Document();
		
		DocumentUtil.toDocument(doc, golfCourse);

		if(golfCourse.getCourseDetails() != null) {
			if(golfCourse.getCourseDetails().getCourseName() != null) {
				doc.add(new StringField("coursename_lower", golfCourse.getCourseDetails().getCourseName().toLowerCase(), Store.NO));
			}
			if(golfCourse.getCourseDetails().getLocation() != null) {
				LatLonPoint latLonPoint = new LatLonPoint("course_position", golfCourse.getCourseDetails().getLocation().getLatitude(), 
						golfCourse.getCourseDetails().getLocation().getLongitude());
				doc.add(latLonPoint);
				
				LatLonDocValuesField latLonDocValue = new LatLonDocValuesField("course_position", golfCourse.getCourseDetails().getLocation().getLatitude(), 
						golfCourse.getCourseDetails().getLocation().getLongitude());
				doc.add(latLonDocValue);
			}
		}
		
		DocumentIndexContextObjectImpl.Builder builder = DocumentIndexContextObjectImpl.builder()
				.withDocuments(Collections.singletonList(doc))
				.withSourceObject(golfCourse).withIndexActionType(indexAction);

		if(IndexActionType.MODIFY.equals(indexAction) || IndexActionType.REMOVE.equals(indexAction)) {
			builder.withIdentifyingTerm(new Term("id", golfCourse.getId()));
		}
		if(deferred != null) {
			builder = builder.withCommitCallback(new DeferredCommitCallback(deferred));
		}

		return builder.build();
	}

}
