/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.golfcourse;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.LatLonPoint;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.WildcardQuery;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.search.document.LuceneIndexService;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.common.DocumentUtil;
import com.playertour.backend.apis.course.CourseSearchService;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;

/**
 * 
 * @author ilenia
 * @since Feb 16, 2021
 */
@Component(immediate = true, scope = ServiceScope.SINGLETON)
public class CourseSearchServiceImpl implements CourseSearchService {

	@Reference(target = "(id=course)")
	private LuceneIndexService courseIndex;

	@Reference
	ResourceSet resourceSet;

	@Reference(service=LoggerFactory.class)
	private Logger logger;

	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.course.CourseSearchService#getCoursesByName(java.lang.String, int)
	 */
	@Override
	public List<GolfCourse> getCoursesByName(String query, int maxResults) {
		IndexSearcher searcher = courseIndex.aquireSearch();
		try {
			try {
				Query q =  new WildcardQuery(new Term("coursename_lower", "*"+ query.toLowerCase() + "*"));
				TopDocs topDocs = searcher.search(q, maxResults);
				if(topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {
					return (GolfCourse) DocumentUtil.toEObject(d, resourceSet);
				})
						.collect(Collectors.toList());
			} catch (Exception e) {
				logger.error( "Exception while search for GolfCourse with name " + query, e);
				return Collections.emptyList();
			}
		} finally {
			courseIndex.releaseSearcher(searcher);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.course.CourseSearchService#getCourseByCourseId(java.lang.String)
	 */
	@Override
	public GolfCourse getCourseByCourseId(String courseId) {

		IndexSearcher searcher = courseIndex.aquireSearch();
		try {
			try {
				TopDocs topDocs = searcher.search(new TermQuery(new Term("courseId", courseId)), 1);
				if(topDocs.scoreDocs.length == 0) {
					return null;
				}
				IndexReader indexReader = searcher.getIndexReader();
				Document document = indexReader.document(topDocs.scoreDocs[0].doc);
				return (GolfCourse) DocumentUtil.toEObject(document, resourceSet);
			} catch (Exception e) {
				logger.error( "Exception while search for GolfCourse with courseId " + courseId, e);
				return null;
			}
		} finally {
			courseIndex.releaseSearcher(searcher);
		}
	}

	
	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.course.CourseSearchService#getCoursesWithinDist(double, double, double)
	 */
	public Map<GolfCourse, Double> getCoursesWithinDist(double latitude, double longitude, double maxDistInMeters) {
		IndexSearcher searcher = courseIndex.aquireSearch();
		try {
			try {
				float weight = 1.f;
				Query q = LatLonPoint.newDistanceFeatureQuery("course_position", weight, latitude, longitude, maxDistInMeters);
//				look if dust is already set
				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if(topDocs.scoreDocs.length == 0) {
					return Collections.emptyMap();
				}
				IndexReader indexReader = searcher.getIndexReader();
				Map<GolfCourse, Double> resultMap = new HashMap<>();
				Arrays.asList(topDocs.scoreDocs).stream().filter(sd-> sd.score >= weight/2.)
				.forEach(sd -> {
					Document d;
					try {
						d = indexReader.document(sd.doc);
					}catch (IOException e) {
						d = null;
					}
					if(d != null) {
						GolfCourse gc = (GolfCourse) DocumentUtil.toEObject(d, resourceSet);
						double dist = (maxDistInMeters/sd.score) - maxDistInMeters;
						dist /= 1000.; //convert in km
						resultMap.put(gc, dist);
					}					
				});
				return resultMap;
			} catch (Exception e) {
				logger.error( "Exception while search for GolfCourse within distance", e);
				return Collections.emptyMap();
			}
		} finally {
			courseIndex.releaseSearcher(searcher);
		}
	}

}
