/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.healthcheck;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.apache.felix.hc.api.Result;
import org.apache.felix.hc.api.Result.Status;
import org.apache.felix.hc.api.execution.HealthCheckExecutionResult;
import org.apache.felix.hc.api.execution.HealthCheckExecutor;
import org.apache.felix.hc.api.execution.HealthCheckMetadata;
import org.apache.felix.hc.api.execution.HealthCheckSelector;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.http.whiteboard.annotations.RequireHttpWhiteboard;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsName;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsResource;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsWhiteboardTarget;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

@RequireHttpWhiteboard
@JaxrsResource
@JaxrsName("Health Check Resource")
@HealthCheckApplicationTarget
@JaxrsWhiteboardTarget("(jersey.jaxrs.whiteboard.name=" + HealthCheckConstants.JAX_RS_WHITEBOARD_NAME + ")")
@Component(name = "HealthCheckResource", service = HealthCheckResource.class, enabled = true, scope = ServiceScope.PROTOTYPE)
@Path("/status")
public class HealthCheckResource {

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	// mapping of HC result status values to HTTP response codes
	static final String HTTP_STATUS_MAPPING = "OK:200,WARN:200,CRITICAL:503,TEMPORARILY_UNAVAILABLE:503,HEALTH_CHECK_ERROR:500";

	// health check names
	static final List<String> HC_NAMES = List.of("CPU", "DS Components Ready Check", "Memory", "Thread Usage",
			"OSGi Framework Ready Check");

	// health check tags
	static final List<String> HC_TAGS = List.of("systemalive");

	private Map<Result.Status, Integer> statusMapping;

	private HealthCheckSelector healthCheckSelector;

	@Reference
	HealthCheckExecutor healthCheckExecutor;

	@Activate
	public void activate() {
		this.statusMapping = getStatusMapping(HTTP_STATUS_MAPPING);

		this.healthCheckSelector = HealthCheckSelector.empty();
		healthCheckSelector.withNames(HC_NAMES.toArray(new String[0]));
		healthCheckSelector.withTags(HC_TAGS.toArray(new String[0]));
	}

	@GET
	@Produces(MediaType.TEXT_PLAIN)
	public Response getHealthStatus() {
		List<HealthCheckExecutionResult> executionResults = this.healthCheckExecutor.execute(this.healthCheckSelector);

		CombinedExecutionResult combinedExecutionResult = new CombinedExecutionResult(executionResults);
		Result overallResult = combinedExecutionResult.getHealthCheckResult();

		Integer httpStatus = statusMapping.get(overallResult.getStatus());

		return Response.status(httpStatus).build();
	}

	// based on: org.apache.felix.hc.core.impl.servlet.HealthCheckExecutorServlet.getStatusMapping(String)
	Map<Result.Status, Integer> getStatusMapping(String mappingStr) {
		Map<Result.Status, Integer> statusMapping = new TreeMap<Result.Status, Integer>();

		try {
			String[] bits = mappingStr.split("[,]");
			for (String bit : bits) {
				String[] tuple = bit.split("[:]");
				statusMapping.put(Result.Status.valueOf(tuple[0]), Integer.parseInt(tuple[1]));
			}
		} catch (Exception e) {
			throw new IllegalArgumentException("Invalid parameter httpStatus=" + mappingStr + " " + e, e);
		}

		if (!statusMapping.containsKey(Result.Status.OK)) {
			statusMapping.put(Result.Status.OK, 200);
		}
		if (!statusMapping.containsKey(Result.Status.WARN)) {
			statusMapping.put(Result.Status.WARN, statusMapping.get(Result.Status.OK));
		}
		if (!statusMapping.containsKey(Result.Status.TEMPORARILY_UNAVAILABLE)) {
			statusMapping.put(Result.Status.TEMPORARILY_UNAVAILABLE, 503);
		}
		if (!statusMapping.containsKey(Result.Status.CRITICAL)) {
			statusMapping.put(Result.Status.CRITICAL, 503);
		}
		if (!statusMapping.containsKey(Result.Status.HEALTH_CHECK_ERROR)) {
			statusMapping.put(Result.Status.HEALTH_CHECK_ERROR, 500);
		}
		return statusMapping;
	}

	// fork of: org.apache.felix.hc.core.impl.executor.CombinedExecutionResult (private package)
	public class CombinedExecutionResult implements HealthCheckExecutionResult {

		final List<HealthCheckExecutionResult> executionResults;
		final Result overallResult;

		public CombinedExecutionResult(List<HealthCheckExecutionResult> executionResults) {
			this(executionResults, Result.Status.OK);
		}

		public CombinedExecutionResult(List<HealthCheckExecutionResult> executionResults,
				Result.Status statusForZeroResult) {
			this.executionResults = Collections.unmodifiableList(executionResults);
			Result.Status mostSevereStatus = executionResults.isEmpty() ? statusForZeroResult : Result.Status.OK;
			for (HealthCheckExecutionResult executionResult : executionResults) {
				Status status = executionResult.getHealthCheckResult().getStatus();
				if (status.ordinal() > mostSevereStatus.ordinal()) {
					mostSevereStatus = status;
				}
			}
			overallResult = new Result(mostSevereStatus, "Overall status " + mostSevereStatus);
		}

		@Override
		public Result getHealthCheckResult() {
			return overallResult;
		}

		public List<HealthCheckExecutionResult> getExecutionResults() {
			return executionResults;
		}

		@Override
		public long getElapsedTimeInMs() {
			long maxElapsed = 0;
			for (HealthCheckExecutionResult result : executionResults) {
				maxElapsed = Math.max(maxElapsed, result.getElapsedTimeInMs());
			}
			return maxElapsed;
		}

		@Override
		public Date getFinishedAt() {
			Date latestFinishedAt = null;
			for (HealthCheckExecutionResult result : executionResults) {
				if (latestFinishedAt == null || latestFinishedAt.before(result.getFinishedAt())) {
					latestFinishedAt = result.getFinishedAt();
				}
			}
			return latestFinishedAt;
		}

		@Override
		public boolean hasTimedOut() {
			for (HealthCheckExecutionResult result : executionResults) {
				if (result.hasTimedOut()) {
					return true;
				}
			}
			return false;
		}

		@Override
		public HealthCheckMetadata getHealthCheckMetadata() {
			throw new UnsupportedOperationException();
		}

		@Override
		public String toString() {
			return "CombinedExecutionResult [size=" + executionResults.size() + " overall status="
					+ getHealthCheckResult().getStatus() + ", finishedAt=" + getFinishedAt() + ", elapsedTimeInMs="
					+ getElapsedTimeInMs() + ", timedOut=" + hasTimedOut() + "]";
		}
	}
}
