/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.igolf.mmt.tests;

import static org.assertj.core.api.Assertions.assertThat;

import org.gecko.qvt.osgi.api.ConfigurableModelTransformatorPool;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.playertour.backend.apis.mmt.common.PlayertourModelTransformator;
import com.playertour.backend.country.model.country.Country;
import com.playertour.backend.country.model.country.CountryPackage;
import com.playertour.backend.country.model.country.State;
import com.playertour.backend.golfcourse.model.golfcourse.CourseDetails;
import com.playertour.backend.golfcourse.model.golfcourse.CourseGPSVector;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecard;
import com.playertour.backend.golfcourse.model.golfcourse.DressCodeType;
import com.playertour.backend.golfcourse.model.golfcourse.GenderType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourse;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseType;
import com.playertour.backend.golfcourse.model.golfcourse.ShapeWrapper;
import com.playertour.backend.golfcourse.model.golfcourse.Tee;
import com.playertour.backend.golfcourse.model.golfcourse.WeekDay;
import com.playertour.backend.igolf.model.igolf.Attributes;
import com.playertour.backend.igolf.model.igolf.CountryInfo;
import com.playertour.backend.igolf.model.igolf.CourseCompleteResponse;
import com.playertour.backend.igolf.model.igolf.CourseDetailsResponse;
import com.playertour.backend.igolf.model.igolf.CourseGPSVectorResponse;
import com.playertour.backend.igolf.model.igolf.CourseInfo;
import com.playertour.backend.igolf.model.igolf.CourseTeeDetailsResponse;
import com.playertour.backend.igolf.model.igolf.IGolfFactory;
import com.playertour.backend.igolf.model.igolf.IGolfPackage;
import com.playertour.backend.igolf.model.igolf.Price;
import com.playertour.backend.igolf.model.igolf.Scorecard;
import com.playertour.backend.igolf.model.igolf.ScorecardDetailsResponse;
import com.playertour.backend.igolf.model.igolf.Shape;
import com.playertour.backend.igolf.model.igolf.ShapeObject;
import com.playertour.backend.igolf.model.igolf.Shapes;
import com.playertour.backend.igolf.model.igolf.StateInfo;
import com.playertour.backend.igolf.model.igolf.TeeDetails;
import com.playertour.backend.igolf.model.igolf.VectorGPSObject;


/**
 * Tests the correct implementation of the m2m transformations from the IGolf model to the GolfCourse and Country models
 * @since 1.0
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class IGolfModelTransformationsTest  {
	
	
	@InjectBundleContext
	BundleContext bundleContext;
	
	
	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 500) ServiceAware<ConfigurableModelTransformatorPool> transformatorPoolAware,
			@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") ServiceAware<PlayertourModelTransformator> golfTransformatorAware, 
			@InjectService ConfigurationAdmin ca)
			throws Exception {
		
		assertThat(transformatorPoolAware.getServices()).hasSize(1);	
		ServiceReference<ConfigurableModelTransformatorPool> poolReference = transformatorPoolAware.getServiceReference();
		assertThat(poolReference).isNotNull();		
				
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		ServiceReference<PlayertourModelTransformator> transformatorReference = golfTransformatorAware.getServiceReference();
		assertThat(transformatorReference).isNotNull();
		
		ConfigurableModelTransformatorPool poolService = transformatorPoolAware.getService();
		assertThat(poolService).isNotNull();		
		assertThat(poolService.getPoolMap()).hasSize(2);
	}
	
	@Test
	public void testCountryTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> golfTransformatorAware) throws Exception {
		
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = golfTransformatorAware.getService();
		assertThat(transformatorService).isNotNull();
				
		CountryInfo countryInfo = IGolfFactory.eINSTANCE.createCountryInfo();
		countryInfo.setCountryFull("Italy");
		countryInfo.setCountryShort("IT");
		countryInfo.setId_country(7);
		countryInfo.setId_continent("EU");
		
		Country country = transformatorService.transform(countryInfo, CountryPackage.Literals.COUNTRY);
		
		assertThat(country).isNotNull();
		assertThat(country.getCountryId()).isEqualTo("7");
		assertThat(country.getFullName()).isEqualTo("Italy");
		assertThat(country.getShortName()).isEqualTo("IT");
		assertThat(country.getContinentId()).isEqualTo("EU");
	}
	
	
	@Test
	public void testStateTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> golfTransformatorAware) throws Exception {
		
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = golfTransformatorAware.getService();
		assertThat(transformatorService).isNotNull();
				
		StateInfo stateInfo = IGolfFactory.eINSTANCE.createStateInfo();
		stateInfo.setStateFull("Thuringia");
		stateInfo.setStateShort("TH");
		stateInfo.setId_country(3);
		stateInfo.setId_state(2);
		
		State state = transformatorService.transform(stateInfo, CountryPackage.Literals.STATE);
		
		assertThat(state).isNotNull();
		assertThat(state.getId()).isEqualTo("2");
		assertThat(state.getFullName()).isEqualTo("Thuringia");
		assertThat(state.getShortName()).isEqualTo("TH");
		assertThat(state.getCountryId()).isEqualTo("3");
	}
	
	@Test
	public void testCourseDetailsTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> golfTransformatorAware) throws Exception {
		
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = golfTransformatorAware.getService();
		assertThat(transformatorService).isNotNull();
				
		CourseCompleteResponse courseResponse = IGolfFactory.eINSTANCE.createCourseCompleteResponse();
		CourseDetailsResponse courseDetailsResponse = IGolfFactory.eINSTANCE.createCourseDetailsResponse();
		
		courseDetailsResponse.setCourseName("Jena Golf Club");
		courseDetailsResponse.setAddress1("The Golf Street 7");
		courseDetailsResponse.setCity("Jena");
		courseDetailsResponse.setClass("public");
		courseDetailsResponse.setCountryFull("Germany");
		courseDetailsResponse.setCountryShort("DE");
		courseDetailsResponse.setDayClosed("Sun");
		courseDetailsResponse.setDressCode("denimok");
		courseDetailsResponse.setEmail("golfclubjena@gmail.com");
		courseDetailsResponse.setFax("034585729");
		courseDetailsResponse.setGpsAvailable(1);
		courseDetailsResponse.setId_country(3);
		courseDetailsResponse.setId_state(2);
		courseDetailsResponse.setImage("www.golfclubjena.com/image.jpg");
		courseDetailsResponse.setLayoutHoles(18);
		courseDetailsResponse.setLayoutName("eighteen");
		courseDetailsResponse.setLayoutTotalHoles(18);
		courseDetailsResponse.setPhone("37482148506");
		
		Price price = IGolfFactory.eINSTANCE.createPrice();
		price.setTwilight(13.46);
		price.setWeekday18(15.50);
		price.setWeekday9(10.50);
		price.setWeekend18(20.50);
		price.setWeekend9(15.50);
		courseDetailsResponse.setPricing(price);
		
		courseDetailsResponse.setScorecardAvailable(1);
		courseDetailsResponse.setSeasonal(0);
		courseDetailsResponse.setStateFull("Thuringia");
		courseDetailsResponse.setStateShort("TH");
		courseDetailsResponse.setUrl("www.golfclubjena.com");
		courseDetailsResponse.setZipCode("07743");
		
		courseResponse.setCourseDetails(courseDetailsResponse);
		
		GolfCourse golfCourse = transformatorService.transform(courseResponse, GolfCoursePackage.Literals.GOLF_COURSE);
		assertThat(golfCourse).isNotNull();
		
		CourseDetails courseDetails = golfCourse.getCourseDetails();
		assertThat(courseDetails).isNotNull();
		
		assertThat(courseDetails.getContact()).isNotNull();
		assertThat(courseDetails.getContact().getEmail()).isEqualTo("golfclubjena@gmail.com");
		assertThat(courseDetails.getContact().getPhone()).isEqualTo("37482148506");
		assertThat(courseDetails.getContact().getFax()).isEqualTo("034585729");
		assertThat(courseDetails.getContact().getUrl()).isEqualTo("www.golfclubjena.com");
		
		assertThat(courseDetails.getAddress()).isNotNull();
		assertThat(courseDetails.getAddress().getCity()).isEqualTo("Jena");
		assertThat(courseDetails.getAddress().getCountry()).isEqualTo("Germany");
		assertThat(courseDetails.getAddress().getRoute()).isEqualTo("The Golf Street 7");
		assertThat(courseDetails.getAddress().getState()).isEqualTo("Thuringia");
		assertThat(courseDetails.getAddress().getZipCode()).isEqualTo("07743");
		
		assertThat(courseDetails.getCountryId()).isEqualTo("3");
		assertThat(courseDetails.getStateId()).isEqualTo("2");
		assertThat(courseDetails.getDayClosed()).isEqualTo(WeekDay.SUN);
		assertThat(courseDetails.getDressCode()).isEqualTo(DressCodeType.DENIM_OK);
		assertThat(courseDetails.getType()).isEqualTo(GolfCourseType.PUBLIC);
		assertThat(courseDetails.getCourseName()).isEqualTo("Jena Golf Club");
		
		assertThat(courseDetails.getPrincing()).isNotNull();
		assertThat(courseDetails.getPrincing().getTwilight()).isEqualTo(13.46);
		assertThat(courseDetails.getPrincing().getWeekday18()).isEqualTo(15.50);
		assertThat(courseDetails.getPrincing().getWeekday9()).isEqualTo(10.50);
		assertThat(courseDetails.getPrincing().getWeekend18()).isEqualTo(20.50);
		assertThat(courseDetails.getPrincing().getWeekend9()).isEqualTo(15.50);		
	}
	
	
	@Test
	public void testCourseGPSVectorTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> golfTransformatorAware) throws Exception {
		
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = golfTransformatorAware.getService();
		assertThat(transformatorService).isNotNull();
				
		CourseCompleteResponse courseResponse = IGolfFactory.eINSTANCE.createCourseCompleteResponse();
		CourseGPSVectorResponse gpsVectorResponse = IGolfFactory.eINSTANCE.createCourseGPSVectorResponse();
		VectorGPSObject gpsObject = IGolfFactory.eINSTANCE.createVectorGPSObject();
		
		gpsObject.setHoleCount(18);
		ShapeObject shapeObj = IGolfFactory.eINSTANCE.createShapeObject();
		shapeObj.setShapeCount(2);
		
		Shape shape1 = IGolfFactory.eINSTANCE.createShape();
		shape1.setPoints("-117.2455445 32.9046491,"
				+ "-117.2452182 32.9046599,"
				+ "-117.2452117 32.9046030,"
				+ "-117.2451891 32.9046030,"
				+ "-117.2451859 32.9045433,"
				+ "-117.2452085 32.9045433,"
				+ "-117.2452053 32.9044077,"
				+ "-117.2451859 32.9044077,"
				+ "-117.2451827 32.9043426,"
				+ "-117.2451988 32.9043453,"
				+ "-117.2451956 32.9042857,"
				+ "-117.2455316 32.9042748,"
				+ "-117.2455348 32.9043752,"
				+ "-117.2454831 32.9043779,"
				+ "-117.2454863 32.9045569,"
				+ "-117.2455445 32.9045542,"
				+ "-117.2455445 32.9046491");
		
		Attributes attributes1 = IGolfFactory.eINSTANCE.createAttributes();
		attributes1.setDescription(1);
		shape1.setAttributes(attributes1);
		
		Shape shape2 = IGolfFactory.eINSTANCE.createShape();
		shape2.setPoints("-117.2474237 32.9058124");
		
		Attributes attributes2 = IGolfFactory.eINSTANCE.createAttributes();
		attributes2.setDescription(2);
		shape2.setAttributes(attributes2);
		
		Shapes shapes = IGolfFactory.eINSTANCE.createShapes();
		shapes.getShape().add(shape1);
		shapes.getShape().add(shape2);
		shapeObj.setShapes(shapes);
		
		gpsObject.setBackground(shapeObj);
		
		gpsVectorResponse.setVectorGPSObject(gpsObject);
		courseResponse.setCourseGPSVector(gpsVectorResponse);
		
		GolfCourse golfCourse = transformatorService.transform(courseResponse, GolfCoursePackage.Literals.GOLF_COURSE);
		assertThat(golfCourse).isNotNull();
		
		CourseGPSVector gpsVector = golfCourse.getCourseGPSVector();
		assertThat(gpsVector).isNotNull();
		
		assertThat(gpsVector.getHoleCount()).isEqualTo(18);
		assertThat(gpsVector.getBackgrund()).isNotNull();
		assertThat(gpsVector.getBackgrund().getShapeCount()).isEqualTo(2);
		assertThat(gpsVector.getBackgrund().getShapes()).isNotEmpty();
		assertThat(gpsVector.getBackgrund().getShapes()).hasSize(2);
		
		ShapeWrapper shapeWrapper1 = gpsVector.getBackgrund().getShapes().get(0).getWrapper();
		assertThat(shapeWrapper1).isNotNull();
		assertThat(shapeWrapper1.getIgolfString()).isEqualTo("-117.2455445 32.9046491,-117.2452182 32.9046599,-117.2452117 32.9046030,-117.2451891 32.9046030,-117.2451859 32.9045433,-117.2452085 32.9045433,-117.2452053 32.9044077,-117.2451859 32.9044077,-117.2451827 32.9043426,-117.2451988 32.9043453,-117.2451956 32.9042857,-117.2455316 32.9042748,-117.2455348 32.9043752,-117.2454831 32.9043779,-117.2454863 32.9045569,-117.2455445 32.9045542,-117.2455445 32.9046491");
		assertThat(shapeWrapper1.getPoints()).isNotEmpty();
		assertThat(shapeWrapper1.getPoints()).hasSize(17);
		
		assertThat(shapeWrapper1.getPoints().get(0).getLatitude()).isEqualTo(32.9046491);
		assertThat(shapeWrapper1.getPoints().get(0).getLongitude()).isEqualTo(-117.2455445);
		
		assertThat(shapeWrapper1.getPoints().get(2).getLatitude()).isEqualTo(32.9046030);
		assertThat(shapeWrapper1.getPoints().get(2).getLongitude()).isEqualTo(-117.2452117);
		
		assertThat(shapeWrapper1.getPoints().get(16).getLatitude()).isEqualTo(32.9046491);
		assertThat(shapeWrapper1.getPoints().get(16).getLongitude()).isEqualTo(-117.2455445);
		
		
		ShapeWrapper shapeWrapper2 = gpsVector.getBackgrund().getShapes().get(1).getWrapper();
		assertThat(shapeWrapper2).isNotNull();
		assertThat(shapeWrapper2.getIgolfString()).isEqualTo("-117.2474237 32.9058124");
		assertThat(shapeWrapper2.getPoints()).isNotEmpty();
		assertThat(shapeWrapper2.getPoints()).hasSize(1);
		
		assertThat(shapeWrapper2.getPoints().get(0).getLatitude()).isEqualTo(32.9058124);
		assertThat(shapeWrapper2.getPoints().get(0).getLongitude()).isEqualTo(-117.2474237);	
	}	
	
	@Test
	public void testScorecardTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> golfTransformatorAware) throws Exception {
		
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = golfTransformatorAware.getService();
		assertThat(transformatorService).isNotNull();
				
		ScorecardDetailsResponse iGolfScorecard = IGolfFactory.eINSTANCE.createScorecardDetailsResponse();
		Scorecard s1 = IGolfFactory.eINSTANCE.createScorecard();
		s1.setParIn(3);
		s1.setParOut(4);
		s1.setParTotal(5);
		s1.getHcpHole().add(1);
		s1.getHcpHole().add(2);
		
		Scorecard s2 = IGolfFactory.eINSTANCE.createScorecard();
		s2.setParIn(6);
		s2.setParOut(7);
		s2.setParTotal(3);
		s2.getHcpHole().add(2);
		s2.getHcpHole().add(7);
		
		Scorecard s3 = IGolfFactory.eINSTANCE.createScorecard();
		s3.setParIn(5);
		s3.setParOut(8);
		s3.setParTotal(2);
		s3.getHcpHole().add(4);
		s3.getHcpHole().add(5);
		
		iGolfScorecard.getMenScorecardList().add(s1);
		iGolfScorecard.getMenScorecardList().add(s2);
		iGolfScorecard.getWmnScorecardList().add(s3);
		
		CourseCompleteResponse response = IGolfFactory.eINSTANCE.createCourseCompleteResponse();
		CourseInfo info = IGolfFactory.eINSTANCE.createCourseInfo();
		info.setId_course("7");
		response.setCourseInfo(info);
		response.setScorecardDetails(iGolfScorecard);
		GolfCourse golfCourse = transformatorService.transform(response, 
				IGolfPackage.Literals.COURSE_COMPLETE_RESPONSE, GolfCoursePackage.Literals.GOLF_COURSE);
		
		assertThat(golfCourse).isNotNull();
		CourseScorecard courseScorecard = golfCourse.getScorecard();
		assertThat(courseScorecard).isNotNull();
		assertThat(courseScorecard.getCourseId()).isEqualTo("7");
		assertThat(courseScorecard.getMenScorecard()).isNotNull();
		assertThat(courseScorecard.getMenScorecard()).isNotEmpty();
		assertThat(courseScorecard.getMenScorecard()).hasSize(2);
		
		assertThat(courseScorecard.getWmnScorecard()).isNotNull();
		assertThat(courseScorecard.getWmnScorecard()).isNotEmpty();
		assertThat(courseScorecard.getWmnScorecard()).hasSize(1);
		
	}
	
	@Test
	public void testTeeTransformation(@InjectService(cardinality = 1, timeout = 500, filter = "(component.name=BEModelTransformator)") 
	ServiceAware<PlayertourModelTransformator> golfTransformatorAware) throws Exception {
		
		assertThat(golfTransformatorAware.getServices()).hasSize(1);	
		PlayertourModelTransformator transformatorService = golfTransformatorAware.getService();
		assertThat(transformatorService).isNotNull();
		
		CourseCompleteResponse iGolfCompleteRes = IGolfFactory.eINSTANCE.createCourseCompleteResponse();
		CourseTeeDetailsResponse iGolfResponse = IGolfFactory.eINSTANCE.createCourseTeeDetailsResponse();
		
		TeeDetails tee1 = IGolfFactory.eINSTANCE.createTeeDetails();
		tee1.setTeeName("Tee 1");
		tee1.setTeeColorName("Blue");
		tee1.setGender("men");
		tee1.setRatingMen(70.5);
		tee1.setSlopeMen(6);
		tee1.setYds10to18("245");
		tee1.setYds1to9("146");
		iGolfResponse.getTeesList().add(tee1);
		
		TeeDetails tee2 = IGolfFactory.eINSTANCE.createTeeDetails();
		tee2.setTeeName("Tee 2");
		tee2.setTeeColorName("Red");
		tee2.setGender("wmn");
		tee2.setRatingWomen(68.7);
		tee2.setSlopeWomen(3);
		tee2.setYds1to18("230");
		tee2.getYdsHole().add(120);
		tee2.getYdsHole().add(290);
		iGolfResponse.getTeesList().add(tee2);
		
		iGolfCompleteRes.setTeeDetails(iGolfResponse);

		GolfCourse golfCourse = transformatorService.transform(iGolfCompleteRes, IGolfPackage.Literals.COURSE_COMPLETE_RESPONSE, 
				GolfCoursePackage.Literals.GOLF_COURSE);
		
		assertThat(golfCourse).isNotNull();
		assertThat(golfCourse.getTee()).isNotNull();
		assertThat(golfCourse.getTee()).isNotEmpty();
		assertThat(golfCourse.getTee()).hasSize(2);
		
		Tee res1 = null, res2 = null;
		for(Tee t : golfCourse.getTee()) {
			if("Tee 1".equals(t.getName())) {
				res1 = t;
			}
			else if("Tee 2".equals(t.getName())) {
				res2 = t;
			}
		}
		assertThat(res1).isNotNull();
		assertThat(res2).isNotNull();
		
		assertThat(res1.getGender()).isEqualTo(GenderType.MEN);
		assertThat(res2.getGender()).isEqualTo(GenderType.WOMEN);
		
		assertThat(res1.getRating()).isEqualTo(70.5);
		assertThat(res1.getSlope()).isEqualTo(6);
		assertThat(res1.getColor()).isEqualTo("Blue");
		assertThat(res1.getYds10to18()).isEqualTo("245");
		assertThat(res1.getYds1to9()).isEqualTo("146");
		
		assertThat(res2.getRating()).isEqualTo(68.7);
		assertThat(res2.getSlope()).isEqualTo(3);
		assertThat(res2.getColor()).isEqualTo("Red");
		assertThat(res2.getYds1to18()).isEqualTo("230");
		assertThat(res2.getYdsHole()).isNotNull();
		assertThat(res2.getYdsHole()).isNotEmpty();
		assertThat(res2.getYdsHole()).hasSize(2);
		assertThat(res2.getYdsHole()).contains(120,290);		
	}
	
}
