/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.igolf.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.List;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.apis.igolf.CountryUpdateService;
import com.playertour.backend.apis.igolf.CourseUpdateService;
import com.playertour.backend.apis.igolf.exceptions.IGolfUpdateException;
import com.playertour.backend.country.model.country.Country;
import com.playertour.backend.country.model.country.CountryPackage;
import com.playertour.backend.country.model.country.State;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.igolf.model.igolf.CountryInfo;
import com.playertour.backend.igolf.model.igolf.CourseCompleteResponse;
import com.playertour.backend.igolf.model.igolf.StateInfo;


/**
 * Tests the implementation of the CourseUpdateService
 * @since 1.0
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class IGolfCourseUpdateTest {
	
	
	@InjectBundleContext
	BundleContext bundleContext;

	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryUpdateService> countryUpdateAware,
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseUpdateService> courseUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) {
				
		assertThat(countryUpdateAware.getServices()).hasSize(1);	
		ServiceReference<CountryUpdateService> coutryUpdateRef = countryUpdateAware.getServiceReference();
		assertThat(coutryUpdateRef).isNotNull();	
		
		assertThat(courseUpdateAware.getServices()).hasSize(1);	
		ServiceReference<CourseUpdateService> courseReference = courseUpdateAware.getServiceReference();
		assertThat(courseReference).isNotNull();		
		
		assertThat(repoAware.getServices()).hasSize(1);	
		ServiceReference<EMFRepository> repoRef = repoAware.getServiceReference();
		assertThat(repoRef).isNotNull();	
	}
	
//	@Test 
	public void testUpdatePerContinent(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryUpdateService> countryUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseUpdateService> courseUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws IGolfUpdateException {
		
		assertThat(countryUpdateAware.getServices()).hasSize(1);	
		CountryUpdateService countryUpdateService = countryUpdateAware.getService();
		assertThat(countryUpdateService).isNotNull();
		
		assertThat(courseUpdateAware.getServices()).hasSize(1);	
		CourseUpdateService courseUpdateService = courseUpdateAware.getService();
		assertThat(courseUpdateService).isNotNull();		
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		courseUpdateService.updateCoursesPerContinent("NA");
	}
	
//	@Test 
	public void testUpdatePerState(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryUpdateService> countryUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseUpdateService> courseUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws IGolfUpdateException {
		
		assertThat(countryUpdateAware.getServices()).hasSize(1);	
		CountryUpdateService countryUpdateService = countryUpdateAware.getService();
		assertThat(countryUpdateService).isNotNull();
		
		assertThat(courseUpdateAware.getServices()).hasSize(1);	
		CourseUpdateService courseUpdateService = courseUpdateAware.getService();
		assertThat(courseUpdateService).isNotNull();		
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		List<CountryInfo> countryList = countryUpdateService.updateCountryList("EU");
		assertThat(countryList).isNotNull();
		assertThat(countryList).isNotEmpty();		
		
		List<StateInfo> stateList = countryUpdateService.updateStateList("276"); //Germany
		assertThat(stateList).isNotNull();
		assertThat(stateList).isNotEmpty();	
		
		List<Country> countries = repo.getAllEObjects(CountryPackage.Literals.COUNTRY);
		assertThat(countries).isNotNull();
		assertThat(countries).isNotEmpty();	
		
		Country country = countries.stream().filter(c->"276".equals(c.getCountryId())).findFirst().orElse(null);
		assertThat(country).isNotNull();
		assertThat(country.getStates()).isNotNull();
		assertThat(country.getStates()).isNotEmpty();	
		
		State state = country.getStates().stream().filter(c->"224".equals(c.getId())).findFirst().orElse(null);
		assertThat(state).isNotNull();
		
		List<CourseCompleteResponse> result = courseUpdateService.updateCoursesPerState("224"); //Berlin
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();		
	}
	
//	@Test 
	public void testUpdatePerCountry(@InjectService(cardinality = 1, timeout = 500) ServiceAware<CountryUpdateService> countryUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<CourseUpdateService> courseUpdateAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws IGolfUpdateException {
		
		assertThat(countryUpdateAware.getServices()).hasSize(1);	
		CountryUpdateService countryUpdateService = countryUpdateAware.getService();
		assertThat(countryUpdateService).isNotNull();
		
		assertThat(courseUpdateAware.getServices()).hasSize(1);	
		CourseUpdateService courseUpdateService = courseUpdateAware.getService();
		assertThat(courseUpdateService).isNotNull();		
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		List<CountryInfo> countryList = countryUpdateService.updateCountryList("EU");
		assertThat(countryList).isNotNull();
		assertThat(countryList).isNotEmpty();		
		
		List<Country> countries = repo.getAllEObjects(CountryPackage.Literals.COUNTRY);
		assertThat(countries).isNotNull();
		assertThat(countries).isNotEmpty();	
		
		Country country = countries.stream().filter(c->"276".equals(c.getCountryId())).findFirst().orElse(null);
		assertThat(country).isNotNull();
		
		List<CourseCompleteResponse> result = courseUpdateService.updateCoursesPerCountry("276"); //Germany
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();	
		
		System.out.println("Country Update Requests " + countryUpdateService.getRequestCounter());
		System.out.println("Course Update Requests " + courseUpdateService.getRequestCounter());
		
	}
	
//	@AfterEach
	public void doAfterEach(@InjectService(cardinality = 1, timeout = 500, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(CountryPackage.Literals.COUNTRY.getName()).drop();
			database.getCollection(GolfCoursePackage.Literals.GOLF_COURSE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}
	
}
