/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.leaderboard.service.impl;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.leaderboard.model.leaderboard.LeaderBoardFactory;
import com.playertour.backend.leaderboard.model.leaderboard.LeaderBoardResult;
import com.playertour.backend.leaderboard.model.leaderboard.LeaderBoardResults;
import com.playertour.backend.leaderboard.service.api.LeaderBoardService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.player.model.player.PlayerPackage;

@Component(name = "LeaderBoardService", scope = ServiceScope.PROTOTYPE, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "LeaderBoardService")
@Designate(ocd = LeaderBoardServiceImpl.Config.class)
public class LeaderBoardServiceImpl implements LeaderBoardService {
	private static final Comparator<Player> LBPOINTS_COMPARATOR = (p1, p2) -> Integer
			.compare(p1.getProfile().getLeaderBoardPoints(), p2.getProfile().getLeaderBoardPoints());

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerService playerService;

	@Reference
	private PlayerSearchService playerSearchService;

	/********************************/
	/** Externalized configuration **/
	private Config config;

	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {

		int maxResults();
	}

	@Activate
	public void activate(Config config) {
		this.config = config;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.leaderboard.service.api.LeaderBoardService#getResults(java.lang.String)
	 */
	@Override
	public LeaderBoardResults getResults(String loginId) {
		Objects.requireNonNull(loginId, "Cannot retrieve leader-board results for Player with null loginId!");
		
		// FIXME: method takes ~ 3 seconds to execute on set of just 3 documents; very serious Lucene index issues - this is another example (recently with Challenges service, also when attempting to view index via 'Luke') 
		//List<Player> rankedPlayers = playerSearchService.getNPlayersSortedByPTA(MAX_RESULTS);
		
		List<Player> rankedPlayers = getNPlayersSortedByLBPoints(config.maxResults());
		
		Optional<Player> currentPlayerOptional = Optional.empty();
		if (!includesCurrentPlayer(rankedPlayers, loginId)) {
			currentPlayerOptional = Optional.of(playerService.getPlayerByLoginId(loginId));
			rankedPlayers.remove(rankedPlayers.size() - 1); // if currently logged in player is not included in the list, remove last element from list so it contains 'maxResults' at most
		}
		
		LeaderBoardResults results = repackagePlayers(rankedPlayers, loginId, currentPlayerOptional);		
		
		return results;
	}

	private boolean includesCurrentPlayer(List<Player> rankedPlayers, String loginId) {
		return rankedPlayers.stream().anyMatch(p -> p.getLoginId().equalsIgnoreCase(loginId));
	}

	private LeaderBoardResults repackagePlayers(List<Player> rankedPlayers, String currentPlayerLoginId,
			Optional<Player> currentPlayer) {
		LeaderBoardResults results = LeaderBoardFactory.eINSTANCE.createLeaderBoardResults();

		for (int i = 0; i < rankedPlayers.size(); i++) {
			LeaderBoardResult result = repackagePlayer(rankedPlayers.get(i), (i + 1), currentPlayerLoginId);
			results.getResults().add(result);
		}

		if (currentPlayer.isPresent()) {
			results.getResults()
					.add(repackagePlayer(currentPlayer.get(), (rankedPlayers.size() + 1), currentPlayerLoginId));
		}

		return results;
	}

	private LeaderBoardResult repackagePlayer(Player rankedPlayer, int rank, String currentPlayerLoginId) {
		LeaderBoardResult result = LeaderBoardFactory.eINSTANCE.createLeaderBoardResult();

		result.setRank(rank);
		result.setPoints(rankedPlayer.getProfile().getLeaderBoardPoints());
		result.setPlayerLoginId(rankedPlayer.getLoginId());
		result.setPlayerLoginName(rankedPlayer.getProfile().getLoginName());
		result.setPlayerName(rankedPlayer.getProfile().getName());
		result.setPlayerProfilePictureId(rankedPlayer.getProfile().getProfilePictureId());

		if (rankedPlayer.getLoginId().equalsIgnoreCase(currentPlayerLoginId)) {
			result.setCurrentlyLoggedIn(true);
		}

		return result;
	}

	private List<Player> getNPlayersSortedByLBPoints(int maxResults) {
		List<Player> allPlayers = repository.getAllEObjects(PlayerPackage.Literals.PLAYER);

		if (!allPlayers.isEmpty()) {
			// @formatter:off
			return allPlayers.stream()
					.sorted(LBPOINTS_COMPARATOR.reversed())
					.limit(maxResults)
					.collect(Collectors.toList());
			// @formatter:on

		} else {
			return Collections.emptyList();
		}
	}

	// FIXME: sorting via 'org.gecko.emf.repository.query.IQuery' does not work
	/*
	private List<Player> getNPlayersSortedByPTA(int maxResults) {
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);
		
		// @formatter:off
		IQuery query = queryRepo.createQueryBuilder()
				.allQuery()
			  //.sort(PlayerApiPackage.Literals.PLAYER_PROFILE__PLAYER_TOUR_ABILITY, SortType.ASCENDING)
			  //.sort(PlayerApiPackage.Literals.PLAYER_PROFILE__PLAYER_TOUR_ABILITY, SortType.DESCENDING)	
				.limit(maxResults)
				.build();
		// @formatter:on
		
		return queryRepo.getEObjectsByQuery(PlayerPackage.Literals.PLAYER, query);
	}
	*/
	
}
