/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.leaderboard.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.apis.player.exceptions.PlayerUsernameValidationException;
import com.playertour.backend.leaderboard.model.leaderboard.LeaderBoardResult;
import com.playertour.backend.leaderboard.model.leaderboard.LeaderBoardResults;
import com.playertour.backend.leaderboard.service.api.LeaderBoardService;
import com.playertour.backend.player.model.player.PlayerPackage;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class LeaderBoardServiceTest {
	
	@InjectBundleContext
	BundleContext bundleContext;
	
	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<LeaderBoardService> leaderBoardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(leaderBoardServiceAware.getServices()).hasSize(1);
		ServiceReference<LeaderBoardService> leaderBoardServiceReference = leaderBoardServiceAware
				.getServiceReference();
		assertThat(leaderBoardServiceReference).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerRef = playerServiceAware.getServiceReference();
		assertThat(playerRef).isNotNull();			
	}
	
	@Test
	public void testGetResults(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<LeaderBoardService> leaderBoardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerAware) throws InterruptedException, PlayerUsernameValidationException {
		
		assertThat(leaderBoardServiceAware.getServices()).hasSize(1);	
		LeaderBoardService leaderBoardService = leaderBoardServiceAware.getService();
		assertThat(leaderBoardService).isNotNull();	
		
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		PlayerProfile profile1 = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile1.setLoginName("mario.rossi");
		profile1.setPlayerTourAbility(10);
		profile1.setLeaderBoardPoints(25);
		String loginId1 = UUID.randomUUID().toString();
		playerService.saveProfile(profile1, loginId1);
		
		PlayerProfile profile2 = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile2.setLoginName("carlo.verdi");
		profile2.setPlayerTourAbility(-5.25);
		profile2.setLeaderBoardPoints(195);
		String loginId2 = UUID.randomUUID().toString();
		playerService.saveProfile(profile2, loginId2);
		
		PlayerProfile profile3 = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile3.setLoginName("engelbert.humperdinck");
		profile3.setPlayerTourAbility(6.5);
		profile3.setLeaderBoardPoints(50);
		String loginId3 = UUID.randomUUID().toString();
		playerService.saveProfile(profile3, loginId3);
		
		PlayerProfile profile4 = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile4.setLoginName("kunegunden.kutasiungen");
		profile4.setPlayerTourAbility(2.75);
		profile4.setLeaderBoardPoints(100);
		String loginId4 = UUID.randomUUID().toString();
		playerService.saveProfile(profile4, loginId4);
		
		Thread.sleep(2000);
		
		LeaderBoardResults results = leaderBoardService.getResults(loginId1);
		assertThat(results).isNotNull();
		assertThat(results.getResults()).isNotEmpty();
		assertThat(results.getResults()).hasSizeBetween(1, 5);
		
		LeaderBoardResult result = results.getResults().get(0);
		assertThat(result.getPlayerLoginId()).isEqualTo(loginId2);
		
		result = results.getResults().get(1);
		assertThat(result.getPlayerLoginId()).isEqualTo(loginId4);
		
		result = results.getResults().get(3);
		assertThat(result.getPlayerLoginId()).isEqualTo(loginId1);
		assertThat(result.isCurrentlyLoggedIn()).isTrue();
	}
	
	@BeforeEach
	@AfterEach
	public void clean(@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> indexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(indexAware.getServices()).hasSize(1);	
		CommonIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();		
		
		indexService.resetIndex();
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
