/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.meritpoints.service.impl;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Objects;

import org.gecko.emf.repository.EMFRepository;
import org.javatuples.Pair;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;

import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsBalanceResult;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsFactory;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsTransaction;
import com.playertour.backend.meritpoints.model.meritpoints.TransactionType;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountIndexService;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountInsufficientBalanceException;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountSearchService;
import com.playertour.backend.meritpoints.service.api.MeritPointsService;

@Component(name = "MeritPointsService", scope = ServiceScope.PROTOTYPE, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "MeritPointsService")
@Designate(ocd = MeritPointsServiceImpl.Config.class)
public class MeritPointsServiceImpl implements MeritPointsService {
	
	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;
	
	@Reference
	private MeritPointsAccountIndexService meritPointsAccountIndexService;
	
	@Reference
	private MeritPointsAccountSearchService meritPointsAccountSearchService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerService playerService;	
	
	private Config config;
	
	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {
		
		// Whether merit points should be migrated from PlayerProfile on new account creation 
		boolean migrateMeritPoints();
	}
	
	@Activate
	public void activate(Config config) {
		this.config = config;
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsService#deposit(java.lang.String, java.math.BigDecimal, java.lang.String)
	 */
	@Override
	public MeritPointsAccount deposit(String receiverUserID, BigDecimal amount, String description) {
		Objects.requireNonNull(receiverUserID, "Receiver user ID is required!");
		Objects.requireNonNull(amount, "Amount is required!");
		Objects.requireNonNull(description, "Transaction description is required!");
		
		MeritPointsAccount account = meritPointsAccountSearchService.searchAccountByUserID(receiverUserID);
		if (account == null) {
			account = createAccount(receiverUserID);
		}

		account.setBalance(account.getBalance().add(amount));

		MeritPointsTransaction transaction = createTransaction(TransactionType.CREDIT, amount, null, receiverUserID,
				description);
		account.getTransactions().add(transaction);

		return saveOrUpdateAccount(account);
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsService#withdraw(java.lang.String, java.math.BigDecimal, java.lang.String)
	 */
	@Override
	public MeritPointsAccount withdraw(String payeeUserID, BigDecimal amount, String description) throws MeritPointsAccountInsufficientBalanceException {
		Objects.requireNonNull(payeeUserID, "Payee user ID is required!");
		Objects.requireNonNull(amount, "Amount is required!");
		Objects.requireNonNull(description, "Transaction description is required!");		
		
		MeritPointsAccount account = meritPointsAccountSearchService.searchAccountByUserID(payeeUserID);
		if (account == null) {
			account = createAccount(payeeUserID);
		}		

		if ((account.getBalance() == null)
				|| (account.getBalance() != null && (account.getBalance().subtract(amount).signum() == -1))) {
			throw new MeritPointsAccountInsufficientBalanceException(
					"Merit points account for user: " + payeeUserID + " has insufficient balance!");
		}

		account.setBalance(account.getBalance().subtract(amount));

		MeritPointsTransaction transaction = createTransaction(TransactionType.DEBIT, amount, payeeUserID, null,
				description);
		account.getTransactions().add(transaction);

		return saveOrUpdateAccount(account);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsService#transfer(java.lang.String, java.lang.String, java.math.BigDecimal, java.lang.String, java.lang.String)
	 */
	@Override
	public Pair<MeritPointsAccount, MeritPointsAccount> transfer(String payeeUserID, String receiverUserID,
			BigDecimal amount, String payeeTransactionDescription, String receiverTransactionDescription) {
		Objects.requireNonNull(payeeUserID, "Payee user ID is required!");
		Objects.requireNonNull(receiverUserID, "Receiver user ID is required!");
		Objects.requireNonNull(amount, "Amount is required!");
		Objects.requireNonNull(payeeTransactionDescription, "Payee transaction description is required!");
		Objects.requireNonNull(receiverTransactionDescription, "Receiver transaction description is required!");

		MeritPointsAccount payeeAccount = withdraw(payeeUserID, amount, payeeTransactionDescription);

		MeritPointsAccount receiverAccount = deposit(receiverUserID, amount, receiverTransactionDescription);

		return Pair.with(payeeAccount, receiverAccount);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsService#transferWithCommission(java.lang.String, java.lang.String, java.math.BigDecimal, double, java.lang.String, java.lang.String)
	 */
	@Override
	public Pair<MeritPointsAccount, MeritPointsAccount> transferWithCommission(String payeeUserID,
			String receiverUserID, BigDecimal amount, double commission, String payeeTransactionDescription,
			String receiverTransactionDescription) {
		Objects.requireNonNull(payeeUserID, "Payee user ID is required!");
		Objects.requireNonNull(receiverUserID, "Receiver user ID is required!");
		Objects.requireNonNull(amount, "Amount is required!");
		Objects.requireNonNull(commission, "Commission is required!");
		Objects.requireNonNull(payeeTransactionDescription, "Payee transaction description is required!");
		Objects.requireNonNull(receiverTransactionDescription, "Receiver transaction description is required!");
		
		BigDecimal payeeAmountWithCommission = calculatePayeeAmountWithCommission(amount, commission);
		
		BigDecimal receiverAmountWithCommission = calculateReceiverAmountWithCommission(amount, commission);

		MeritPointsAccount payeeAccount = withdraw(payeeUserID, payeeAmountWithCommission, payeeTransactionDescription);

		MeritPointsAccount receiverAccount = deposit(receiverUserID, receiverAmountWithCommission, receiverTransactionDescription);

		return Pair.with(payeeAccount, receiverAccount);
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsService#getBalance(java.lang.String)
	 */
	@Override
	public BigDecimal getBalance(String userId) {
		Objects.requireNonNull(userId, "User ID is required!");
		
		MeritPointsAccount account = meritPointsAccountSearchService.searchAccountByUserID(userId);
		if (account == null) {
			account = createAccount(userId);
		}
		
		return saveOrUpdateAccount(account).getBalance();
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsService#createMeritPointsBalanceResult(java.math.BigDecimal)
	 */
	@Override
	public MeritPointsBalanceResult createMeritPointsBalanceResult(BigDecimal balance) {
		Objects.requireNonNull(balance, "Balance is required!");
		
		MeritPointsBalanceResult result = MeritPointsFactory.eINSTANCE.createMeritPointsBalanceResult();
		result.setBalance(balance);
		return result;
	}	

	private MeritPointsAccount saveOrUpdateAccount(MeritPointsAccount account) {
		Objects.requireNonNull(account, "Merit points account is required!");
		Objects.requireNonNull(account.getUserID(), "Merit points account user ID is required!");

		MeritPointsAccount existingAccount = meritPointsAccountSearchService.searchAccountByUserID(account.getUserID());
		boolean isFirstSave = existingAccount == null;

		repository.save(account);

		meritPointsAccountIndexService.indexAccount(account, isFirstSave);

		return account;
	}
	
	private MeritPointsAccount createAccount(String userId) {
		MeritPointsAccount account = MeritPointsFactory.eINSTANCE.createMeritPointsAccount();
		account.setUserID(userId);
		
		BigDecimal balance = BigDecimal.ZERO;
		
		if (config.migrateMeritPoints()) {
			PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(userId);
			if ((playerProfile != null) && (playerProfile.getMeritPoints() != null)) {
				balance = new BigDecimal(playerProfile.getMeritPoints());
			}
		}
		
		account.setBalance(balance);
		return account;
	}	

	private MeritPointsTransaction createTransaction(TransactionType type, BigDecimal amount, String payeeUserID,
			String receiverUserID, String description) {
		MeritPointsTransaction transaction = MeritPointsFactory.eINSTANCE.createMeritPointsTransaction();

		transaction.setType(type);
		transaction.setAmount(amount);
		if (payeeUserID != null) {
			transaction.setPayeeUserID(payeeUserID);
		}
		if (receiverUserID != null) {
			transaction.setReceiverUserID(receiverUserID);
		}
		if (description != null) {
			transaction.setDescription(description);
		}
		transaction.setTimestamp(LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli());

		return transaction;
	}
	
	private BigDecimal calculatePayeeAmountWithCommission(BigDecimal amount, double commission) {
		return amount.add(amount.multiply(BigDecimal.valueOf(commission)));
	}

	private BigDecimal calculateReceiverAmountWithCommission(BigDecimal amount, double commission) {
		return amount.subtract(amount.multiply(BigDecimal.valueOf(commission)));
	}	
}
