/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.meritpoints.service.impl;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.UriMapProvider;
import org.gecko.emf.pushstream.EPushStreamProvider;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.LuceneIndexService;
import org.gecko.search.util.CommonIndexService;
import org.gecko.util.pushstreams.GeckoPushbackPolicyOption;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.QueuePolicyOption;

import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsPackage;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountIndexService;

@Component(immediate = true, name = "MeritPointsAccountIndexService", service = MeritPointsAccountIndexService.class, scope = ServiceScope.SINGLETON, reference = {
		@Reference(name = "mongoCondition", service = UriMapProvider.class, target = "(uri.map.src=mongodb://playertour/)", cardinality = ReferenceCardinality.MANDATORY),
		@Reference(name = "modelCondition", service = ResourceSetFactory.class, target = "(emf.model.name=meritpoints)", cardinality = ReferenceCardinality.MANDATORY) })
public class MeritPointsAccountIndexServiceImpl implements MeritPointsAccountIndexService {
	
	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(target = "(id=meritPointsAccounts)")
	private LuceneIndexService meritPointsAccountsIndex;
	
	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));
	
	@Activate
	public void activate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(100, TimeUnit.MILLISECONDS);
			initializeIndex();
			return true;
		}).onFailure(t -> t.printStackTrace());	
	}	

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.search.util.CommonIndexService#initializeIndex()
	 */
	@Override
	public Promise<Void> initializeIndex() {
		EMFRepository repository = repositoryServiceObjects.getService();
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		IQuery query = queryRepo.createQueryBuilder().allQuery().build();
		logger.info("Starting index build for Purchases");

		resetIndex();

		EPushStreamProvider psp = queryRepo.getEObjectByQuery(MeritPointsPackage.Literals.MERIT_POINTS_ACCOUNT, query,
				CommonIndexService.getLoadOptions());
		if (psp == null) {
			return null;
		}

		// @formatter:off
		PushStream<EObject> indexNew = psp.createPushStreamBuilder()
				.withPushbackPolicy(GeckoPushbackPolicyOption.LINEAR_AFTER_THRESHOLD.getPolicy(650))
				.withQueuePolicy(QueuePolicyOption.BLOCK)
				.withExecutor(Executors.newSingleThreadExecutor())
				.withBuffer(new ArrayBlockingQueue<PushEvent<? extends EObject>>(1200))
				.build();

		return indexNew
				.map(eo -> (MeritPointsAccount) eo)
				.map(MeritPointsAccountIndexHelper::mapAccountNew)
				.forEach(d -> {
					repository.detach(d.getObject());
					meritPointsAccountsIndex.handleContext(d);
				}).onFailure(t -> repositoryServiceObjects.ungetService(repository))
				.thenAccept(result -> logger.info("Finished index build for merit points accounts"))
				.thenAccept(result -> repositoryServiceObjects.ungetService(repository));
		// @formatter:on
	}

	/* 
	 * (non-Javadoc)
	 * @see org.gecko.search.util.CommonIndexService#resetIndex()
	 */
	@Override
	public void resetIndex() {
		try {
			meritPointsAccountsIndex.getIndexWriter().deleteAll();
			meritPointsAccountsIndex.commit();
		} catch (IOException e) {
			logger.error("Could not delete merit points accounts index", e);
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsAccountIndexService#indexAccount(com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount, boolean)
	 */
	@Override
	public void indexAccount(MeritPointsAccount account, boolean isFirstSave) {
		if (isFirstSave) {
			indexAccount(account, IndexActionType.ADD);
		} else {
			indexAccount(account, IndexActionType.MODIFY);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsAccountIndexService#deleteAccount(com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount)
	 */
	@Override
	public void deleteAccount(MeritPointsAccount account) {
		indexAccount(account, IndexActionType.REMOVE);
	}

	private void indexAccount(MeritPointsAccount account, IndexActionType actionType) {
		DocumentIndexContextObject context = MeritPointsAccountIndexHelper.mapAccount(account, actionType, null);
		meritPointsAccountsIndex.handleContextSync(context);
	}
}
