/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.meritpoints.service.impl;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.document.LuceneIndexService;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountSearchService;

@Component(name = "MeritPointsAccountSearchService", immediate = true, scope = ServiceScope.SINGLETON)
public class MeritPointsAccountSearchServiceImpl implements MeritPointsAccountSearchService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference(target = "(id=meritPointsAccounts)")
	private LuceneIndexService meritPointsAccountsIndex;
	
	@Reference
	private ResourceSet resourceSet;
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.meritpoints.service.api.MeritPointsAccountSearchService#searchAccountByUserID(java.lang.String)
	 */
	@Override
	public MeritPointsAccount searchAccountByUserID(String userId) {
		IndexSearcher searcher = meritPointsAccountsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				String exactMatch = userId.trim().toLowerCase();
				Query q1 = new TermQuery(new Term(MeritPointsAccountIndexHelper.USER_ID_LOWER, exactMatch));
				Query q = new BooleanQuery.Builder().add(new BooleanClause(q1, Occur.SHOULD)).build();

				TopDocs topDocs = searcher.search(q, 1);

				if (topDocs.scoreDocs.length == 0) {
					return null;
				}

				IndexReader indexReader = searcher.getIndexReader();

				Document doc = indexReader.document(topDocs.scoreDocs[0].doc);
				
				return (MeritPointsAccount) LuceneIndexHelper.toEObject(doc, repository);
				
			} catch (Exception e) {
				logger.error("Exception while searching for accounts by user ID " + userId, e);

				return null;
			}
		} finally {
			meritPointsAccountsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}
}
