/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.meritpoints.service.tests;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatExceptionOfType;

import java.math.BigDecimal;
import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.javatuples.Pair;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsAccount;
import com.playertour.backend.meritpoints.model.meritpoints.MeritPointsPackage;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountIndexService;
import com.playertour.backend.meritpoints.service.api.MeritPointsAccountInsufficientBalanceException;
import com.playertour.backend.meritpoints.service.api.MeritPointsService;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class MeritPointsServiceTest {
	
	@InjectBundleContext
	BundleContext bundleContext;

	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		ServiceReference<MeritPointsService> meritPointsServiceReference = meritPointsServiceAware
				.getServiceReference();
		assertThat(meritPointsServiceReference).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerRef = playerServiceAware.getServiceReference();
		assertThat(playerRef).isNotNull();		
	}
	
	@Test
	public void testGetMeritPointsBalance(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		String userId1 = UUID.randomUUID().toString();
		String username1 = "engelbert.humperdinck";		
		
		PlayerProfile profile1 = playerService.getPlayerProfile(userId1, username1);
		assertThat(profile1).isNotNull();
		
		Thread.sleep(2000);
		
		BigDecimal balance = meritPointsService.getBalance(userId1);
		assertThat(balance).isNotNull();
		assertThat(balance).isEqualTo(BigDecimal.ZERO);
	}	
	
	@Test
	public void testDepositMeritPoints(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		String userId1 = UUID.randomUUID().toString();
		String username1 = "engelbert.humperdinck";
		
		BigDecimal depositAmount = BigDecimal.TEN;
		String depositDescription = String.format("Deposit of %.2f merit points", depositAmount);
		
		PlayerProfile profile1 = playerService.getPlayerProfile(userId1, username1);
		assertThat(profile1).isNotNull();
		
		Thread.sleep(2000);
		
		MeritPointsAccount account = meritPointsService.deposit(userId1, depositAmount, depositDescription);
		assertThat(account).isNotNull();
		assertThat(account.getUserID()).isEqualTo(userId1);
		assertThat(account.getBalance()).isEqualTo(depositAmount);
		assertThat(account.getTransactions()).isNotNull();
		assertThat(account.getTransactions()).isNotEmpty();
		assertThat(account.getTransactions()).hasSize(1);
	}
	
	@Test
	public void testWithdrawMeritPoints(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		String userId1 = UUID.randomUUID().toString();
		String username1 = "engelbert.humperdinck";
		
		BigDecimal depositAmount = BigDecimal.TEN;
		String depositDescription = String.format("Deposit of %.2f merit points", depositAmount);
		
		BigDecimal withdrawalAmount = BigDecimal.ONE;
		String withdrawalDescription = String.format("Withdrawal of %.2f merit point", withdrawalAmount);		
		
		PlayerProfile profile1 = playerService.getPlayerProfile(userId1, username1);
		assertThat(profile1).isNotNull();
		
		Thread.sleep(2000);
		
		MeritPointsAccount account = meritPointsService.deposit(userId1, depositAmount, depositDescription);
		assertThat(account).isNotNull();
		assertThat(account.getUserID()).isEqualTo(userId1);
		assertThat(account.getBalance()).isEqualTo(depositAmount);
		assertThat(account.getTransactions()).isNotNull();
		assertThat(account.getTransactions()).isNotEmpty();
		assertThat(account.getTransactions()).hasSize(1);
		
		Thread.sleep(2000);
		
		account = meritPointsService.withdraw(userId1, withdrawalAmount, withdrawalDescription);
		assertThat(account.getBalance()).isEqualTo(depositAmount.subtract(withdrawalAmount));
		assertThat(account.getTransactions()).isNotNull();
		assertThat(account.getTransactions()).isNotEmpty();
		assertThat(account.getTransactions()).hasSize(2);
	}
	
	@Test
	public void testWithdrawMeritPointsInsufficientBalance(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		String userId1 = UUID.randomUUID().toString();
		String username1 = "engelbert.humperdinck";
		
		BigDecimal depositAmount = BigDecimal.ZERO;
		String depositDescription = String.format("Deposit of %.2f merit points", depositAmount);
		
		BigDecimal withdrawalAmount = BigDecimal.ONE;
		String withdrawalDescription = String.format("Withdrawal of %.2f merit point", withdrawalAmount);		
		
		PlayerProfile profile1 = playerService.getPlayerProfile(userId1, username1);
		assertThat(profile1).isNotNull();
		
		Thread.sleep(2000);
		
		assertThatExceptionOfType(MeritPointsAccountInsufficientBalanceException.class).isThrownBy(() -> { 
			
			MeritPointsAccount account = meritPointsService.deposit(userId1, depositAmount, depositDescription);
			assertThat(account).isNotNull();
			assertThat(account.getUserID()).isEqualTo(userId1);
			assertThat(account.getBalance()).isEqualTo(depositAmount);
			assertThat(account.getTransactions()).isNotNull();
			assertThat(account.getTransactions()).isNotEmpty();
			assertThat(account.getTransactions()).hasSize(1);
			
			Thread.sleep(2000);
			
			meritPointsService.withdraw(userId1, withdrawalAmount, withdrawalDescription);
		});		
	}
	
	@Test
	public void testTransferMeritPoints(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();		
		
		String userId1 = UUID.randomUUID().toString();
		String username1 = "engelbert.humperdinck";
		
		BigDecimal depositAmount = BigDecimal.TEN;
		String depositDescription = String.format("Deposit of %.2f merit points", depositAmount);
		
		PlayerProfile profile1 = playerService.getPlayerProfile(userId1, username1);
		assertThat(profile1).isNotNull();
		
		Thread.sleep(2000);
		
		String userId2 = UUID.randomUUID().toString();
		String username2 = "kunegunden.kutasiungen";
		
		BigDecimal withdrawalAmount = BigDecimal.ONE;
		BigDecimal transferAmount = withdrawalAmount;
		String withdrawalDescription = String.format("Withdrawal of %.2f merit point", withdrawalAmount);
		String transferDescription = String.format("Transfer of %.2f merit point", transferAmount);	
		
		Thread.sleep(2000);
		
		PlayerProfile profile2 = playerService.getPlayerProfile(userId2, username2);
		assertThat(profile2).isNotNull();
		
		MeritPointsAccount account = meritPointsService.deposit(userId1, depositAmount, depositDescription);
		assertThat(account).isNotNull();
		assertThat(account.getUserID()).isEqualTo(userId1);
		assertThat(account.getBalance()).isEqualTo(depositAmount);
		assertThat(account.getTransactions()).isNotNull();
		assertThat(account.getTransactions()).isNotEmpty();
		assertThat(account.getTransactions()).hasSize(1);
		
		Thread.sleep(2000);
		
		Pair<MeritPointsAccount, MeritPointsAccount> payeeReceiverAccounts = meritPointsService.transfer(userId1, userId2, transferAmount, withdrawalDescription, transferDescription);
		assertThat(payeeReceiverAccounts.getValue0()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue1()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue0().getUserID()).isEqualTo(userId1);
		assertThat(payeeReceiverAccounts.getValue1().getUserID()).isEqualTo(userId2);
		assertThat(payeeReceiverAccounts.getValue0().getBalance()).isEqualTo(depositAmount.subtract(withdrawalAmount));
		assertThat(payeeReceiverAccounts.getValue1().getBalance()).isEqualTo(transferAmount);
		assertThat(payeeReceiverAccounts.getValue0().getTransactions()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue0().getTransactions()).isNotEmpty();
		assertThat(payeeReceiverAccounts.getValue0().getTransactions()).hasSize(2);
		assertThat(payeeReceiverAccounts.getValue1().getTransactions()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue1().getTransactions()).isNotEmpty();
		assertThat(payeeReceiverAccounts.getValue1().getTransactions()).hasSize(1);		
	}
	
	@Test
	public void testTransferMeritPointsWithCommission(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsService> meritPointsServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(meritPointsServiceAware.getServices()).hasSize(1);
		MeritPointsService meritPointsService = meritPointsServiceAware.getService();
		assertThat(meritPointsService).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();		
		
		String userId1 = UUID.randomUUID().toString();
		String username1 = "engelbert.humperdinck";
		
		BigDecimal depositAmount = BigDecimal.TEN;
		String depositDescription = String.format("Deposit of %.2f merit points", depositAmount);
		
		PlayerProfile profile1 = playerService.getPlayerProfile(userId1, username1);
		assertThat(profile1).isNotNull();
		
		Thread.sleep(2000);
		
		String userId2 = UUID.randomUUID().toString();
		String username2 = "kunegunden.kutasiungen";
		
		BigDecimal withdrawalAmount = BigDecimal.ONE;
		BigDecimal transferAmount = withdrawalAmount;
		double commission = 0.10;		
		String withdrawalDescription = String.format("Withdrawal of %.2f merit point", withdrawalAmount);
		String transferDescription = String.format("Transfer of %.2f merit point", transferAmount);	
		
		Thread.sleep(2000);
		
		PlayerProfile profile2 = playerService.getPlayerProfile(userId2, username2);
		assertThat(profile2).isNotNull();
		
		MeritPointsAccount account = meritPointsService.deposit(userId1, depositAmount, depositDescription);
		assertThat(account).isNotNull();
		assertThat(account.getUserID()).isEqualTo(userId1);
		assertThat(account.getBalance()).isEqualTo(depositAmount);
		assertThat(account.getTransactions()).isNotNull();
		assertThat(account.getTransactions()).isNotEmpty();
		assertThat(account.getTransactions()).hasSize(1);
		
		Thread.sleep(2000);
		
		Pair<MeritPointsAccount, MeritPointsAccount> payeeReceiverAccounts = meritPointsService.transferWithCommission(
				userId1, userId2, transferAmount, commission, withdrawalDescription, transferDescription);
		
		assertThat(payeeReceiverAccounts.getValue0()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue1()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue0().getUserID()).isEqualTo(userId1);
		assertThat(payeeReceiverAccounts.getValue1().getUserID()).isEqualTo(userId2);
		assertThat(payeeReceiverAccounts.getValue0().getBalance()).isEqualTo(depositAmount.subtract(withdrawalAmount.add(withdrawalAmount.multiply(BigDecimal.valueOf(commission)))));				
		assertThat(payeeReceiverAccounts.getValue1().getBalance()).isEqualTo(transferAmount.subtract(transferAmount.multiply(BigDecimal.valueOf(commission))));		
		assertThat(payeeReceiverAccounts.getValue0().getTransactions()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue0().getTransactions()).isNotEmpty();
		assertThat(payeeReceiverAccounts.getValue0().getTransactions()).hasSize(2);
		assertThat(payeeReceiverAccounts.getValue1().getTransactions()).isNotNull();
		assertThat(payeeReceiverAccounts.getValue1().getTransactions()).isNotEmpty();
		assertThat(payeeReceiverAccounts.getValue1().getTransactions()).hasSize(1);	
	}		
	
	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<MeritPointsAccountIndexService> meritPointsAccountIndexServiceAware) {

		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		assertThat(meritPointsAccountIndexServiceAware.getServices()).hasSize(1);
		MeritPointsAccountIndexService meritPointsAccountIndexService = meritPointsAccountIndexServiceAware.getService();
		assertThat(meritPointsAccountIndexService).isNotNull();

		try {
			dbProvider.getDatabase().getCollection(MeritPointsPackage.Literals.MERIT_POINTS_ACCOUNT.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}

		meritPointsAccountIndexService.resetIndex();
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}	
}
