/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.notifications.firebasemessaging.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.notifications.firebasemessaging.service.api.FirebaseMessagingService;
import com.playertour.backend.player.model.player.PlayerPackage;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class FirebaseMessagingIntegrationTest {

	private static final String PLAYER_USER_NAME = "testuser";

	private static final String PLAYER_LOGIN_ID = "1ce9d204-50b8-4f8a-8270-5222db24514d";

	// valid token
	private static final String FIREBASE_MESSAGING_TOKEN_VALID = "c_dw3KCNSYOd5liwU031IM:APA91bEOZzg25ZLM8fmGDVrpaKdChZ8oF933jD9J3rP1QhNNaU_HYCmc2EJorBBQB3POVhsjxyk7rmyBPUtU2gs1kGGt_VXyU0ZgteF9a5FrMTW44S7Ybac015_ufLSwHbMxjGhxtYRL";

	// incorrect token #1 (400/INVALID_ARGUMENT error)
	private static final String FIREBASE_MESSAGING_TOKEN_INVALID = "cIA93p_XTOWwGCASZcFiIq:APA91bGHTp7Rdbx7_I0d2xWewg7PCrp6YzLnXuiTwLKrz8SWXomTPQwOwFNqLqBxBOUQXopP92c50iMMKmBryjT_3tcFvmonbyJn16-53vNq0gzkeYNu6dEgct-SpCCe8sm0PEDm2Tj30";

	// incorrect token (404/UNREGISTERED error)
	private static final String FIREBASE_MESSAGING_TOKEN_EXPIRED = "dC2DU3DvQAWlWAbd0FvLIc:APA91bHEh-M3w5mkaRK-Qf4ZUFdo_-qdfXqnnFWHWs8Edfbb9JJWtAiZtqmOp3XBFX_DdSn_i44Wyj0oIgsdgl8KsjAxxhZSXMGs19kO8Yva6hJPr9lyv0DKDQNQhmAShUXawxrQwOOz";

	private static final Map<String, String> FIREBASE_MESSAGING_MESSAGE_DATA;
	static {
		FIREBASE_MESSAGING_MESSAGE_DATA = new HashMap<String, String>();
		FIREBASE_MESSAGING_MESSAGE_DATA.put("score", "850");
		FIREBASE_MESSAGING_MESSAGE_DATA.put("time", "2:45");
	}
	private static final String FIREBASE_MESSAGING_MESSAGE_TITLE = "Message sent via integration test";
	private static final String FIREBASE_MESSAGING_MESSAGE_BODY = "Body of message sent via integration test";

	@InjectBundleContext
	BundleContext bundleContext;

	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		ServiceReference<FirebaseMessagingService> firebaseMessagingServiceReference = firebaseMessagingServiceAware
				.getServiceReference();
		assertThat(firebaseMessagingServiceReference).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerRef = playerServiceAware.getServiceReference();
		assertThat(playerRef).isNotNull();
	}

	@Test
	public void testSaveToken(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();
	}

	@Test
	public void testGetTokens(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		List<String> savedTokens = firebaseMessagingService.getTokens(PLAYER_LOGIN_ID);
		assertThat(savedTokens).isNotNull();
		assertThat(savedTokens).isNotEmpty();
		assertThat(savedTokens).hasSize(1);
	}
	
	@Test
	public void testGetMultipleTokens(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		String token1 = UUID.randomUUID().toString();
		String token2 = UUID.randomUUID().toString();
		String token3 = UUID.randomUUID().toString();
		String token4 = UUID.randomUUID().toString();

		boolean token1Added = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, token1);
		assertThat(token1Added).isTrue();
		
		boolean token2Added = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, token2);
		assertThat(token2Added).isTrue();
		
		boolean token3Added = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, token3);
		assertThat(token3Added).isTrue();
		
		boolean token4Added = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, token4);
		assertThat(token4Added).isTrue();
		
		Thread.sleep(2000);

		List<String> savedTokens = firebaseMessagingService.getTokens(PLAYER_LOGIN_ID);
		assertThat(savedTokens).isNotNull();
		assertThat(savedTokens).isNotEmpty();
		assertThat(savedTokens).hasSize(4);

		assertThat(savedTokens.get(0)).isEqualTo(token4);
		assertThat(savedTokens.get(1)).isEqualTo(token3);
		assertThat(savedTokens.get(2)).isEqualTo(token2);
		assertThat(savedTokens.get(3)).isEqualTo(token1);		
	}

	@Test
	public void testDeleteToken(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		boolean tokenRemoved = firebaseMessagingService.deleteToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenRemoved).isTrue();
	}

	@Test
	public void testSendDataOnlyMessage(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		boolean messageSent = firebaseMessagingService.sendDataOnlyMessage(PLAYER_LOGIN_ID,
				FIREBASE_MESSAGING_MESSAGE_DATA
		);
		assertThat(messageSent).isTrue();
	}

	@Test
	public void testSendNotificationOnlyMessage(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		boolean messageSent = firebaseMessagingService.sendNotificationOnlyMessage(PLAYER_LOGIN_ID,
				FIREBASE_MESSAGING_MESSAGE_TITLE, FIREBASE_MESSAGING_MESSAGE_BODY);
		assertThat(messageSent).isTrue();
	}

	@Test
	public void testSendNotificationWithDataMessage(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		boolean messageSent = firebaseMessagingService.sendNotificationWithDataMessage(PLAYER_LOGIN_ID,
				FIREBASE_MESSAGING_MESSAGE_TITLE, FIREBASE_MESSAGING_MESSAGE_BODY, FIREBASE_MESSAGING_MESSAGE_DATA);
		assertThat(messageSent).isTrue();
	}

	@Test
	public void testSendMessageWithInvalidToken(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_INVALID);
		assertThat(tokenAdded).isTrue();

		boolean messageSent = firebaseMessagingService.sendDataOnlyMessage(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_MESSAGE_DATA);
		assertThat(messageSent).isFalse();
	}

	@Test
	public void testSendMessageWithExpiredToken(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_EXPIRED);
		assertThat(tokenAdded).isTrue();

		boolean messageSent = firebaseMessagingService.sendDataOnlyMessage(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_MESSAGE_DATA);
		assertThat(messageSent).isFalse();
	}

	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> indexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {

		assertThat(indexAware.getServices()).hasSize(1);
		CommonIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();

		indexService.resetIndex();

		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(PlayerApiPackage.Literals.PLAYER_PROFILE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
