/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.notifications.omnichannel.service.impl;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.api.Notification;
import com.playertour.backend.api.NotificationMetadataAttribute;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.notifications.email.service.api.EmailService;
import com.playertour.backend.notifications.firebasemessaging.service.api.FirebaseMessagingDataTriggeredActions;
import com.playertour.backend.notifications.firebasemessaging.service.api.FirebaseMessagingService;
import com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService;
import com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService;
import com.playertour.backend.notifications.templates.service.api.NotificationTemplate;
import com.playertour.backend.notifications.templates.service.api.NotificationTemplateService;

@Component(name = "OmniChannelNotificationService", scope = ServiceScope.PROTOTYPE)
public class OmniChannelNotificationServiceImpl implements OmniChannelNotificationService {

	@Reference(service = LoggerFactory.class)
	private Logger logger;
	
	@Reference
	private NotificationTemplateService notificationTemplateService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PersistentNotificationService notificationService;
	
	@Reference
	private FirebaseMessagingService firebaseMessagingService;	
	
	@Reference(cardinality = ReferenceCardinality.OPTIONAL) // optional for now
	private EmailService emailService;
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendWelcomeNotification(java.lang.String, java.lang.String)
	 */
	@Override
	public void sendWelcomeNotification(String notificationId, String loginId, String playerName) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");

		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.WELCOME_MSG, 
					Map.of(
							"player_name", playerName
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.WELCOME_MSG + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.WELCOME_MSG + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendGolfCourseEnteredNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendGolfCourseEnteredNotification(String notificationId, String loginId, String golfcourseId, String golfcourseName) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseId, "Golf course ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.GOLFCOURSE_ENTERED,
					Map.of(
							"golfcourse_name", golfcourseName
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			// turns out maps are not supported by Gecko Search ('org.gecko.search.util.DocumentUtil') - only lists - hence the workaround
			//notification.getMetadata().put(FirebaseMessagingService.MESSAGE_DATA_KEY_GOLFCOURSE_ID, golfcourseId); // this does not work when hits Gecko Search ('org.gecko.search.util.DocumentUtil')
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_GOLFCOURSE_ID, golfcourseId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.GOLFCOURSE_ENTERED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_GOLFCOURSE_ID, golfcourseId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.GOLFCOURSE_ENTERED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}		
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRoundOpenedNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRoundOpenedNotification(String notificationId, String loginId, String golfcourseName) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.ROUND_OPENED,
					Map.of(
							"golfcourse_name", golfcourseName
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.ROUND_OPENED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.ROUND_OPENED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}		
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRemindRoundOpenedNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRemindRoundOpenedNotification(String notificationId, String loginId, String golfcourseName) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.REMIND_ROUND_OPENED,
					Map.of(
							"golfcourse_name", golfcourseName
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.REMIND_ROUND_OPENED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.REMIND_ROUND_OPENED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRoundClosedNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRoundClosedNotification(String notificationId, String loginId, String golfcourseName, String closureDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		Objects.requireNonNull(closureDate, "Closure date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.ROUND_CLOSED,
					Map.of(
							"golfcourse_name", golfcourseName, 
							"closure_date", closureDate
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(
					FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED, 
					FirebaseMessagingDataTriggeredActions.STATISTICS_RESULTS_GET_REQUESTED,
					FirebaseMessagingDataTriggeredActions.LEADERBOARD_RESULTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.ROUND_CLOSED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.ROUND_CLOSED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRoundAutoClosedNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRoundAutoClosedNotification(String notificationId, String loginId, String golfcourseName,
			String closureDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		Objects.requireNonNull(closureDate, "Closure date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.ROUND_AUTO_CLOSED,
					Map.of(
							"golfcourse_name", golfcourseName, 
							"closure_date", closureDate
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(
					FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED,
					FirebaseMessagingDataTriggeredActions.STATISTICS_RESULTS_GET_REQUESTED,
					FirebaseMessagingDataTriggeredActions.LEADERBOARD_RESULTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.ROUND_AUTO_CLOSED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.ROUND_AUTO_CLOSED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRoundAutoForceClosedNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRoundAutoForceClosedNotification(String notificationId, String loginId, String golfcourseName,
			String closureDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		Objects.requireNonNull(closureDate, "Closure date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.ROUND_AUTO_FORCE_CLOSED,
					Map.of(
							"golfcourse_name", golfcourseName, 
							"closure_date", closureDate
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(
					FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED,
					FirebaseMessagingDataTriggeredActions.STATISTICS_RESULTS_GET_REQUESTED,
					FirebaseMessagingDataTriggeredActions.LEADERBOARD_RESULTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.ROUND_AUTO_FORCE_CLOSED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.ROUND_AUTO_FORCE_CLOSED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRoundCancelledNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRoundCancelledNotification(String notificationId, String loginId, String golfcourseName) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.ROUND_CANCELLED,
					Map.of(
							"golfcourse_name", golfcourseName
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.ROUND_CANCELLED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.ROUND_CANCELLED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}		
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendRoundAutoCancelledNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendRoundAutoCancelledNotification(String notificationId, String loginId, String golfcourseName) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(golfcourseName, "Golf course name is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.ROUND_AUTO_CANCELLED,
					Map.of(
							"golfcourse_name", golfcourseName
						),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.ROUND_AUTO_CANCELLED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.ROUND_AUTO_CANCELLED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendSuccessfulPurchaseNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendSuccessfulPurchaseNotification(String notificationId, String loginId, String meritpointsAmount) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(meritpointsAmount, "Merit points amount is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.SUCCESSFUL_PURCHASE, 
					Map.of(
							"meritpoints_amount", meritpointsAmount
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED, 
					FirebaseMessagingDataTriggeredActions.PLAYERPROFILE_MERITPOINTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.SUCCESSFUL_PURCHASE + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible; in addition, get latest total for merit points
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.SUCCESSFUL_PURCHASE + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendBonusReceivedNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendBonusReceivedNotification(String notificationId, String loginId, String meritpointsAmount) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(meritpointsAmount, "Merit points amount is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.BONUS_RECEIVED, 
					Map.of(
							"meritpoints_amount", meritpointsAmount
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED, 
					FirebaseMessagingDataTriggeredActions.PLAYERPROFILE_MERITPOINTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.BONUS_RECEIVED + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible; in addition, get latest total for merit points
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.BONUS_RECEIVED + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}		
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeIssuedSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeIssuedSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeId, "Challenge ID is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_ISSUED_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_ISSUED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_ISSUED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeIssuedOtherNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeIssuedOtherNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_ISSUED_OTHER, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_ISSUED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_ISSUED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeAcceptedSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeAcceptedSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_ACCEPTED_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_ACCEPTED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_ACCEPTED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeAcceptedOtherNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeAcceptedOtherNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_ACCEPTED_OTHER, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_ACCEPTED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_ACCEPTED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeDeclinedSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeDeclinedSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_DECLINED_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_DECLINED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_DECLINED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeDeclinedOtherNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeDeclinedOtherNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_DECLINED_OTHER, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_DECLINED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_DECLINED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeClosedSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeClosedSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_CLOSED_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_CLOSED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_CLOSED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeClosedOtherNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeClosedOtherNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_CLOSED_OTHER, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_CLOSED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_CLOSED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteOtherNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteOtherNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_OTHER, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteAcceptedSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteAcceptedSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteAcceptedOtherForNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteAcceptedOtherForNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_OTHER_FOR, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_OTHER_FOR + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_OTHER_FOR + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteAcceptedOtherAgainstNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteAcceptedOtherAgainstNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_OTHER_AGAINST, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_OTHER_AGAINST + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_ACCEPTED_OTHER_AGAINST + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteDeclinedSelfNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteDeclinedSelfNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_DECLINED_SELF, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_DECLINED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_DECLINED_SELF + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeInviteDeclinedOtherNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeInviteDeclinedOtherNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_INVITE_DECLINED_OTHER, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_INVITE_DECLINED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_INVITE_DECLINED_OTHER + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}	

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeWonNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeWonNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_WON, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED, 
					FirebaseMessagingDataTriggeredActions.PLAYERPROFILE_MERITPOINTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_WON + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_WON + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendChallengeLostNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendChallengeLostNotification(String notificationId, String loginId, String playerName, String challengeId, String challengeDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(challengeDate, "Challenge date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.CHALLENGE_LOST, 
					Map.of(
							"player_name", playerName, 
							"challenge_date", challengeDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			notification.getMetadata().add(createNotificationMetadataAttribute(FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId));
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED, 
					FirebaseMessagingDataTriggeredActions.PLAYERPROFILE_MERITPOINTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.CHALLENGE_LOST + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp()),
							FirebaseMessagingService.MESSAGE_DATA_KEY_CHALLENGE_ID, challengeId
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.CHALLENGE_LOST + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}
		
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendTournamentAnnouncementNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendTournamentAnnouncementNotification(String notificationId, String loginId, String playerName,
			String tournamentName, String tournamentPlace, String tournamentDate, String tournamentSite) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(tournamentName, "Tournament name is required!");
		Objects.requireNonNull(tournamentPlace, "Tournament place is required!");
		Objects.requireNonNull(tournamentDate, "Tournament date is required!");
		Objects.requireNonNull(tournamentSite, "Tournament site is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.TOURNAMENT_ANNOUNCEMENT, 
					Map.of(
							"player_name", playerName, 
							"tournament_name", tournamentName,
							"tournament_place", tournamentPlace, 
							"tournament_date", tournamentDate, 
							"tournament_site", tournamentSite
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.TOURNAMENT_ANNOUNCEMENT + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.TOURNAMENT_ANNOUNCEMENT + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendScheduledMaintenanceNotification(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendScheduledMaintenanceNotification(String notificationId, String loginId, String playerName, String maintenanceDate) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(playerName, "Player name is required!");
		Objects.requireNonNull(maintenanceDate, "Maintenance date is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {

			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.SCHEDULED_MAINTENANCE, 
					Map.of(
							"player_name", playerName, 
							"maintenance_date", maintenanceDate
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED);
			
			logger.info("Sending omni-channel (" + NotificationTemplate.SCHEDULED_MAINTENANCE + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.SCHEDULED_MAINTENANCE + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService#sendLeaderboardRankUpNotification(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void sendLeaderboardRankUpNotification(String notificationId, String loginId, String rankNumber) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		Objects.requireNonNull(rankNumber, "Rank number is required!");
		
		if (!notificationSentAlready(notificationId, loginId)) {
			
			// @formatter:off
			Notification notification = notificationTemplateService.getNotificationByTemplate(
					NotificationTemplate.LEADERBOARD_RANK_UP, 
					Map.of(
							"rank_number", rankNumber
					),
					Optional.of(Locale.ENGLISH)
			);
			// @formatter:on
			
			setAdditionalNotificationAttributes(notification, notificationId);
			
			String triggeredActions = constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions.NOTIFICATIONS_LIST_REQUESTED, 
					FirebaseMessagingDataTriggeredActions.PLAYERPROFILE_MERITPOINTS_GET_REQUESTED);

			logger.info("Sending omni-channel (" + NotificationTemplate.LEADERBOARD_RANK_UP + ") notification ID " + notificationId + " to player login ID: " + loginId);
			
			// @formatter:off
			sendOmniChannelNotification(loginId, 
					notification, 
					Map.of( 
							FirebaseMessagingService.MESSAGE_DATA_KEY_TRIGGERED_ACTIONS, triggeredActions, // pass additional metadata to FCM notification: trigger refresh of notifications list in client app, so both persistent and FCM notifications are made visible; in addition, get latest total for merit points
							FirebaseMessagingService.MESSAGE_DATA_KEY_TIMESTAMP, String.valueOf(notification.getTimeStamp())
						)
					);
			// @formatter:on
			
		} else {
			logger.info("Already sent (" + NotificationTemplate.LEADERBOARD_RANK_UP + ") notification ID " + notificationId + " to player login ID: " + loginId);
		}		
	}	
	
	private void sendOmniChannelNotification(String loginId, Notification notification, Map<String, String> data) {
		// create persistent notification - this will be presented to user in front-end app
		@SuppressWarnings("unused")
		Notification savedNotification = sendPersistentNotification(loginId, notification);

		// send notification via Firebase Messaging with metadata - if user has any tokens registered already and they are valid / not expired
		@SuppressWarnings("unused")
		boolean fcmNotificationSent = sendFCMNotification(loginId, notification, data);

		// TODO: send notification via email ...
	}
	
	@SuppressWarnings("unused")
	private void sendOmniChannelNotification(String loginId, Notification notification) {
		// create persistent notification - this will be presented to user in front-end app
		@SuppressWarnings("unused")
		Notification savedNotification = sendPersistentNotification(loginId, notification);

		// send notification via Firebase Messaging - if user has any tokens registered already and they are valid / not expired
		@SuppressWarnings("unused")
		boolean fcmNotificationSent = sendFCMNotification(loginId, notification);

		// TODO: send notification via email ...
	}	

	private Notification sendPersistentNotification(String loginId, Notification notification) {
		logger.debug("Sending persistent notification to player login ID: " + loginId);

		return notificationService.createNotification(notification, loginId);
	}

	private boolean sendFCMNotification(String loginId, Notification notification) {
		logger.debug("Sending FCM notification to player login ID: " + loginId);

		return firebaseMessagingService.sendNotificationOnlyMessage(loginId, notification.getTitle(),
				notification.getMessage());
	}

	private boolean sendFCMNotification(String loginId, Notification notification, Map<String, String> data) {
		logger.debug("Sending FCM notification with data to player login ID: " + loginId);
		return firebaseMessagingService.sendNotificationWithDataMessage(loginId, notification.getTitle(),
				notification.getMessage(), data);
	}
	
	// it was observed Event Admin sometimes publishes same event more than once, which results in handlers being called more than once .. 
	// so this is a workaround to ensure that for same event ID (which becomes a notification ID in this context) event is processed once only
	private boolean notificationSentAlready(String notificationId, String loginId) {
		return notificationService.notificationExists(notificationId, loginId);
	}
	
	private void setAdditionalNotificationAttributes(Notification notification, String notificationId) {
		notification.setId(notificationId);
		notification.setTimeStamp(LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toInstant().toEpochMilli());
	}

	private String constructTriggeredActionsList(FirebaseMessagingDataTriggeredActions... action) {
		// @formatter:off
		return Arrays
				.toString(Arrays.stream(action)
						.map(a -> a.getInternalName())
						.collect(Collectors.toList())
						.toArray());
		// @formatter:on
	}
	
	private NotificationMetadataAttribute createNotificationMetadataAttribute(String attributeKey,
			String attributeValue) {
		NotificationMetadataAttribute metadataAttr = PlayerApiFactory.eINSTANCE.createNotificationMetadataAttribute();
		metadataAttr.setKey(attributeKey);
		metadataAttr.setValue(attributeValue);
		return metadataAttr;
	}
}
