/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.notifications.omnichannel.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.Notification;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.notifications.firebasemessaging.service.api.FirebaseMessagingService;
import com.playertour.backend.notifications.omnichannel.service.api.OmniChannelNotificationService;
import com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService;
import com.playertour.backend.player.model.player.PlayerPackage;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class OmniChannelNotificationServiceIntegrationTest {

	private static final String PLAYER_NAME = "Test Player";
	private static final String PLAYER_USER_NAME = "testuser";
	private static final String PLAYER_LOGIN_ID = "1ce9d204-50b8-4f8a-8270-5222db24514d";

	private static final String FIREBASE_MESSAGING_TOKEN_VALID = "cIA93p_XTOWwGCASZcFiIq:APA91bGHTp7Rdbx7_I0d2xWewg7PCrp6YzLnXuiTwLKrz8SWXomTPQwOwFNqLqBxBOUQXopP92c50iMMKmBryjT_3tcFvmonbyJn16-53vNq0gzkeYNu6dEgct-SpCCe8sm0PEDm2Tj3";

	@InjectBundleContext
	BundleContext bundleContext;

	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		ServiceReference<OmniChannelNotificationService> firebaseMessagingServiceReference = omniChannelNotificationServiceAware
				.getServiceReference();
		assertThat(firebaseMessagingServiceReference).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		ServiceReference<PersistentNotificationService> notificationRef = notificationAware.getServiceReference();
		assertThat(notificationRef).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerRef = playerServiceAware.getServiceReference();
		assertThat(playerRef).isNotNull();
	}
	
	@Test
	public void testSendWelcomeNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendWelcomeNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendGolfCourseEnteredNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseId = UUID.randomUUID().toString();
		String golfcourseName = "Test Golf Course name";
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendGolfCourseEnteredNotification(notificationId, PLAYER_LOGIN_ID, golfcourseId, golfcourseName);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
		
	@Test
	public void testSendRoundOpenedNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRoundOpenedNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}
	
	@Test
	public void testSendRemindRoundOpenedNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRemindRoundOpenedNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}	
	
	@Test
	public void testSendRoundClosedNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		String closureDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRoundClosedNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName, closureDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendRoundAutoClosedNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		String closureDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRoundAutoClosedNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName, closureDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendRoundAutoForceClosedNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		String closureDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRoundAutoForceClosedNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName, closureDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}	
			
	@Test
	public void testSendRoundCancelledNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRoundCancelledNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}
	
	@Test
	public void testSendRoundAutoCancelledNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String golfcourseName = "Test Golf Course name";
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendRoundAutoCancelledNotification(notificationId, PLAYER_LOGIN_ID, golfcourseName);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}	
	
	@Test
	public void testSendBonusReceivedNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		int meritpointsAmount = 1000;
		
		String notificationId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendBonusReceivedNotification(notificationId, PLAYER_LOGIN_ID, String.valueOf(meritpointsAmount));
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendSuccessfulPurchaseNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		int meritpointsAmount = 1000;
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendSuccessfulPurchaseNotification(notificationId, PLAYER_LOGIN_ID, String.valueOf(meritpointsAmount));
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendChallengeIssuedSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendChallengeIssuedSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendChallengeIssuedOtherNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeIssuedOtherNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}
	
	@Test
	public void testSendChallengeAcceptedSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeAcceptedSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation			
	}
	
	@Test
	public void testSendChallengeAcceptedOtherNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendChallengeAcceptedOtherNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendChallengeDeclinedSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeDeclinedSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}
	
	@Test
	public void testSendChallengeDeclinedOtherNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeDeclinedOtherNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}
	
	@Test
	public void testSendChallengeClosedSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeClosedSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation			
	}
	
	@Test
	public void testSendChallengeClosedOtherNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeClosedOtherNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeInviteSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation			
	}
	
	@Test
	public void testSendChallengeInviteOtherNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteOtherNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeInviteAcceptedSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteAcceptedSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeInviteAcceptedOtherForNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteAcceptedOtherForNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeInviteAcceptedOtherAgainstNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteAcceptedOtherAgainstNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeInviteDeclinedSelfNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteDeclinedSelfNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeInviteDeclinedOtherNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {
		
		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendChallengeInviteDeclinedOtherNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation	
	}
	
	@Test
	public void testSendChallengeWonNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendChallengeWonNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendChallengeLostNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String challengeDate = LocalDateTime.now().toString();
		
		String notificationId = UUID.randomUUID().toString();
		
		String challengeId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendChallengeLostNotification(notificationId, PLAYER_LOGIN_ID, PLAYER_NAME, challengeId, challengeDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendTournamentAnnouncementNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String tournamentName = "Test tournament name";
		String tournamentPlace = "Test tournament place";
		String tournamentDate = LocalDateTime.now().plusWeeks(3).toString();
		String tournamentSite = "Test tournament site";
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendTournamentAnnouncementNotification(notificationId, PLAYER_LOGIN_ID,
				PLAYER_NAME, tournamentName, tournamentPlace, tournamentDate, tournamentSite);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendScheduledMaintenanceNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();

		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		String maintenanceDate = LocalDateTime.now().plusHours(3).toString();
		
		String notificationId = UUID.randomUUID().toString();

		omniChannelNotificationService.sendScheduledMaintenanceNotification(notificationId, PLAYER_LOGIN_ID,
				PLAYER_NAME, maintenanceDate);
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation
	}
	
	@Test
	public void testSendLeaderboardRankUpNotification(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<OmniChannelNotificationService> omniChannelNotificationServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> notificationAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<FirebaseMessagingService> firebaseMessagingServiceAware,
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware) throws InterruptedException {

		assertThat(omniChannelNotificationServiceAware.getServices()).hasSize(1);
		OmniChannelNotificationService omniChannelNotificationService = omniChannelNotificationServiceAware
				.getService();
		assertThat(omniChannelNotificationService).isNotNull();

		assertThat(notificationAware.getServices()).hasSize(1);
		PersistentNotificationService notificationService = notificationAware.getService();
		assertThat(notificationService).isNotNull();

		assertThat(firebaseMessagingServiceAware.getServices()).hasSize(1);
		FirebaseMessagingService firebaseMessagingService = firebaseMessagingServiceAware.getService();
		assertThat(firebaseMessagingService).isNotNull();

		assertThat(playerServiceAware.getServices()).hasSize(1);
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		PlayerProfile profile = playerService.getPlayerProfile(PLAYER_LOGIN_ID, PLAYER_USER_NAME);
		assertThat(profile).isNotNull();
		
		Thread.sleep(2000);

		boolean tokenAdded = firebaseMessagingService.saveToken(PLAYER_LOGIN_ID, FIREBASE_MESSAGING_TOKEN_VALID);
		assertThat(tokenAdded).isTrue();

		int rankNumber = 3;
		
		String notificationId = UUID.randomUUID().toString();
		
		omniChannelNotificationService.sendLeaderboardRankUpNotification(notificationId, PLAYER_LOGIN_ID, String.valueOf(rankNumber));
		
		Thread.sleep(2000);

		List<Notification> result = notificationService.getNotificationByUser(PLAYER_LOGIN_ID);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(2); // two as there's an additional persistent notification created on player profile creation		
	}
	
	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> indexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {

		assertThat(indexAware.getServices()).hasSize(1);
		CommonIndexService indexService = indexAware.getService();
		assertThat(indexService).isNotNull();

		indexService.resetIndex();

		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(PlayerApiPackage.Literals.PLAYER_PROFILE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}
}
