/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.notifications.persistent.service.impl;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.pushstream.PushStream;

import com.playertour.backend.api.Notification;
import com.playertour.backend.api.NotificationSearchResults;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.apis.player.exceptions.PlayerUsernameValidationException;
import com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService;
import com.playertour.backend.player.model.player.Player;

@Component(name = "PersistentNotificationService", scope = ServiceScope.PROTOTYPE)
public class PersistentNotificationServiceImpl implements PersistentNotificationService {

	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerService playerService;

	@Reference(service=LoggerFactory.class)
	private Logger logger;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#getNotificationByUser(java.lang.String)
	 */
	@Override
	public List<Notification> getNotificationByUser(String loginId) {

		PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(loginId);
		if(playerProfile != null) {
			return playerProfile.getNotifications().stream().filter(n->!n.isDeleted()).collect(Collectors.toList());
		}
		return Collections.emptyList();
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#getNotificationById(java.lang.String, java.lang.String)
	 */
	@Override
	public Notification getNotificationById(String notificationId, String loginId) {

		if(notificationId == null) {
			throw new IllegalArgumentException("Cannot retrieve Notification with null id!");
		}
		List<Notification> notifications = getNotificationByUser(loginId);
		Notification notification = notifications.stream().filter(n->notificationId.equals(n.getId())).findFirst().orElse(null);
		return notification;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#markNotificationAsRead(java.lang.String, java.lang.String)
	 */
	@Override
	public Notification markNotificationAsRead(String notificationId, String loginId) {

		if(notificationId == null) {
			throw new IllegalArgumentException("Cannot mark Notification with null id!");
		}
		PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(loginId);
		if(playerProfile != null) {
			List<Notification> notifications = playerProfile.getNotifications();
			Notification notification = notifications.stream().filter(n->notificationId.equals(n.getId())).findFirst().orElse(null);
			if(notification != null) {
				notification.setRead(true);
				try {
					playerService.saveProfile(playerProfile, loginId);
				} catch (PlayerUsernameValidationException e) {
					logger.error(e.getMessage());
				}
				return EcoreUtil.copy(notification);
			}
		}
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#deleteNotification(java.lang.String, java.lang.String)
	 */
	@Override
	public Notification deleteNotification(String notificationId, String loginId) {

		if(notificationId == null) {
			throw new IllegalArgumentException("Cannot delete Notification with null id!");
		}
		PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(loginId);
		if(playerProfile != null) {
			List<Notification> notifications = playerProfile.getNotifications();
			Notification notification = notifications.stream().filter(n->notificationId.equals(n.getId())).findFirst().orElse(null);
			if(notification != null) {
				notification.setDeleted(true);
				try {
					playerService.saveProfile(playerProfile, loginId);
				} catch (PlayerUsernameValidationException e) {
					logger.error(e.getMessage());
				}
				return EcoreUtil.copy(notification);
			}
		}
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#saveNotificationForAllPlayers(com.playertour.backend.api.Notification)
	 */
	@Override
	public Promise<Void> saveNotificationForAllPlayers(Notification notification) {
		PushStream<EObject> playersPushStream = playerService.getPlayers();
		return playersPushStream
				.map(eo -> (Player) eo)
				.forEach(p -> {
					saveNotificationForPlayer(notification, p);
				}).onFailure(t -> t.printStackTrace())
				.thenAccept(result -> logger.info("Notification has been saved for all Players!"));
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#saveNotificationForPlayer(com.playertour.backend.api.Notification, com.playertour.backend.player.model.player.Player)
	 */
	@Override
	public void saveNotificationForPlayer(Notification notification, Player player) {
		if(notification.getId() == null) {
			notification.setId(UUID.randomUUID().toString());
		}
		PlayerProfile playerProfile = player.getProfile();
		if(playerProfile != null) {
			List<Notification> notifications = playerProfile.getNotifications();
			Notification existingNotification = notifications.stream()
					.filter(n->notification.getId().equals(n.getId()))
					.findFirst().orElse(null);
			if(existingNotification == null) {
				playerProfile.getNotifications().add(notification);
				try {
					playerService.saveProfile(playerProfile, player.getLoginId());
				} catch (PlayerUsernameValidationException e) {
					logger.error(e.getMessage());
				}
			}
		}		
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#createNotification(com.playertour.backend.api.Notification, java.lang.String)
	 */
	@Override
	public Notification createNotification(Notification notification, String loginId) {

		PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(loginId);
		if (playerProfile != null) {
			if(notification.getId() == null) {
				notification.setId(UUID.randomUUID().toString());
			}
			playerProfile.getNotifications().add(notification);
			try {
				playerService.saveProfile(playerProfile, loginId);
			} catch (PlayerUsernameValidationException e) {
				logger.error(e.getMessage());
			}
			return EcoreUtil.copy(notification);
		}
		return null;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService#createNotification(com.playertour.backend.api.Notification, java.lang.String[])
	 */
	@Override
	public Notification createNotification(Notification notification, String... loginIds) {
		if(notification.getId() == null) {
			notification.setId(UUID.randomUUID().toString());
		}
		
		for(String loginId : loginIds) {
			PlayerProfile playerProfile = playerService.getPlayerProfileByLoginId(loginId);
			if (playerProfile != null) {				
				playerProfile.getNotifications().add(EcoreUtil.copy(notification));
				try {
					playerService.saveProfile(playerProfile, loginId);
				} catch (PlayerUsernameValidationException e) {
					logger.error(e.getMessage());
				}
			}
		}
		
		return notification;
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.NotificationService#createNotificationSearchResults(java.util.List)
	 */
	@Override
	public NotificationSearchResults createNotificationSearchResults(List<Notification> notifications) {
		NotificationSearchResults results = PlayerApiFactory.eINSTANCE.createNotificationSearchResults();
		results.getResults().addAll(notifications);	
		return results;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService#notificationExists(java.lang.String, java.lang.String)
	 */
	@Override
	public boolean notificationExists(String notificationId, String loginId) {
		Objects.requireNonNull(notificationId, "Notification ID is required!");
		Objects.requireNonNull(loginId, "Player login ID is required!");
		
		return (getNotificationById(notificationId, loginId) != null);
	}
}
