/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.notifications.templates.service.impl;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import org.javatuples.Pair;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.api.Notification;
import com.playertour.backend.api.NotificationType;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.notifications.templates.service.api.NotificationTemplate;
import com.playertour.backend.notifications.templates.service.api.NotificationTemplateService;
import com.samskivert.mustache.Mustache;
import com.samskivert.mustache.Template;

/**
 * 
 * @author ilenia, mhs
 * @since May 12, 2021
 */
@Component(name = "NotificationTemplateService", scope = ServiceScope.SINGLETON)
public class NotificationTemplateServiceImpl implements NotificationTemplateService {
	static final String NOTIFICATION_TEMPLATES_HEADER = "X-Notification-Templates";
	static final String NOTIFICATION_TEMPLATES_EXTENSION = "mustache";
	
	// KEY: PAIR of: normalized_locale (Locale.toLanguageTag()) + template_name (NotificationTemplate.name())
	// VALUE: PAIR of: compiled template for notification title + compiled template for notification message
	private final Map<Pair<String, String>, Pair<Template, Template>> templateCache = new ConcurrentHashMap<Pair<String, String>, Pair<Template, Template>>();

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	private Mustache.Compiler mustacheCompiler = Mustache.compiler().defaultValue("N/A");

	private String notificationTemplatesPath;

	@Activate
	public void activate() {
		notificationTemplatesPath = getNotificationTemplatesPath(
				FrameworkUtil.getBundle(getClass()).getBundleContext());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.playertour.backend.apis.notification.TemplateNotificationService#getNotificationByTemplate(com.playertour.backend.apis.notification.NotificationTemplate, java.util.Map, java.util.Optional)
	 */
	@Override
	public Notification getNotificationByTemplate(NotificationTemplate notificationTemplate, Map<String, String> data,
			Optional<Locale> localeOptional) {
		Locale locale = getLocale(localeOptional);

		Pair<Template, Template> titleMessageTemplates = loadTitleMessageTemplates(notificationTemplate, locale);

		Notification notification = createSystemNotification(notificationTemplate.getLabel(),
				substituteVariables(titleMessageTemplates.getValue0(), data),
				substituteVariables(titleMessageTemplates.getValue1(), data));

		return notification;
	}
	
	private Locale getLocale(Optional<Locale> locale) {
		if (locale.isPresent()) {
			return locale.get();
		} else {
			return Locale.getDefault();
		}
	}
	
	private Pair<Template, Template> loadTitleMessageTemplates(NotificationTemplate notificationTemplate,
			Locale locale) {
		String normalizedLocale = getNormalizedLocale(locale);
		String notificationTemplateName = notificationTemplate.name();
		Pair<String, String> templateCacheKey = Pair.with(normalizedLocale, notificationTemplateName);

		synchronized (templateCache) {
			if (templateCache.containsKey(templateCacheKey)) {
				return templateCache.get(templateCacheKey);
			}

			Template titleTemplate = compileTitleTemplate(notificationTemplate.getTitle());
			Template messageTemplate = compileMessageTemplate(
					constructTemplateFilePath(constructTemplateFileName(notificationTemplate, locale)));

			if (titleTemplate != null && messageTemplate != null) {
				Pair<Template, Template> titleMessageTemplates = Pair.with(titleTemplate, messageTemplate);
				templateCache.put(templateCacheKey, titleMessageTemplates);
				return titleMessageTemplates;
			}
		}

		return null;
	}	

	private String substituteVariables(Template template, Map<String, String> data) {
		return template.execute(data);
	}

	private Notification createSystemNotification(String label, String title, String message) {
		Notification notification = PlayerApiFactory.eINSTANCE.createNotification();
		notification.setLabel(label);
		notification.setType(NotificationType.SYSTEM_CREATED);
		notification.setTitle(title);
		notification.setMessage(message);

		return notification;
	}

	private String getNotificationTemplatesPath(BundleContext ctx) {
		boolean isDefined = Collections.list(ctx.getBundle().getHeaders().keys()).stream()
				.anyMatch(key -> NOTIFICATION_TEMPLATES_HEADER.equalsIgnoreCase(key));
		if (!isDefined) {
			throw new IllegalStateException("Notification templates path must be defined!");
		}

		// OSGI-INF/l10n/templates
		String notificationTemplatesPath = ctx.getBundle().getHeaders().get(NOTIFICATION_TEMPLATES_HEADER);

		return notificationTemplatesPath;
	}

	private Template compileTitleTemplate(String titleTemplate) {
		return mustacheCompiler.compile(titleTemplate);
	}

	private Template compileMessageTemplate(Path messageTemplateFilePath) {
		Template messageTemplate = null;

		try (InputStreamReader isr = new InputStreamReader(
				FrameworkUtil.getBundle(getClass()).getEntry(messageTemplateFilePath.toString()).openStream())) {

			messageTemplate = mustacheCompiler.compile(isr);

		} catch (IOException e) {
			e.printStackTrace();
		}

		return messageTemplate;
	}

	private String getNormalizedLocale(Locale locale) {
		return locale.toLanguageTag();
	}

	private String constructTemplateFileName(NotificationTemplate notificationTemplate, Locale locale) {
		String normalizedLocale = getNormalizedLocale(locale);

		// OSGI-INF/l10n/templates/[notification_template_name]_[normalized_locale].mustache
		StringBuilder sb = new StringBuilder();
		sb.append(notificationTemplate.name().toLowerCase());
		sb.append(".");
		sb.append(normalizedLocale);
		sb.append(".");
		sb.append(NOTIFICATION_TEMPLATES_EXTENSION);

		return sb.toString();
	}

	private Path constructTemplateFilePath(String templateFileName) {
		return Paths.get(notificationTemplatesPath, templateFileName);
	}

	@Deprecated(forRemoval = true)
	private static final String MAINTENANCE_MSG = "Dear User, \n A scheduled maintenance is foreseen for <DATE_OF_MAINTENANCE>.\n"
			+ " The app could be temporarily unavailable for that time. We apologize for the inconvenience.\n \n"
			+ " Your PlayerTour Team";
				
	@Deprecated(forRemoval = true)
	private static final String TOURNAMENT_MSG = "Dear User, \n We are thrilled to announce the upcoming tournament "
			+ "<TOURNAMENT_NAME>, which will take place in <TOURNAMENT_PLACE> on <TOURNAMENT_DATE>.\n"
			+ " For further information, please visit the site <TORUNAMENT_SITE>. \n \n Your PlayerTour Team";
	
	@Deprecated(forRemoval = true)
	private static final String WELCOME_MSG = "Welcome to the PlayerTour!";	
	
	@Deprecated(forRemoval = true)
	private static final String OPEN_BET_MSG = "Player <LAUNCHER_LOGIN_NAME> has challenged you to a Bet with id <BET_ID>!";
	
	@Deprecated(forRemoval = true)
	private static final String ACCEPT_BET_MSG = "Player <PLAYER_LOGIN_NAME> has accepted the Bet!";
	
	@Deprecated(forRemoval = true)
	private static final String DECLINE_BET_MSG = "Player <PLAYER_LOGIN_NAME> has declined the Bet!";
	
	@Deprecated(forRemoval = true)
	private static final String INVITE_BET_MSG = "Player <PLAYER_LOGIN_NAME> has invited you to join the Bet with id <BET_ID>!";
	
	@Deprecated(forRemoval = true)
	private static final String INVITE_ACCEPT_WITH_BET_MSG = "Player <PLAYER_LOGIN_NAME> has accepted to join the Bet and is betting in favour of the Bet content!";
	
	@Deprecated(forRemoval = true)
	private static final String INVITE_ACCEPT_AGAINST_BET_MSG = "Player <PLAYER_LOGIN_NAME> has accepted to join the Bet and is betting against the Bet content!";

	@Deprecated(forRemoval = true)
	private static final String INVITE_DECLINE_BET_MSG = "Player <PLAYER_LOGIN_NAME> has declined the invitation to join the Bet!";
	
	@Deprecated(forRemoval = true)
	private static final String RESOLVE_BET_MSG = "Bet with id <BET_ID> has been resolved. \n Winners are <WINNERS_LOGIN_NAMES>. \n Losers are <LOSERS_LOGIN_NAMES>.";
	
	@Deprecated(forRemoval = true)
	private static final String CLOSE_BET_MSG = "Bet with id <BET_ID> has been closed by Player <PLAYER_LOGIN_NAME>";
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.notification.TemplateNotificationService#getNotificationByTemplate(java.lang.String)
	 */
	@Override
	@Deprecated(forRemoval = true)
	public Notification getNotificationByTemplate(String templateName) {
		
		Notification notification = PlayerApiFactory.eINSTANCE.createNotification();
		notification.setType(NotificationType.SYSTEM_CREATED);
		notification.setId(UUID.randomUUID().toString());
		
		switch(templateName) {
		case SCHEDULED_MAINTENANCE_TEMPLATE:
			notification.setMessage(MAINTENANCE_MSG);
			return notification;
		case TOURNAMENT_ANNOUNCEMENT_TEMPLATE:
			notification.setMessage(TOURNAMENT_MSG);
			return notification;
		case WELCOME_MSG_TEMPLATE:
			notification.setMessage(WELCOME_MSG);
			return notification;
		case OPEN_BET_TEMPLATE:
			notification.setMessage(OPEN_BET_MSG);
			return notification;
		case ACCEPT_BET_TEMPLATE:
			notification.setMessage(ACCEPT_BET_MSG);
			return notification;
		case DECLINE_BET_TEMPLATE:
			notification.setMessage(DECLINE_BET_MSG);
			return notification;
		case INVITE_BET_TEMPLATE:
			notification.setMessage(INVITE_BET_MSG);
			return notification;
		case INVITE_ACCEPT_WITH_BET_TEMPLATE:
			notification.setMessage(INVITE_ACCEPT_WITH_BET_MSG);
			return notification;
		case INVITE_ACCEPT_AGAINST_BET_TEMPLATE:
			notification.setMessage(INVITE_ACCEPT_AGAINST_BET_MSG);
			return notification;
		case INVITE_DECLINE_BET_TEMPLATE:
			notification.setMessage(INVITE_DECLINE_BET_MSG);
			return notification;
		case RESOLVE_BET_TEMPLATE:
			notification.setMessage(RESOLVE_BET_MSG);
			return notification;
		case CLOSE_BET_TEMPLATE:
			notification.setMessage(CLOSE_BET_MSG);
			return notification;
		default:
			throw new IllegalArgumentException("Current Notification template " + templateName + " is not yet supported!");
		}		
	}	
}
