/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player.scorecardtracker.service.impl;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ComponentPropertyType;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.service.metatype.annotations.Designate;
import org.osgi.service.metatype.annotations.ObjectClassDefinition;
import org.osgi.util.promise.PromiseFactory;

import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.api.PlayerScorecardTrackingStatus;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.player.model.player.PlayerPackage;
import com.playertour.backend.player.scorecardtracker.service.api.PlayerScorecardTrackerService;

@Component(name = "PlayerScorecardTrackerService", scope = ServiceScope.SINGLETON, immediate = true, configurationPolicy = ConfigurationPolicy.REQUIRE, configurationPid = "PlayerScorecardTrackerService")
@Designate(ocd = PlayerScorecardTrackerServiceImpl.Config.class)
public class PlayerScorecardTrackerServiceImpl implements PlayerScorecardTrackerService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;	
	
	@Reference(service=LoggerFactory.class)
	private Logger logger;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerScorecardService playerScorecardService;
	
	private PromiseFactory factory = new PromiseFactory(Executors.newSingleThreadExecutor(),
			Executors.newSingleThreadScheduledExecutor());
	
	private AtomicInteger counter = new AtomicInteger();
	
	private ScheduledFuture<?> scheduledWithFixedDelay;
	
	/********************************/
	/** Externalized configuration **/	
	private Config config;
	
	@ComponentPropertyType
	@ObjectClassDefinition
	public @interface Config {
		
		boolean autoStart() default true;
		
		// the time to delay first execution
		long trackerInitialDelay();
		
		// how often tracker is run
		long trackerFrequency();
		
		// the time unit of the 'trackerInitialDelay' and 'trackerFrequency' parameters
		String trackerTimeUnit();
		
		// the time since scorecard is opened after which tracking starts
		long scorecardInitialDelay();
		
		// the time since reminder about opened scorecard is sent after which it is automatically closed or cancelled
		long scorecardAfterRemindedDelay();
		
		// the time unit of the 'scorecardInitialDelay' and 'scorecardAfterRemindedDelay' parameters
		String scorecardTimeUnit();
	}
	
	@Activate
	public void activate(Config config) {
		this.config = config;
		
		if (this.config.autoStart()) {
			logger.info("Starting player scorecard tracker...");
			
			// @formatter:off
			scheduledWithFixedDelay = factory.scheduledExecutor().scheduleWithFixedDelay(
					() -> factory.executor().execute(
							() -> trackOpenPlayerScorecards()), 
					this.config.trackerInitialDelay(), 
					this.config.trackerFrequency(), 
					TimeUnit.valueOf(this.config.trackerTimeUnit()));
			// @formatter:on
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.player.scorecardtracker.service.api.PlayerScorecardTrackerService#trackOpenPlayerScorecards()
	 */
	@Override
	public void trackOpenPlayerScorecards() {
		String currentDateTime = LocalDateTime.now().toString();
		int currentCount = counter.addAndGet(1);

		logger.debug("Tracking open player scorecards: run number " + currentCount + " at: " + currentDateTime);

		List<Player> playersWithOpenScorecards = getPlayersWithOpenScorecards();
		if (!playersWithOpenScorecards.isEmpty()) {

			for (Player playerWithOpenScorecards : playersWithOpenScorecards) {

				List<PlayerScorecard> openScorecards = getOpenScorecards(playerWithOpenScorecards);

				for (PlayerScorecard openScorecard : openScorecards) {

					Date nowDateTime = new Date();

					long scorecardOpenedElapsedTime = TimeUnit.valueOf(config.scorecardTimeUnit()).convert(
							Math.abs(nowDateTime.getTime() - openScorecard.getStart().getTime()),
							TimeUnit.MILLISECONDS);

					if ((openScorecard.getTrackingStatus() == null || (openScorecard.getTrackingStatus() != null
							&& openScorecard.getTrackingStatus() == PlayerScorecardTrackingStatus.DEFAULT))
							&& (scorecardOpenedElapsedTime >= config.scorecardInitialDelay())) {

						// remind user round is still opened
						playerScorecardService.remindRoundOpened(playerWithOpenScorecards.getLoginId(),
								openScorecard.getId());

					} else if (openScorecard.getTrackingStatus() != null
							&& openScorecard.getTrackingStatus() == PlayerScorecardTrackingStatus.REMINDER_SENT) {

						long scorecardTrackingStatusUpdateElapsedTime = TimeUnit.valueOf(config.scorecardTimeUnit())
								.convert(
										Math.abs(nowDateTime.getTime()
												- openScorecard.getTrackingStatusUpdateTimeStamp().getTime()),
										TimeUnit.MILLISECONDS);

						if (scorecardTrackingStatusUpdateElapsedTime >= config.scorecardAfterRemindedDelay()) {

							// if scorecard is complete, automatically close it
							if (playerScorecardService.isScorecardComplete(playerWithOpenScorecards.getLoginId(),
									openScorecard.getId())) {
								playerScorecardService.autoCloseScorecard(playerWithOpenScorecards.getLoginId(),
										openScorecard.getId());

							} else {
								// if scorecard is not complete and user is below NR limit, cancel scorecard and
								// mark it as no return
								if (playerScorecardService.allowNRCancel(playerWithOpenScorecards.getLoginId())) {
									playerScorecardService.autoCancelScorecard(playerWithOpenScorecards.getLoginId(),
											openScorecard.getId());

									// if scorecard is not complete and user is over NR limit, force close scorecard,
									// i.e. set every remaining hole at max amount (par per hole + max over par) and close it
								} else {
									playerScorecardService.autoForceCloseScorecard(
											playerWithOpenScorecards.getLoginId(), openScorecard.getId());
								}
							}
						}
					}
				}

				openScorecards = Collections.emptyList();
			}
		}

		playersWithOpenScorecards = Collections.emptyList();
	}
	
	private List<Player> getPlayersWithOpenScorecards() {
		List<Player> allPlayers;

		EMFRepository repository = repositoryServiceObjects.getService();

		try {

			allPlayers = repository.getAllEObjects(PlayerPackage.Literals.PLAYER);

			if (!allPlayers.isEmpty()) {
				return allPlayers.stream().filter(p -> hasOpenScorecard(p)).collect(Collectors.toList());
			} else {
				return Collections.emptyList();
			}

		} finally {
			allPlayers = Collections.emptyList();
			repositoryServiceObjects.ungetService(repository);
		}
	}
	
	private boolean hasOpenScorecard(Player player) {
		if (player.getPlayerScorecards() != null && 
				!player.getPlayerScorecards().isEmpty()) {

			// @formatter:off
			return !player.getPlayerScorecards().stream()
					.filter(s-> (s.getEnd() == null && s.isNoReturn() == false))
					.collect(Collectors.toList())
					.isEmpty();
			// @formatter:on
		}

		return false;
	}

	public List<PlayerScorecard> getOpenScorecards(Player player) {
		// @formatter:off
		return player.getPlayerScorecards().stream()
				.filter(s-> (s.getEnd() == null && s.isNoReturn() == false))
				.collect(Collectors.toList());
		// @formatter:on
	}
	
	@Deactivate
	public void deactivate() {
		if (scheduledWithFixedDelay != null
				&& (scheduledWithFixedDelay.isDone() || scheduledWithFixedDelay.isCancelled())) {
			scheduledWithFixedDelay.cancel(true);
		}
	}
}
