/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player.scorecardtracker.service.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.List;
import java.util.UUID;

import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.gecko.search.util.CommonIndexService;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.MethodOrderer;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.platform.commons.annotation.Testable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.LandingZone;
import com.playertour.backend.api.Notification;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.api.Stroke;
import com.playertour.backend.apis.course.CourseService;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.apis.player.exceptions.PlayerUsernameValidationException;
import com.playertour.backend.golfcourse.model.golfcourse.CourseDetails;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecardInfo;
import com.playertour.backend.golfcourse.model.golfcourse.CourseScorecards;
import com.playertour.backend.golfcourse.model.golfcourse.GenderType;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCourseFactory;
import com.playertour.backend.golfcourse.model.golfcourse.GolfCoursePackage;
import com.playertour.backend.golfcourse.model.golfcourse.Tee;
import com.playertour.backend.notifications.persistent.service.api.PersistentNotificationService;
import com.playertour.backend.player.model.player.PlayerPackage;
import com.playertour.backend.player.scorecardtracker.service.api.PlayerScorecardTrackerService;

@Testable
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
public class PlayerScorecardTrackerIntegrationTest {

	@InjectBundleContext
	BundleContext bundleContext;
	
	@Order(value = -1)
	@Test
	public void testServices(
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardTrackerService> playerScorecardTrackerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> persistentNotificationServiceAware) {

		assertThat(playerScorecardTrackerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerScorecardTrackerService> playerScorecardTrackerServiceReference = playerScorecardTrackerServiceAware
				.getServiceReference();
		assertThat(playerScorecardTrackerServiceReference).isNotNull();
		
		assertThat(playerServiceAware.getServices()).hasSize(1);
		ServiceReference<PlayerService> playerServiceReference = playerServiceAware.getServiceReference();
		assertThat(playerServiceReference).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		ServiceReference<PlayerScorecardService> playerScorecardServiceReference = playerScorecardServiceAware.getServiceReference();
		assertThat(playerScorecardServiceReference).isNotNull();
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		ServiceReference<CourseService> courseServiceReference = courseServiceAware.getServiceReference();
		assertThat(courseServiceReference).isNotNull();
		
		assertThat(persistentNotificationServiceAware.getServices()).hasSize(1);
		ServiceReference<PersistentNotificationService> persistentNotificationServiceReference = persistentNotificationServiceAware.getServiceReference();
		assertThat(persistentNotificationServiceReference).isNotNull();
	}
	
	// open scorecard and do not close it - verify reminder was sent that round is still opened
	@Disabled("Disabled until Jenkins is fixed; see issues from 2022/10/26 and 2022/10/27")
	@Test
	public void testRemindRoundOpened(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardTrackerService> playerScorecardTrackerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware,  
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> persistentNotificationServiceAware) throws InterruptedException, PlayerUsernameValidationException {
		
		assertThat(playerScorecardTrackerServiceAware.getServices()).hasSize(1);	
		PlayerScorecardTrackerService playerScorecardTrackerService = playerScorecardTrackerServiceAware.getService();
		assertThat(playerScorecardTrackerService).isNotNull();	
		
		assertThat(playerServiceAware.getServices()).hasSize(1);	
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		assertThat(persistentNotificationServiceAware.getServices()).hasSize(1);
		PersistentNotificationService persistentNotificationService = persistentNotificationServiceAware.getService();
		assertThat(persistentNotificationService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		String loginName = "engelbert.humperdinck";
		GenderType gender = GenderType.MEN;
		
		PlayerProfile profileInitial = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileInitial).isNotNull();
		
		com.playertour.backend.golfcourse.model.golfcourse.GolfCourse golfCourse = constructGolfCourse(gender);
		String courseId = golfCourse.getCourseId();
		int holesTotal = golfCourse.getCourseDetails().getHoleNum();
				
		golfCourse = courseService.saveCourse(golfCourse);
		assertThat(golfCourse).isNotNull();
		
		Thread.sleep(2000);
		
		PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
		assertThat(openedPlayerScorecard).isNotNull();
		assertThat(openedPlayerScorecard.getStart()).isNotNull();
		assertThat(openedPlayerScorecard.getEnd()).isNull();
		assertThat(openedPlayerScorecard.getCourseScorecardInfo()).isNotNull();
		assertThat(openedPlayerScorecard.getCourseId()).isEqualTo(courseId);
		assertThat(openedPlayerScorecard.getHoleStats()).isNotNull();
		assertThat(openedPlayerScorecard.getHoleStats()).isNotEmpty();
		assertThat(openedPlayerScorecard.getHoleStats()).hasSize(holesTotal);
		
		String playerScorecardId = openedPlayerScorecard.getId();
		assertThat(playerScorecardId).isNotNull();
				
		Thread.sleep(2000);
		
		PlayerScorecard updatedPlayerScorecard = generateStrokes(loginId, playerScorecardId, openedPlayerScorecard, playerScorecardService, false);
		assertThat(updatedPlayerScorecard).isNotNull();
		
		Thread.sleep(2000);
		
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(2000);
		
		List<Notification> result = persistentNotificationService.getNotificationByUser(loginId);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(3);
	}	
	
	// open scorecard, fill it completely, and do not close it - verify it was closed automatically
	@Disabled("Disabled until Jenkins is fixed; see issues from 2022/10/26 and 2022/10/27")
	@Test
	public void testAutoCloseScorecard(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardTrackerService> playerScorecardTrackerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware,  
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> persistentNotificationServiceAware) throws InterruptedException, PlayerUsernameValidationException {
		
		assertThat(playerScorecardTrackerServiceAware.getServices()).hasSize(1);	
		PlayerScorecardTrackerService playerScorecardTrackerService = playerScorecardTrackerServiceAware.getService();
		assertThat(playerScorecardTrackerService).isNotNull();	
		
		assertThat(playerServiceAware.getServices()).hasSize(1);	
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		assertThat(persistentNotificationServiceAware.getServices()).hasSize(1);
		PersistentNotificationService persistentNotificationService = persistentNotificationServiceAware.getService();
		assertThat(persistentNotificationService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		String loginName = "engelbert.humperdinck";
		GenderType gender = GenderType.MEN;
		
		PlayerProfile profileInitial = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileInitial).isNotNull();
		
		int lbPointsInitial = profileInitial.getLeaderBoardPoints();
		assertThat(lbPointsInitial).isEqualTo(0);
		
		double ptaInitial = profileInitial.getPlayerTourAbility();
		assertThat(ptaInitial).isEqualTo(0);
		
		com.playertour.backend.golfcourse.model.golfcourse.GolfCourse golfCourse = constructGolfCourse(gender);
		String courseId = golfCourse.getCourseId();
		int holesTotal = golfCourse.getCourseDetails().getHoleNum();
				
		golfCourse = courseService.saveCourse(golfCourse);
		assertThat(golfCourse).isNotNull();
		
		Thread.sleep(2000);
		
		PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
		assertThat(openedPlayerScorecard).isNotNull();
		assertThat(openedPlayerScorecard.getStart()).isNotNull();
		assertThat(openedPlayerScorecard.getEnd()).isNull();
		assertThat(openedPlayerScorecard.getCourseScorecardInfo()).isNotNull();
		assertThat(openedPlayerScorecard.getCourseId()).isEqualTo(courseId);
		assertThat(openedPlayerScorecard.getHoleStats()).isNotNull();
		assertThat(openedPlayerScorecard.getHoleStats()).isNotEmpty();
		assertThat(openedPlayerScorecard.getHoleStats()).hasSize(holesTotal);
		
		String playerScorecardId = openedPlayerScorecard.getId();
		assertThat(playerScorecardId).isNotNull();
				
		Thread.sleep(2000);
		
		PlayerScorecard updatedPlayerScorecard = generateStrokes(loginId, playerScorecardId, openedPlayerScorecard, playerScorecardService, false);
		assertThat(updatedPlayerScorecard).isNotNull();
		
		Thread.sleep(2000);
		
		// first run - reminder will be sent
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(6500);
		
		// second run - scorecard will be closed automatically
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(2000);
		
		PlayerScorecard autoClosedPlayerScorecard = playerScorecardService.getScorecardById(loginId, playerScorecardId);
		assertThat(autoClosedPlayerScorecard).isNotNull();
		assertThat(autoClosedPlayerScorecard.getEnd()).isNotNull();
		assertThat(autoClosedPlayerScorecard.isNoReturn()).isFalse();

		List<Notification> result = persistentNotificationService.getNotificationByUser(loginId);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(4);
		
		PlayerProfile profileUpdated = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileUpdated).isNotNull();
		
		int lbPointsUpdated = profileUpdated.getLeaderBoardPoints();
		assertThat(lbPointsUpdated).isEqualTo(72);
		
		double ptaUpdated = profileUpdated.getPlayerTourAbility();
		assertThat(ptaUpdated).isEqualTo(-52);
	}	
	
	// open scorecard, fill it partially, and do not close it - verify reminder was sent that round is still opened, verify it was cancelled automatically
	@Disabled("Disabled until Jenkins is fixed; see issues from 2022/10/26 and 2022/10/27")
	@Test
	public void testAutoCancelScorecard(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardTrackerService> playerScorecardTrackerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware,  
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> persistentNotificationServiceAware) throws InterruptedException, PlayerUsernameValidationException {
		
		assertThat(playerScorecardTrackerServiceAware.getServices()).hasSize(1);	
		PlayerScorecardTrackerService playerScorecardTrackerService = playerScorecardTrackerServiceAware.getService();
		assertThat(playerScorecardTrackerService).isNotNull();	
		
		assertThat(playerServiceAware.getServices()).hasSize(1);	
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		assertThat(persistentNotificationServiceAware.getServices()).hasSize(1);
		PersistentNotificationService persistentNotificationService = persistentNotificationServiceAware.getService();
		assertThat(persistentNotificationService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		String loginName = "engelbert.humperdinck";
		GenderType gender = GenderType.MEN;
		
		PlayerProfile profileInitial = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileInitial).isNotNull();
		
		com.playertour.backend.golfcourse.model.golfcourse.GolfCourse golfCourse = constructGolfCourse(gender);
		String courseId = golfCourse.getCourseId();
		int holesTotal = golfCourse.getCourseDetails().getHoleNum();
				
		golfCourse = courseService.saveCourse(golfCourse);
		assertThat(golfCourse).isNotNull();
		
		Thread.sleep(2000);
		
		PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
		assertThat(openedPlayerScorecard).isNotNull();
		assertThat(openedPlayerScorecard.getStart()).isNotNull();
		assertThat(openedPlayerScorecard.getEnd()).isNull();
		assertThat(openedPlayerScorecard.getCourseScorecardInfo()).isNotNull();
		assertThat(openedPlayerScorecard.getCourseId()).isEqualTo(courseId);
		assertThat(openedPlayerScorecard.getHoleStats()).isNotNull();
		assertThat(openedPlayerScorecard.getHoleStats()).isNotEmpty();
		assertThat(openedPlayerScorecard.getHoleStats()).hasSize(holesTotal);
		
		String playerScorecardId = openedPlayerScorecard.getId();
		assertThat(playerScorecardId).isNotNull();
				
		Thread.sleep(2000);
		
		PlayerScorecard updatedPlayerScorecard = generateStrokes(loginId, playerScorecardId, openedPlayerScorecard, playerScorecardService, true);
		assertThat(updatedPlayerScorecard).isNotNull();
		
		Thread.sleep(2000);
		
		// first run - reminder will be sent
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(7500);
		
		// second run - scorecard will be cancelled automatically
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(2000);
		
		PlayerScorecard autoCancelledPlayerScorecard = playerScorecardService.getScorecardById(loginId, playerScorecardId);
		assertThat(autoCancelledPlayerScorecard).isNotNull();
		assertThat(autoCancelledPlayerScorecard.getEnd()).isNotNull();
		assertThat(autoCancelledPlayerScorecard.isNoReturn()).isTrue();

		List<Notification> result = persistentNotificationService.getNotificationByUser(loginId);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(4);
	}
	
	// open scorecard, cancel it - repeat three times; open scorecard, fill it partially, and do not close it - verify reminder was sent that round is still opened, verify it was force closed automatically as user is above "no return" limit
	@Disabled("Disabled until Jenkins is fixed; see issues from 2022/10/26 and 2022/10/27")
	@Test
	public void testAutoForceCloseScorecard(@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardTrackerService> playerScorecardTrackerServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerService> playerServiceAware,  
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PlayerScorecardService> playerScorecardServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<CourseService> courseServiceAware, 
			@InjectService(cardinality = 1, timeout = 5000) ServiceAware<PersistentNotificationService> persistentNotificationServiceAware) throws InterruptedException, PlayerUsernameValidationException {
		
		assertThat(playerScorecardTrackerServiceAware.getServices()).hasSize(1);	
		PlayerScorecardTrackerService playerScorecardTrackerService = playerScorecardTrackerServiceAware.getService();
		assertThat(playerScorecardTrackerService).isNotNull();	
		
		assertThat(playerServiceAware.getServices()).hasSize(1);	
		PlayerService playerService = playerServiceAware.getService();
		assertThat(playerService).isNotNull();
		
		assertThat(playerScorecardServiceAware.getServices()).hasSize(1);	
		PlayerScorecardService playerScorecardService = playerScorecardServiceAware.getService();
		assertThat(playerScorecardService).isNotNull();	
		
		assertThat(courseServiceAware.getServices()).hasSize(1);	
		CourseService courseService = courseServiceAware.getService();
		assertThat(courseService).isNotNull();
		
		assertThat(persistentNotificationServiceAware.getServices()).hasSize(1);
		PersistentNotificationService persistentNotificationService = persistentNotificationServiceAware.getService();
		assertThat(persistentNotificationService).isNotNull();
		
		String loginId = UUID.randomUUID().toString();
		String loginName = "engelbert.humperdinck";
		GenderType gender = GenderType.MEN;
		
		PlayerProfile profileInitial = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileInitial).isNotNull();
		
		int lbPointsInitial = profileInitial.getLeaderBoardPoints();
		assertThat(lbPointsInitial).isEqualTo(0);
		
		double ptaInitial = profileInitial.getPlayerTourAbility();
		assertThat(ptaInitial).isEqualTo(0);		
		
		com.playertour.backend.golfcourse.model.golfcourse.GolfCourse golfCourse = constructGolfCourse(gender);
		String courseId = golfCourse.getCourseId();
		int holesTotal = golfCourse.getCourseDetails().getHoleNum();
				
		golfCourse = courseService.saveCourse(golfCourse);
		assertThat(golfCourse).isNotNull();
		
		Thread.sleep(2000);
		
		// #1
		{
			PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
			assertThat(openedPlayerScorecard).isNotNull();
			
			String playerScorecardId = openedPlayerScorecard.getId();
			assertThat(playerScorecardId).isNotNull();

			Thread.sleep(2000);
			
			boolean allowNRCancel = playerScorecardService.allowNRCancel(loginId);
			assertThat(allowNRCancel).isTrue();
			
			PlayerScorecard cancelledPlayerScorecard = playerScorecardService.cancelScorecard(loginId, playerScorecardId);
			assertThat(cancelledPlayerScorecard).isNotNull();
			assertThat(cancelledPlayerScorecard.getEnd()).isNotNull();
			assertThat(cancelledPlayerScorecard.isNoReturn()).isTrue();
		}
		
		// #2
		{
			PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
			assertThat(openedPlayerScorecard).isNotNull();
			
			String playerScorecardId = openedPlayerScorecard.getId();
			assertThat(playerScorecardId).isNotNull();

			Thread.sleep(2000);
			
			boolean allowNRCancel = playerScorecardService.allowNRCancel(loginId);
			assertThat(allowNRCancel).isTrue();
			
			PlayerScorecard cancelledPlayerScorecard = playerScorecardService.cancelScorecard(loginId, playerScorecardId);
			assertThat(cancelledPlayerScorecard).isNotNull();
			assertThat(cancelledPlayerScorecard.getEnd()).isNotNull();
			assertThat(cancelledPlayerScorecard.isNoReturn()).isTrue();
		}
		
		// #3
		{
			PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
			assertThat(openedPlayerScorecard).isNotNull();
			
			String playerScorecardId = openedPlayerScorecard.getId();
			assertThat(playerScorecardId).isNotNull();

			Thread.sleep(2000);
			
			boolean allowNRCancel = playerScorecardService.allowNRCancel(loginId);
			assertThat(allowNRCancel).isTrue();
			
			PlayerScorecard cancelledPlayerScorecard = playerScorecardService.cancelScorecard(loginId, playerScorecardId);
			assertThat(cancelledPlayerScorecard).isNotNull();
			assertThat(cancelledPlayerScorecard.getEnd()).isNotNull();
			assertThat(cancelledPlayerScorecard.isNoReturn()).isTrue();
		}		
		
		PlayerScorecard openedPlayerScorecard = playerScorecardService.openScorecard(loginId, courseId, gender);
		assertThat(openedPlayerScorecard).isNotNull();
		assertThat(openedPlayerScorecard.getStart()).isNotNull();
		assertThat(openedPlayerScorecard.getEnd()).isNull();
		assertThat(openedPlayerScorecard.getCourseScorecardInfo()).isNotNull();
		assertThat(openedPlayerScorecard.getCourseId()).isEqualTo(courseId);
		assertThat(openedPlayerScorecard.getHoleStats()).isNotNull();
		assertThat(openedPlayerScorecard.getHoleStats()).isNotEmpty();
		assertThat(openedPlayerScorecard.getHoleStats()).hasSize(holesTotal);
		
		String playerScorecardId = openedPlayerScorecard.getId();
		assertThat(playerScorecardId).isNotNull();
				
		Thread.sleep(2000);
		
		PlayerScorecard updatedPlayerScorecard = generateStrokes(loginId, playerScorecardId, openedPlayerScorecard, playerScorecardService, true);
		assertThat(updatedPlayerScorecard).isNotNull();
		
		Thread.sleep(2000);
		
		boolean allowNRCancel = playerScorecardService.allowNRCancel(loginId);
		assertThat(allowNRCancel).isFalse();
		
		// first run - reminder will be sent
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(7500);
		
		// second run - scorecard will be force closed automatically
		playerScorecardTrackerService.trackOpenPlayerScorecards();
		
		Thread.sleep(2000);
		
		PlayerScorecard autoForceClosedPlayerScorecard = playerScorecardService.getScorecardById(loginId, playerScorecardId);
		assertThat(autoForceClosedPlayerScorecard).isNotNull();
		assertThat(autoForceClosedPlayerScorecard.getEnd()).isNotNull();
		assertThat(autoForceClosedPlayerScorecard.isNoReturn()).isFalse();

		List<Notification> result = persistentNotificationService.getNotificationByUser(loginId);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
	//	assertThat(result).hasSize(10);
		
		PlayerProfile profileUpdated = playerService.getPlayerProfile(loginId, loginName);			
		assertThat(profileUpdated).isNotNull();
		
		int lbPointsUpdated = profileUpdated.getLeaderBoardPoints();
		assertThat(lbPointsUpdated).isEqualTo(72);
		
		double ptaUpdated = profileUpdated.getPlayerTourAbility();
		assertThat(ptaUpdated).isEqualTo(-20);
	}
	
	private com.playertour.backend.golfcourse.model.golfcourse.GolfCourse constructGolfCourse(GenderType gender) {
		String courseId = UUID.randomUUID().toString();
		
		CourseScorecardInfo scorecardInfo = GolfCourseFactory.eINSTANCE.createCourseScorecardInfo();
		scorecardInfo.setParIn(37);
		scorecardInfo.setParOut(35);
		scorecardInfo.setParTotal(72);
		scorecardInfo.getHcpHole().addAll(List.of(9, 11, 15, 1, 3, 13, 17, 5, 7, 6, 14, 18, 4, 8, 16, 2, 10, 12));
		scorecardInfo.getParHole().addAll(List.of(4, 4, 3, 5, 4, 4, 3, 4, 4, 5, 5, 4, 4, 4, 3, 5, 3, 4));
		
		CourseScorecards courseScorecards = GolfCourseFactory.eINSTANCE.createCourseScorecards();
		courseScorecards.getMenScorecard().add(scorecardInfo);
	//	courseScorecards.getWmnScorecard().add(scorecardInfo); // FIXME: adding 'CourseScorecardInfo' to 'wmnScorecard' list somehow erases 'CourseScorecardInfo' from 'menScorecard' list
				
		com.playertour.backend.golfcourse.model.golfcourse.GolfCourse golfCourse = GolfCourseFactory.eINSTANCE.createGolfCourse();
		golfCourse.setScorecards(courseScorecards);
		golfCourse.setCourseId(courseId);
		golfCourse.setId(courseId);
		
		int holesTotal = 18;
		
		Tee tee = GolfCourseFactory.eINSTANCE.createTee();
		tee.setColor("Yellow");
		tee.setGender(gender);
		golfCourse.getTee().add(tee);
		
		CourseDetails courseDetails = GolfCourseFactory.eINSTANCE.createCourseDetails();		
		courseDetails.setCourseName("Golfclub Gera e.V.");
		courseDetails.setHoleNum(holesTotal);
		
		golfCourse.setCourseDetails(courseDetails);
		
		return golfCourse;
	}
	
	private PlayerScorecard generateStrokes(String loginId, String playerScorecardId, PlayerScorecard playerScorecard, PlayerScorecardService playerScorecardService, boolean skipHoles) throws InterruptedException {
		int holesTotal = playerScorecard.getHoleStats().size();
		
		for (int i = 0; i < holesTotal; i++) {
			int holeNumber = (i + 1);
			
			if (skipHoles && (holeNumber % 3 == 0)) {
				continue;
			}
			
			boolean hasExtraFirstStroke = false;
			
			if (holeNumber % 9 == 0) {
				Stroke stroke = generateStroke(holeNumber, LandingZone.ROUGH);
				
				playerScorecard = playerScorecardService.submitStroke(loginId, playerScorecardId, stroke);
				assertThat(playerScorecard).isNotNull();
				assertThat(playerScorecard.getHoleStats()).isNotEmpty();
				
				hasExtraFirstStroke = true;
				
				Thread.sleep(250);
			}
			
			LandingZone landingZone;
			if (holeNumber % 2 == 0) {
				landingZone = LandingZone.FAIRWAY;
			} else {
				landingZone = LandingZone.GREEN;
			}
			
			Stroke stroke = generateStroke(holeNumber, landingZone);
			
			playerScorecard = playerScorecardService.submitStroke(loginId, playerScorecardId, stroke);
			assertThat(playerScorecard).isNotNull();
			assertThat(playerScorecard.getHoleStats()).isNotEmpty();
			
			HoleScore holeScore = playerScorecard.getHoleStats().get(i);
			assertThat(holeScore.getStrokes()).isNotEmpty();
			assertThat(holeScore.getStrokes().get(0).getHoleNumber()).isEqualTo(holeNumber);
			if (hasExtraFirstStroke) {				
				assertThat(holeScore.getStrokes()).hasSize(2);
				assertThat(holeScore.getStrokes().get(0).getZone()).isEqualTo(LandingZone.ROUGH);
				assertThat(holeScore.getStrokes().get(1).getZone()).isEqualTo(landingZone);
				
				assertThat(holeScore.getScore()).isEqualTo(2);
				assertThat(holeScore.getPTScore()).isEqualTo(2);				
			} else {
				assertThat(holeScore.getStrokes()).hasSize(1);
				assertThat(holeScore.getStrokes().get(0).getZone()).isEqualTo(landingZone);
				
				assertThat(holeScore.getScore()).isEqualTo(1);
				assertThat(holeScore.getPTScore()).isEqualTo(1);				
			}
			
			Thread.sleep(250);
		}
		
		return playerScorecard;
	}
	
	private Stroke generateStroke(int holeNumber, LandingZone landingZone) {
		Stroke stroke = PlayerApiFactory.eINSTANCE.createStroke();
		stroke.setHoleNumber(holeNumber);
		stroke.setZone(landingZone);
		return stroke;
	}
	
	@BeforeEach
	@AfterEach
	public void clean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=CourseIndexService)") ServiceAware<CommonIndexService> courseIndexAware,
			@InjectService(cardinality = 1, timeout = 5000, filter = "(component.name=PlayerIndexService)") ServiceAware<CommonIndexService> playerIndexAware, 
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(playerIndexAware.getServices()).hasSize(1);	
		CommonIndexService playerIndexService = playerIndexAware.getService();
		assertThat(playerIndexService).isNotNull();		
		
		playerIndexService.resetIndex();
		
		assertThat(courseIndexAware.getServices()).hasSize(1);	
		CommonIndexService courseIndexService = courseIndexAware.getService();
		assertThat(courseIndexService).isNotNull();		
		
		courseIndexService.resetIndex();
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(PlayerApiPackage.Literals.GOLF_COURSE.getName()).drop();
			database.getCollection(GolfCoursePackage.Literals.GOLF_COURSE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}

	@AfterAll
	public static void brutalClean(
			@InjectService(cardinality = 1, timeout = 5000, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		assertThat(dbProviderAware.getServices()).hasSize(1);
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();

		MongoDatabase database = dbProvider.getDatabase();
		database.drop();
	}	
}
