/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.List;
import java.util.UUID;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.GolfCourse;
import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerApiPackage;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.player.model.player.PlayerPackage;


/**
 * Tests the implementation of the PlayerService
 * @since 1.0
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class PlayerIntegrationTest  {
	
	@InjectBundleContext
	BundleContext bundleContext;
	

	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) {
				
		assertThat(playerAware.getServices()).hasSize(1);	
		ServiceReference<PlayerService> playerRef = playerAware.getServiceReference();
		assertThat(playerRef).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		ServiceReference<EMFRepository> repoRef = repoAware.getServiceReference();
		assertThat(repoRef).isNotNull();	
	}
	
	@Test
	public void testGetProfile(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) {
				
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		String userId = UUID.randomUUID().toString();
		PlayerProfile profile = playerService.getPlayerProfile(userId);
		
		assertThat(profile).isNotNull();
		
		List<Player> players = repo.getAllEObjects(PlayerPackage.Literals.PLAYER);
		assertThat(players).isNotNull();
		assertThat(players).isNotEmpty();
		assertThat(players).hasSize(1);
		
		Player player = players.get(0);
		assertThat(player).isNotNull();
		assertThat(player.getLoginId()).isEqualTo(userId);
		
		String playerId = player.getId();
		assertThat(playerId).isNotNull();
		
		profile = playerService.getPlayerProfile(userId);
		assertThat(profile).isNotNull();
		
		players = repo.getAllEObjects(PlayerPackage.Literals.PLAYER);
		assertThat(players).isNotNull();
		assertThat(players).isNotEmpty();
		assertThat(players).hasSize(1);
		
		player = players.get(0);
		assertThat(player).isNotNull();
		assertThat(player.getLoginId()).isEqualTo(userId);
		assertThat(player.getId()).isNotNull();
		assertThat(player.getId()).isEqualTo(playerId);
	}
	
	
	@Test
	public void testSaveProfile(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) {
				
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		String userId = UUID.randomUUID().toString();
		PlayerProfile profile = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile.setActive(true);
		profile.setCity("Jena");
		profile.setCountry("Germany");
		assertThat(profile.getId()).isNull();
		
		PlayerProfile savedProfile = playerService.saveProfile(profile, userId);
		assertThat(savedProfile).isNotNull();
		assertThat(savedProfile.getCity()).isEqualTo("Jena");
		assertThat(savedProfile.getCountry()).isEqualTo("Germany");
		assertThat(savedProfile.getId()).isNotNull();
		
		List<Player> players = repo.getAllEObjects(PlayerPackage.Literals.PLAYER);
		assertThat(players).isNotNull();
		assertThat(players).isNotEmpty();
		assertThat(players).hasSize(1);
		
		Player player = players.get(0);
		assertThat(player).isNotNull();
		assertThat(player.getLoginId()).isEqualTo(userId);
		assertThat(player.getProfile()).isNotNull();
		assertThat(player.getProfile().getCity()).isEqualTo("Jena");
		assertThat(player.getProfile().getCountry()).isEqualTo("Germany");
		assertThat(player.getProfile().getId()).isEqualTo(savedProfile.getId());
		
		profile = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile.setActive(true);
		profile.setCity("Belluno");
		profile.setCountry("Italy");
		
		savedProfile = playerService.saveProfile(profile, userId);
		assertThat(savedProfile).isNotNull();
		assertThat(savedProfile.getCity()).isEqualTo("Belluno");
		assertThat(savedProfile.getCountry()).isEqualTo("Italy");
		assertThat(savedProfile.getId()).isNotNull();
		
		players = repo.getAllEObjects(PlayerPackage.Literals.PLAYER);
		assertThat(players).isNotNull();
		assertThat(players).isNotEmpty();
		assertThat(players).hasSize(1);
		
		player = players.get(0);
		assertThat(player).isNotNull();
		assertThat(player.getLoginId()).isEqualTo(userId);
		assertThat(player.getProfile()).isNotNull();
		assertThat(player.getProfile().getCity()).isEqualTo("Belluno");
		assertThat(player.getProfile().getCountry()).isEqualTo("Italy");
		assertThat(player.getProfile().getId()).isEqualTo(savedProfile.getId());
	}
	
	@Test
	public void testVisitedCourses(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws InterruptedException {
				
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		String userId = UUID.randomUUID().toString();
		PlayerProfile profile = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		profile.setActive(true);
		profile.setCity("Jena");
		profile.setCountry("Germany");
		GolfCourse gc = PlayerApiFactory.eINSTANCE.createGolfCourse();
		gc.setName("Jena Golf Club");
		profile.getGolfCourseHistory().add(gc);
		assertThat(profile.getId()).isNull();
		
		playerService.saveProfile(profile, userId);
		
		Thread.sleep(2000);
		
		List<GolfCourseSearchResult> result = playerService.visitedGolfCourses(userId);
		assertThat(result).isNotNull();
		assertThat(result).isNotEmpty();
		assertThat(result).hasSize(1);
		
		GolfCourseSearchResult res = result.get(0);
		assertThat(res.getCourse()).isNotNull();
		assertThat(res.getCourse().getName()).isEqualTo("Jena Golf Club");		
	}
	
	@AfterEach
	public void doAfterEach(@InjectService(cardinality = 1, timeout = 500, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
			database.getCollection(PlayerApiPackage.Literals.GOLF_COURSE.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}
}
