/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player.tests;

import static org.assertj.core.api.Assertions.assertThat;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.gecko.emf.repository.EMFRepository;
import org.gecko.mongo.osgi.MongoDatabaseProvider;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.test.common.annotation.InjectBundleContext;
import org.osgi.test.common.annotation.InjectService;
import org.osgi.test.common.service.ServiceAware;
import org.osgi.test.junit5.context.BundleContextExtension;
import org.osgi.test.junit5.service.ServiceExtension;

import com.mongodb.client.MongoDatabase;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.api.ScoreCard;
import com.playertour.backend.apis.player.APIScorecardService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.player.model.player.PlayerPackage;


/**
 * Tests the implementation of the APIScorecardService
 * @since 1.0
 */
@ExtendWith(BundleContextExtension.class)
@ExtendWith(ServiceExtension.class)
public class ScorecardIntegrationTest  {
	
	@InjectBundleContext
	BundleContext bundleContext;
	

	@Test
	public void testServices(@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIScorecardService> scorecardAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) {
				
		assertThat(scorecardAware.getServices()).hasSize(1);	
		ServiceReference<APIScorecardService> scorecardRef = scorecardAware.getServiceReference();
		assertThat(scorecardRef).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		ServiceReference<EMFRepository> repoRef = repoAware.getServiceReference();
		assertThat(repoRef).isNotNull();	
	}
	
	@Test
	public void testSaveScorecard(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIScorecardService> scorecardAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws InterruptedException {
			
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		APIScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		String userId = UUID.randomUUID().toString();
		PlayerProfile profile = playerService.getPlayerProfile(userId);
		
		Thread.sleep(2000);
		
		assertThat(profile).isNotNull();
		
		ScoreCard scorecard = PlayerApiFactory.eINSTANCE.createScoreCard();
		scorecard.setCourseId("abcdef");
		Date start = new Date();
		scorecard.setStart(start);
		
		scorecard = scorecardService.saveScorecard(userId, scorecard);
		assertThat(scorecard).isNotNull();
		
		Thread.sleep(2000);
		
		List<Player> players = repo.getAllEObjects(PlayerPackage.Literals.PLAYER);
		assertThat(players).isNotNull();
		assertThat(players).isNotEmpty();
		assertThat(players).hasSize(1);
		
		Player player = players.get(0);
				
		List<ScoreCard> scorecards = player.getScoreCards();
		assertThat(scorecards).isNotNull();
		assertThat(scorecards).isNotEmpty();
		assertThat(scorecards).hasSize(1);
		
		ScoreCard savedScorecard = scorecards.get(0);
		assertThat(savedScorecard.getId()).isNotNull();
		assertThat(savedScorecard.getCourseId()).isEqualTo("abcdef");
		assertThat(savedScorecard.getStart()).isEqualTo(start);		
	}
	
	@Test
	public void testGetScorecardsByUser(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIScorecardService> scorecardAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws InterruptedException {
			
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		APIScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		String userId = UUID.randomUUID().toString();
		PlayerProfile profile = playerService.getPlayerProfile(userId);
		
		Thread.sleep(2000);
		
		assertThat(profile).isNotNull();
		
		ScoreCard scorecard = PlayerApiFactory.eINSTANCE.createScoreCard();
		scorecard.setCourseId("abcdef");
		Date start = new Date();
		scorecard.setStart(start);
		
		scorecard = scorecardService.saveScorecard(userId, scorecard);
		assertThat(scorecard).isNotNull();
		
		Thread.sleep(2000);
				
		List<ScoreCard> scorecards = scorecardService.getScorecardsByUser(userId);
		assertThat(scorecards).isNotNull();
		assertThat(scorecards).isNotEmpty();
		assertThat(scorecards).hasSize(1);
		
		ScoreCard savedScorecard = scorecards.get(0);
		assertThat(savedScorecard.getId()).isNotNull();
		assertThat(savedScorecard.getCourseId()).isEqualTo("abcdef");
		assertThat(savedScorecard.getStart()).isEqualTo(start);		
	}
	
	
	@Test
	public void testGetScorecardsByCourse(@InjectService(cardinality = 1, timeout = 500) ServiceAware<PlayerService> playerAware, 
			@InjectService(cardinality = 1, timeout = 500) ServiceAware<APIScorecardService> scorecardAware, 
			@InjectService(cardinality = 1, timeout = 500, filter = "(repo_id=playertour.playertour)") ServiceAware<EMFRepository> repoAware) throws InterruptedException {
			
		assertThat(playerAware.getServices()).hasSize(1);	
		PlayerService playerService = playerAware.getService();
		assertThat(playerService).isNotNull();	
		
		assertThat(scorecardAware.getServices()).hasSize(1);	
		APIScorecardService scorecardService = scorecardAware.getService();
		assertThat(scorecardService).isNotNull();	
		
		assertThat(repoAware.getServices()).hasSize(1);	
		EMFRepository repo = repoAware.getService();
		assertThat(repo).isNotNull();	
		
		String userId = UUID.randomUUID().toString();
		PlayerProfile profile = playerService.getPlayerProfile(userId);
		
		Thread.sleep(2000);
		
		assertThat(profile).isNotNull();
		
		ScoreCard scorecard1 = PlayerApiFactory.eINSTANCE.createScoreCard();
		scorecard1.setCourseId("abcdef");
		Date start = new Date();
		scorecard1.setStart(start);
		
		scorecard1 = scorecardService.saveScorecard(userId, scorecard1);
		assertThat(scorecard1).isNotNull();
		
		Thread.sleep(2000);
		
		ScoreCard scorecard2 = PlayerApiFactory.eINSTANCE.createScoreCard();
		scorecard2.setCourseId("fedcba");
		Date start2 = new Date();
		scorecard2.setStart(start2);
		
		scorecard2 = scorecardService.saveScorecard(userId, scorecard2);
		assertThat(scorecard2).isNotNull();
		
		Thread.sleep(2000);
				
		List<ScoreCard> scorecards = scorecardService.getScorecardsByCourse(userId, "abcdef");
		assertThat(scorecards).isNotNull();
		assertThat(scorecards).isNotEmpty();
		assertThat(scorecards).hasSize(1);
		
		ScoreCard savedScorecard = scorecards.get(0);
		assertThat(savedScorecard.getId()).isNotNull();
		assertThat(savedScorecard.getCourseId()).isEqualTo("abcdef");
		assertThat(savedScorecard.getStart()).isEqualTo(start);		
		
		
		scorecards = scorecardService.getScorecardsByCourse(userId, "fedcba");
		assertThat(scorecards).isNotNull();
		assertThat(scorecards).isNotEmpty();
		assertThat(scorecards).hasSize(1);
		
		savedScorecard = scorecards.get(0);
		assertThat(savedScorecard.getId()).isNotNull();
		assertThat(savedScorecard.getCourseId()).isEqualTo("fedcba");
		assertThat(savedScorecard.getStart()).isEqualTo(start2);		
	}
	
	
	
	
	
	@AfterEach
	public void doAfterEach(@InjectService(cardinality = 1, timeout = 500, filter = "(database=playertour)") ServiceAware<MongoDatabaseProvider> dbProviderAware) {
		
		assertThat(dbProviderAware.getServices()).hasSize(1);	
		MongoDatabaseProvider dbProvider = dbProviderAware.getService();
		assertThat(dbProvider).isNotNull();	
		
		MongoDatabase database = dbProvider.getDatabase();
		try {
			database.getCollection(PlayerPackage.Literals.PLAYER.getName()).drop();
		} catch (IllegalArgumentException e) {
			System.out.println("Collection does not exist. Nothing to remove.");
		}
	}
}
