/**
 * Copyright (c) 2012 - 2018 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.api.IndexActionType;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.api.GolfCourse;
import com.playertour.backend.api.GolfCourseSearchResult;
import com.playertour.backend.api.PlayerApiFactory;
import com.playertour.backend.api.PlayerProfile;
import com.playertour.backend.apis.player.PlayerIndexService;
import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.apis.player.PlayerService;
import com.playertour.backend.player.model.player.Player;
import com.playertour.backend.player.model.player.PlayerFactory;

@Component(scope = ServiceScope.PROTOTYPE)
public class PlayerServiceImpl implements PlayerService{
	
	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	EMFRepository repository;
	
	@Reference
	PlayerIndexService playerIndexService;

	@Reference
	PlayerSearchService playerSearchService;

	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.PlayerService#getPlayerProfile(java.lang.String)
	 */
	@Override
	public PlayerProfile getPlayerProfile(String userId) {
		if(userId == null) {
			throw new IllegalArgumentException("Cannot get PlayerProfile for user with null login id!");
		}
		Player player = playerSearchService.getPlayerByLoginId(userId);
		if(player != null) {
			return (PlayerProfile) player.getProfile();
		}
		Player newPlayer = PlayerFactory.eINSTANCE.createPlayer();
		newPlayer.setLoginId(userId);
		PlayerProfile newProfile = PlayerApiFactory.eINSTANCE.createPlayerProfile();
		newPlayer.setProfile(newProfile);
		repository.attach(newPlayer);
		repository.save(newPlayer);
		playerIndexService.indexPlayer(newPlayer, IndexActionType.ADD);
		return newProfile;
	}

	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.PlayerService#saveProfile(com.playertour.backend.api.PlayerProfile, java.lang.String)
	 */
	@Override
	public PlayerProfile saveProfile(PlayerProfile profile, String userId) {
		if(userId == null) {
			throw new IllegalArgumentException("Cannot save PlayerProfile for user with null login id!");
		}
		Player player = playerSearchService.getPlayerByLoginId(userId);
		boolean isFirstSave = false;
		if(player == null) {
			isFirstSave = true;
			player = PlayerFactory.eINSTANCE.createPlayer();
			player.setLoginId(userId);
			repository.attach(player);
		}
		player.setProfile(profile);
		repository.save(player);
		playerIndexService.indexPlayer(player, isFirstSave ? IndexActionType.ADD : IndexActionType.MODIFY);
		return profile;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.PlayerService#visitedGolfCourses(java.lang.String)
	 */
	@Override
	public List<GolfCourseSearchResult> visitedGolfCourses(String userId) {
		if(userId == null) {
			throw new IllegalArgumentException("Cannot retrieve visited GolfCourses for user with null login id!");
		}
		List<GolfCourseSearchResult> result = new LinkedList<>();
		Player player = playerSearchService.getPlayerByLoginId(userId);
		if(player != null && player.getProfile() != null) {
			List<GolfCourse> golfCourses = player.getProfile().getGolfCourseHistory();
			golfCourses.stream().forEach(gc -> {
				GolfCourseSearchResult res = PlayerApiFactory.eINSTANCE.createGolfCourseSearchResult();
				res.setCourse(EcoreUtil.copy(gc));
				result.add(res);
			});
		}
		return result;
	}
}
