/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.api.IndexActionType;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;

import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.ScoreCard;
import com.playertour.backend.api.Stroke;
import com.playertour.backend.apis.player.APIScorecardService;
import com.playertour.backend.apis.player.PlayerIndexService;
import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.player.model.player.Player;

/**
 * 
 * @author ilenia
 * @since Mar 2, 2021
 */
@Component(scope = ServiceScope.PROTOTYPE)
public class APIScorecardServiceImpl implements APIScorecardService {

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	EMFRepository repository;

	@Reference
	PlayerIndexService playerIndexService;

	@Reference
	PlayerSearchService playerSearchService;


	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#getScorecardsByUser(java.lang.String)
	 */
	@Override
	public List<ScoreCard> getScorecardsByUser(String userId) {
		if(userId == null) {
			throw new IllegalArgumentException("Cannot get PlayerProfile for user with null login id!");
		}
		Player player = playerSearchService.getPlayerByLoginId(userId);
		if(player == null) {
			throw new IllegalArgumentException("No Player associated with the user id " + userId + ". Create a PlayerProfile first!");
		}		
		return player.getScoreCards();
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#getScorecardsByCourse(java.lang.String, java.lang.String)
	 */
	@Override
	public List<ScoreCard> getScorecardsByCourse(String userId, String courseId) {
		if(userId == null) {
			throw new IllegalArgumentException("Cannot get Scorecard by course for a null course id!");
		}
		List<ScoreCard> scorecards = getScorecardsByUser(userId);
		scorecards = scorecards.stream().filter(s->courseId.equals(s.getCourseId())).collect(Collectors.toList());
		return scorecards;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#saveScorecard(java.lang.String, com.playertour.backend.api.ScoreCard)
	 */
	@Override
	public ScoreCard saveScorecard(String userId, ScoreCard scorecard) {
		if(userId == null) {
			throw new IllegalArgumentException("Cannot get PlayerProfile for user with null login id!");
		}
		Player player = playerSearchService.getPlayerByLoginId(userId);
		if(player == null) {
			throw new IllegalArgumentException("No Player associated with the user id " + userId + ". Create a PlayerProfile first!");
		}		
		boolean isFirstSave = scorecard.getId() == null ? true : false;
		if(!isFirstSave) {
			for(Iterator<ScoreCard> iter = player.getScoreCards().iterator(); iter.hasNext();) {
				ScoreCard s = iter.next();
				if(scorecard.getId().equals(s.getId())) {
					iter.remove();
					break;
				}
			}
		}
		player.getScoreCards().add(scorecard);
		save(player.getProfile().getGolfCourseHistory());
		save(player.getProfile().getFavCourses());
		repository.save(player);
		playerIndexService.indexPlayer(player, IndexActionType.MODIFY);
		return scorecard;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#openScorecard(java.lang.String)
	 */
	@Override
	public ScoreCard openScorecard(String courseId) {
		// TODO Auto-generated method stub
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#submitStroke(java.lang.String, com.playertour.backend.api.Stroke)
	 */
	@Override
	public ScoreCard submitStroke(String scorecardId, Stroke stroke) {
		// TODO Auto-generated method stub
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#closeHole(java.lang.String, com.playertour.backend.api.HoleScore)
	 */
	@Override
	public ScoreCard closeHole(String scorecardId, HoleScore holeScore) {
		// TODO Auto-generated method stub
		return null;
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.APIScorecardService#closeScorecard(java.lang.String, java.lang.String)
	 */
	@Override
	public ScoreCard closeScorecard(String userId, String scorecardId) {
		// TODO Auto-generated method stub
		return null;
	}

	@SuppressWarnings("unchecked")
	private void save(Collection<? extends EObject> objects) {
		repository.save((Collection<EObject>)  objects);
	}

}
