/**
 * Copyright (c) 2012 - 2021 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.player;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.search.document.LuceneIndexService;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.common.DocumentUtil;
import com.playertour.backend.apis.player.PlayerSearchService;
import com.playertour.backend.player.model.player.Player;

/**
 * 
 * @author ilenia
 * @since Mar 2, 2021
 */
@Component(immediate = true, scope = ServiceScope.SINGLETON)
public class PlayerSearchServiceImpl implements PlayerSearchService {
	
	@Reference(target = "(id=player)")
	private LuceneIndexService playerIndex;

	@Reference
	ResourceSet resourceSet;

	@Reference(service=LoggerFactory.class)
	private Logger logger;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.apis.player.PlayerSearchService#getPlayerByLoginId(java.lang.String)
	 */
	@Override
	public Player getPlayerByLoginId(String loginId) {
		IndexSearcher searcher = playerIndex.aquireSearch();
		try {
			try {
				TopDocs topDocs = searcher.search(new TermQuery(new Term("loginId", loginId)), 1);
				if(topDocs.scoreDocs.length == 0) {
					return null;
				}
				IndexReader indexReader = searcher.getIndexReader();
				Document document = indexReader.document(topDocs.scoreDocs[0].doc);
				return (Player) DocumentUtil.toEObject(document, resourceSet);
			} catch (Exception e) {
				logger.error( "Exception while search for Player with loginId " + loginId, e);
				return null;
			}
		} finally {
			playerIndex.releaseSearcher(searcher);
		}
	}

}
