/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.possiblecheats.service.impl;

import java.util.Collections;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.DocumentIndexContextObjectImpl;
import org.gecko.search.util.DeferredCommitCallback;
import org.osgi.util.promise.Deferred;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatsPackage;

public class PossibleCheatsReportingIndexHelper {
	static final String POSSIBLECHEATREPORT_USER_ID = "userID";
	static final String POSSIBLECHEATREPORT_SCORECARD_ID = "scorecardID";
	static final String POSSIBLECHEATREPORT_STATUS = "status";
	static final String POSSIBLECHEATREPORT_TYPE = "type";

	public static DocumentIndexContextObject mapPossibleCheatReportNew(PossibleCheatReport possibleCheatReport) {
		return mapPossibleCheatReport(possibleCheatReport, IndexActionType.ADD, null);
	}

	public static DocumentIndexContextObject mapPossibleCheatReportNew(PossibleCheatReport possibleCheatReport,
			Deferred<Boolean> deferred) {
		return mapPossibleCheatReport(possibleCheatReport, IndexActionType.ADD, deferred);
	}

	public static DocumentIndexContextObject mapPossibleCheatReportUpdate(PossibleCheatReport possibleCheatReport,
			Deferred<Boolean> deferred) {
		return mapPossibleCheatReport(possibleCheatReport, IndexActionType.MODIFY, deferred);
	}

	public static DocumentIndexContextObject mapPossibleCheatReport(PossibleCheatReport possibleCheatReport,
			IndexActionType indexAction, Deferred<Boolean> deferred) {
		Document doc = new Document();

		doc.add(new StringField(LuceneIndexHelper.FIELD__ECLASS_URI, constructEClassUri(), Store.YES));
		doc.add(new StringField(LuceneIndexHelper.FIELD__ID, possibleCheatReport.getId(), Store.YES));
		
		doc.add(new StringField(POSSIBLECHEATREPORT_USER_ID, possibleCheatReport.getUserID(), Store.NO));
		doc.add(new StringField(POSSIBLECHEATREPORT_SCORECARD_ID, possibleCheatReport.getScorecardID(), Store.NO));
		doc.add(new StringField(POSSIBLECHEATREPORT_STATUS, possibleCheatReport.getStatus().getLiteral(), Store.NO));
		doc.add(new StringField(POSSIBLECHEATREPORT_TYPE, possibleCheatReport.getType().getLiteral(), Store.NO));

		DocumentIndexContextObjectImpl.Builder builder = DocumentIndexContextObjectImpl.builder()
				.withDocuments(Collections.singletonList(doc)).withSourceObject(possibleCheatReport)
				.withIndexActionType(indexAction);

		if (IndexActionType.MODIFY.equals(indexAction) || IndexActionType.REMOVE.equals(indexAction)) {
			builder.withIdentifyingTerm(new Term("id", possibleCheatReport.getId()));
		}
		if (deferred != null) {
			builder = builder.withCommitCallback(new DeferredCommitCallback(deferred));
		}

		return builder.build();
	}

	private static String constructEClassUri() {
		return EcoreUtil.getURI(PossibleCheatsPackage.Literals.POSSIBLE_CHEAT_REPORT).toString();
	}
}
