/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.possiblecheats.service.impl;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.Objects;

import org.gecko.emf.repository.EMFRepository;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceScope;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.api.HoleScore;
import com.playertour.backend.api.LandingZone;
import com.playertour.backend.api.PlayerScorecard;
import com.playertour.backend.apis.player.PlayerScorecardService;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatsFactory;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatsPackage;
import com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingIndexService;
import com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService;
import com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService;

@Component(name = "PossibleCheatsReportingService", scope = ServiceScope.PROTOTYPE)
public class PossibleCheatsReportingServiceImpl implements PossibleCheatsReportingService {
	
	private static final String PCR_STROKE_LANDING_ZONE_DETAILS = "Detected possible cheat of type %s: stroke no. %d for hole no. %d landed on %s";

	@Reference(target = "(repo_id=playertour.playertour)", scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private EMFRepository repository;

	@Reference
	private EventAdmin eventAdmin;

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	@Reference
	private PossibleCheatsReportingIndexService possibleCheatsReportingIndexService;

	@Reference
	private PossibleCheatsReportingSearchService possibleCheatsReportingSearchService;
	
	@Reference(scope = ReferenceScope.PROTOTYPE_REQUIRED)
	private PlayerScorecardService playerScorecardService;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#savePossibleCheatReport(com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport)
	 */
	@Override
	public PossibleCheatReport savePossibleCheatReport(PossibleCheatReport possibleCheatReport) {
		Objects.requireNonNull(possibleCheatReport, "Possible cheat report is required!");

		if (possibleCheatReportExists(possibleCheatReport.getId())) {
			logger.warn("Possible cheat report id " + possibleCheatReport.getId() + " already exists!");
			return getPossibleCheatsReport(possibleCheatReport.getId());
		}

		repository.save(possibleCheatReport);

		possibleCheatsReportingIndexService.indexPossibleCheatReport(possibleCheatReport, true);

		return possibleCheatReport;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReports(java.lang.String)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReports(String loginId) {
		Objects.requireNonNull(loginId, "Player login Id is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByPlayerLoginId(loginId);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsWithStatus(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsWithStatus(String loginId,
			PossibleCheatReportStatus status) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(status, "Possible cheat report status is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByPlayerLoginIdWithStatus(loginId, status);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsWithType(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsWithType(String loginId, PossibleCheatReportType type) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(type, "Possible cheat report type is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByPlayerLoginIdWithType(loginId, type);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsWithStatusAndType(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsWithStatusAndType(String loginId,
			PossibleCheatReportStatus status, PossibleCheatReportType type) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(status, "Possible cheat report status is required!");
		Objects.requireNonNull(type, "Possible cheat report type is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByPlayerLoginIdWithStatusAndType(loginId,
				status, type);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsByScorecardId(java.lang.String, java.lang.String)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsByScorecardId(String loginId, String scorecardId) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(scorecardId, "Scorecard Id is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByScorecardId(scorecardId);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsByScorecardIdWithStatus(java.lang.String, java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsByScorecardIdWithStatus(String loginId, String scorecardId,
			PossibleCheatReportStatus status) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(scorecardId, "Scorecard Id is required!");
		Objects.requireNonNull(status, "Possible cheat report status is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByScorecardIdWithStatus(scorecardId, status);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsByScorecardIdWithType(java.lang.String, java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsByScorecardIdWithType(String loginId, String scorecardId,
			PossibleCheatReportType type) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(scorecardId, "Scorecard Id is required!");
		Objects.requireNonNull(type, "Possible cheat report type is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByScorecardIdWithType(scorecardId, type);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReportsByScorecardIdWithStatusAndType(java.lang.String, java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> getPossibleCheatsReportsByScorecardIdWithStatusAndType(String loginId,
			String scorecardId, PossibleCheatReportStatus status, PossibleCheatReportType type) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(scorecardId, "Scorecard Id is required!");
		Objects.requireNonNull(status, "Possible cheat report status is required!");
		Objects.requireNonNull(type, "Possible cheat report type is required!");

		return possibleCheatsReportingSearchService.searchPossibleCheatsByScorecardIdWithStatusAndType(scorecardId,
				status, type);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#getPossibleCheatsReport(java.lang.String)
	 */
	@Override
	public PossibleCheatReport getPossibleCheatsReport(String possibleCheatReportId) {
		Objects.requireNonNull(possibleCheatReportId, "Possible cheat report ID is required!");

		return repository.getEObject(PossibleCheatsPackage.Literals.POSSIBLE_CHEAT_REPORT, possibleCheatReportId);
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#updateStatus(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus)
	 */
	@Override
	public PossibleCheatReport updateStatus(String possibleCheatReportId, PossibleCheatReportStatus status) {
		Objects.requireNonNull(possibleCheatReportId, "Possible cheat report ID is required!");
		Objects.requireNonNull(status, "Possible cheat report status is required!");

		PossibleCheatReport existingPossibleCheatReport = getPossibleCheatsReport(possibleCheatReportId);

		existingPossibleCheatReport.setStatus(status);

		repository.save(existingPossibleCheatReport);

		possibleCheatsReportingIndexService.indexPossibleCheatReport(existingPossibleCheatReport, false);

		return existingPossibleCheatReport;
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingService#detectPossibleCheats(java.lang.String, java.lang.String)
	 */
	@Override
	public void detectPossibleCheats(String loginId, String scorecardId) {
		Objects.requireNonNull(loginId, "Player login Id is required!");
		Objects.requireNonNull(scorecardId, "Scorecard Id is required!");
		
		PlayerScorecard playerScorecard = playerScorecardService.getScorecardById(loginId, scorecardId);
		
		if (isPlayerScorecardValid(playerScorecard)) {
			detectGreenHitsPossibleCheats(loginId, playerScorecard);
		}
	}

	private boolean possibleCheatReportExists(String possibleCheatReportID) {
		if (possibleCheatReportID != null && !possibleCheatReportID.isBlank()) {
			return (getPossibleCheatsReport(possibleCheatReportID) != null);
		} else {
			return false;
		}
	}
	
	private boolean isPlayerScorecardValid(PlayerScorecard playerScorecard) {
		return (playerScorecard != null 
				&& playerScorecard.getEnd() != null 
				&& playerScorecard.isNoReturn() == false);
	}
	
	private void detectGreenHitsPossibleCheats(String loginId, PlayerScorecard playerScorecard) {
		for (HoleScore hs : playerScorecard.getHoleStats()) {
			if (!hs.getStrokes().isEmpty()) {
				
				/* 
				 * Green hit means:
				 * 	* 1st stroke on the green on a par 3 hole
				 *  * 2nd stroke on the green on a par 4 hole and 
				 *  * 3rd stroke on the green on a par 5 hole.
				 *  
				 * Therefore, if for par 4 hole, 1st stroke lands on green, it is considered a possible cheat. 
				 * 
				 * Similarly, if for par 5 hole, 1st or 2nd stroke lands on green, it is considered a possible cheat.
				 * 
				 */
				
				detectGreenHitsPossibleCheats(loginId, playerScorecard.getId(), hs, 4, 1,
						PossibleCheatReportType.GREEN_HITS_PAR4S_RESULT);

				detectGreenHitsPossibleCheats(loginId, playerScorecard.getId(), hs, 5, 1,
						PossibleCheatReportType.GREEN_HITS_PAR5S_RESULT);

				detectGreenHitsPossibleCheats(loginId, playerScorecard.getId(), hs, 5, 2,
						PossibleCheatReportType.GREEN_HITS_PAR5S_RESULT);
			}
		}
	}

	private void detectGreenHitsPossibleCheats(String loginId, String playerScorecardId, HoleScore hs, int par,
			int strokeNumber, PossibleCheatReportType type) {
		if (hs.getHoleInfo().getPar().intValue() == par && hs.getStrokes().size() >= strokeNumber
				&& hs.getStrokes().get(strokeNumberToIndex(strokeNumber)).getZone() == LandingZone.GREEN) {

			recordGreenHitsPossibleCheatReport(loginId, playerScorecardId, type, hs.getHoleNumber(), strokeNumber);
		}
	}

	private void recordGreenHitsPossibleCheatReport(String loginId, String playerScorecardId,
			PossibleCheatReportType type, int holeNumber, int strokeNumber) {
		// @formatter:off
		PossibleCheatReport possibleCheatReport = constructPossibleCheatReport(
				loginId, 
				playerScorecardId,
				PossibleCheatReportStatus.NEW, 
				type, 
				String.format(PCR_STROKE_LANDING_ZONE_DETAILS, 
						type.getLiteral(), 
						strokeNumber, 
						holeNumber,
						LandingZone.GREEN), 
				LocalDateTime.now().atZone(ZoneOffset.systemDefault()).toEpochSecond());
		// @formatter:on

		savePossibleCheatReport(possibleCheatReport);
	}

	private PossibleCheatReport constructPossibleCheatReport(String userID, String scorecardID,
			PossibleCheatReportStatus status, PossibleCheatReportType type, String details, long timeStamp) {
		PossibleCheatReport possibleCheatReport = PossibleCheatsFactory.eINSTANCE.createPossibleCheatReport();
		possibleCheatReport.setUserID(userID);
		possibleCheatReport.setScorecardID(scorecardID);
		possibleCheatReport.setStatus(status);
		possibleCheatReport.setType(type);
		possibleCheatReport.setDetails(details);
		possibleCheatReport.setTimeStamp(timeStamp);
		return possibleCheatReport;
	}

	private int strokeNumberToIndex(int strokeNumber) {
		return (strokeNumber - 1);
	}	
}
