/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.possiblecheats.service.impl;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.eclipse.emf.ecore.EObject;
import org.gecko.emf.mongo.Options;
import org.gecko.emf.osgi.ResourceSetFactory;
import org.gecko.emf.osgi.UriMapProvider;
import org.gecko.emf.pushstream.EPushStreamProvider;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.emf.repository.query.IQuery;
import org.gecko.emf.repository.query.QueryRepository;
import org.gecko.search.api.IndexActionType;
import org.gecko.search.document.DocumentIndexContextObject;
import org.gecko.search.document.LuceneIndexService;
import org.gecko.search.util.CommonIndexService;
import org.gecko.util.pushstreams.GeckoPushbackPolicyOption;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.PromiseFactory;
import org.osgi.util.pushstream.PushEvent;
import org.osgi.util.pushstream.PushStream;
import org.osgi.util.pushstream.QueuePolicyOption;

import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatsPackage;
import com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingIndexService;

@Component(immediate = true, name = "PossibleCheatsReportingIndexService", service = {PossibleCheatsReportingIndexService.class, CommonIndexService.class}, scope = ServiceScope.SINGLETON, reference = {
		@Reference(name = "mongoCondition", service = UriMapProvider.class, target = "(uri.map.src=mongodb://playertour/)", cardinality = ReferenceCardinality.MANDATORY),
		@Reference(name = "modelCondition", service = ResourceSetFactory.class, target = "(emf.model.name=possiblecheats)", cardinality = ReferenceCardinality.MANDATORY)
})
public class PossibleCheatsReportingIndexServiceImpl
		implements PossibleCheatsReportingIndexService, CommonIndexService {

	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;

	@Reference(target = "(id=possibleCheatsReports)")
	private LuceneIndexService possibleCheatsReportsIndex;

	@Reference(service = LoggerFactory.class)
	private Logger logger;

	private PromiseFactory factory = new PromiseFactory(Executors.newFixedThreadPool(4));

	@Activate
	public void activate() {
		factory.submit(() -> {
			CountDownLatch latch = new CountDownLatch(1);
			latch.await(100, TimeUnit.MILLISECONDS);
			initializeIndex();
			return true;
		}).onSuccess(t -> logger.info("Finished indexing PossibleCheatReport objects!"))
				.onFailure(t -> t.printStackTrace());
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingIndexService#indexPossibleCheatReport(com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport, boolean)
	 */
	@Override
	public void indexPossibleCheatReport(PossibleCheatReport possibleCheatReport, boolean isFirstSave) {
		if (isFirstSave) {
			indexPossibleCheatReport(possibleCheatReport, IndexActionType.ADD);
		} else {
			indexPossibleCheatReport(possibleCheatReport, IndexActionType.MODIFY);
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingIndexService#deletePossibleCheatReport(com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport)
	 */
	@Override
	public void deletePossibleCheatReport(PossibleCheatReport possibleCheatReport) {
		indexPossibleCheatReport(possibleCheatReport, IndexActionType.REMOVE);
	}	
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.search.util.CommonIndexService#initializeIndex()
	 */
	@Override
	public Promise<Void> initializeIndex() {
		EMFRepository repository = repositoryServiceObjects.getService();
		QueryRepository queryRepo = (QueryRepository) repository.getAdapter(QueryRepository.class);

		IQuery query = queryRepo.createQueryBuilder().allQuery().build();
		logger.info("Starting index build for PossibleCheatReport");

		resetIndex();

		Map<Object, Object> loadOptions = CommonIndexService.getLoadOptions();
		loadOptions.put(Options.OPTION_READ_DETACHED, true);
		EPushStreamProvider psp = queryRepo.getEObjectByQuery(PossibleCheatsPackage.Literals.POSSIBLE_CHEAT_REPORT,
				query, loadOptions);
		if (psp == null) {
			return null;
		}

		PushStream<EObject> indexNew = psp.createPushStreamBuilder()
				.withPushbackPolicy(GeckoPushbackPolicyOption.LINEAR_AFTER_THRESHOLD.getPolicy(650))
				.withQueuePolicy(QueuePolicyOption.BLOCK).withExecutor(Executors.newSingleThreadExecutor())
				.withBuffer(new ArrayBlockingQueue<PushEvent<? extends EObject>>(10)).build();

		List<DocumentIndexContextObject> contexts = new LinkedList<>();
		Promise<Void> resultPromise = indexNew.map(eo -> (PossibleCheatReport) eo)
				.map(PossibleCheatsReportingIndexHelper::mapPossibleCheatReportNew).forEach(d -> {
					repository.detach(d.getObject());
					if (contexts.size() < 10) {
						contexts.add(d);
					} else {
						possibleCheatsReportsIndex.handleContextsSync(contexts);
						contexts.clear();
						contexts.add(d);
					}
				}).thenAccept(result -> {
					if (!contexts.isEmpty()) {
						try {
							possibleCheatsReportsIndex.handleContextsSync(contexts);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				});

		return resultPromise.onFailure(t -> repositoryServiceObjects.ungetService(repository))
				.thenAccept(result -> repositoryServiceObjects.ungetService(repository));
	}
	
	/* 
	 * (non-Javadoc)
	 * @see org.gecko.search.util.CommonIndexService#resetIndex()
	 */
	@Override
	public void resetIndex() {
		try {
			possibleCheatsReportsIndex.getIndexWriter().deleteAll();
			possibleCheatsReportsIndex.commit();
		} catch (IOException e) {
			logger.error("Could not delete PossibleCheatReport index", e);
		}
	}

	private void indexPossibleCheatReport(PossibleCheatReport possibleCheatReport, IndexActionType actionType) {
		DocumentIndexContextObject context = PossibleCheatsReportingIndexHelper
				.mapPossibleCheatReport(possibleCheatReport, actionType, null);
		possibleCheatsReportsIndex.handleContextSync(context);
	}
}
