/**
 * Copyright (c) 2012 - 2022 Data In Motion and others.
 * All rights reserved. 
 * 
 * This program and the accompanying materials are made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Data In Motion - initial API and implementation
 */
package com.playertour.backend.possiblecheats.service.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.gecko.emf.repository.EMFRepository;
import org.gecko.search.document.LuceneIndexService;
import org.osgi.service.component.ComponentServiceObjects;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ServiceScope;
import org.osgi.service.log.Logger;
import org.osgi.service.log.LoggerFactory;

import com.playertour.backend.apis.common.LuceneIndexHelper;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReport;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus;
import com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType;
import com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService;

@Component(name = "PossibleCheatsReportingSearchService", immediate = true, scope = ServiceScope.SINGLETON)
public class PossibleCheatsReportingSearchServiceImpl implements PossibleCheatsReportingSearchService {
	
	@Reference(target = "(repo_id=playertour.playertour)", cardinality = ReferenceCardinality.MANDATORY)
	private ComponentServiceObjects<EMFRepository> repositoryServiceObjects;
	
	@Reference(target = "(id=possibleCheatsReports)")
	private LuceneIndexService possibleCheatsReportsIndex;

	@Reference
	private ResourceSet resourceSet;

	@Reference(service=LoggerFactory.class)
	private Logger logger;

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByPlayerLoginId(java.lang.String)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByPlayerLoginId(String loginId) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_USER_ID, loginId));
				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by login id " + loginId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByPlayerLoginIdWithStatus(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByPlayerLoginIdWithStatus(String loginId,
			PossibleCheatReportStatus status) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q1 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_USER_ID, loginId));
				Query q2 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_STATUS, status.getLiteral()));

				// @formatter:off
				Query q = new BooleanQuery.Builder()
						.add(new BooleanClause(q1, Occur.MUST))
						.add(new BooleanClause(q2, Occur.MUST))
						.build();
				// @formatter:on

				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by login id " + loginId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByPlayerLoginIdWithType(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByPlayerLoginIdWithType(String loginId, PossibleCheatReportType type) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q1 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_USER_ID, loginId));
				Query q2 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_TYPE, type.getLiteral()));

				// @formatter:off
				Query q = new BooleanQuery.Builder()
						.add(new BooleanClause(q1, Occur.MUST))
						.add(new BooleanClause(q2, Occur.MUST))
						.build();
				// @formatter:on

				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by login id " + loginId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByPlayerLoginIdWithStatusAndType(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByPlayerLoginIdWithStatusAndType(String loginId, PossibleCheatReportStatus status,
			PossibleCheatReportType type) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q1 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_USER_ID, loginId));
				Query q2 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_STATUS, status.getLiteral()));
				Query q3 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_TYPE, type.getLiteral()));

				// @formatter:off
				Query q = new BooleanQuery.Builder()
						.add(new BooleanClause(q1, Occur.MUST))
						.add(new BooleanClause(q2, Occur.MUST))
						.add(new BooleanClause(q3, Occur.MUST))
						.build();
				// @formatter:on

				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by login id " + loginId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByScorecardId(java.lang.String)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByScorecardId(String scorecardId) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_SCORECARD_ID, scorecardId));
				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by scorecard id " + scorecardId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByScorecardIdWithStatus(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByScorecardIdWithStatus(String scorecardId,
			PossibleCheatReportStatus status) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q1 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_SCORECARD_ID, scorecardId));
				Query q2 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_STATUS, status.getLiteral()));

				// @formatter:off
				Query q = new BooleanQuery.Builder()
						.add(new BooleanClause(q1, Occur.MUST))
						.add(new BooleanClause(q2, Occur.MUST))
						.build();
				// @formatter:on

				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by scorecard id " + scorecardId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByScorecardIdWithType(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByScorecardIdWithType(String scorecardId,
			PossibleCheatReportType type) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q1 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_SCORECARD_ID, scorecardId));
				Query q2 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_TYPE, type.getLiteral()));

				// @formatter:off
				Query q = new BooleanQuery.Builder()
						.add(new BooleanClause(q1, Occur.MUST))
						.add(new BooleanClause(q2, Occur.MUST))
						.build();
				// @formatter:on

				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by scorecard id " + scorecardId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}

	/* 
	 * (non-Javadoc)
	 * @see com.playertour.backend.possiblecheats.service.api.PossibleCheatsReportingSearchService#searchPossibleCheatsByScorecardIdWithStatusAndType(java.lang.String, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportStatus, com.playertour.backend.possiblecheats.model.possiblecheats.PossibleCheatReportType)
	 */
	@Override
	public List<PossibleCheatReport> searchPossibleCheatsByScorecardIdWithStatusAndType(String scorecardId,
			PossibleCheatReportStatus status, PossibleCheatReportType type) {
		IndexSearcher searcher = possibleCheatsReportsIndex.aquireSearch();
		EMFRepository repository = repositoryServiceObjects.getService();
		try {
			try {
				Query q1 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_SCORECARD_ID, scorecardId));
				Query q2 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_STATUS, status.getLiteral()));
				Query q3 = new TermQuery(
						new Term(PossibleCheatsReportingIndexHelper.POSSIBLECHEATREPORT_TYPE, type.getLiteral()));

				// @formatter:off
				Query q = new BooleanQuery.Builder()
						.add(new BooleanClause(q1, Occur.MUST))
						.add(new BooleanClause(q2, Occur.MUST))
						.add(new BooleanClause(q3, Occur.MUST))
						.build();
				// @formatter:on

				TopDocs topDocs = searcher.search(q, Integer.MAX_VALUE);
				if (topDocs.scoreDocs.length == 0) {
					return Collections.emptyList();
				}
				IndexReader indexReader = searcher.getIndexReader();
				return Arrays.asList(topDocs.scoreDocs).stream().map(sd -> sd.doc).map(id -> {
					Document d;
					try {
						d = indexReader.document(id);
						return d;
					} catch (IOException e) {
						return null;
					}
				}).filter(d -> d != null).map(d -> {

					return (PossibleCheatReport) LuceneIndexHelper.toEObject(d, repository);

				}).collect(Collectors.toList());
			} catch (Exception e) {
				logger.error("Exception while searching for PossibleCheatReport by scorecard id " + scorecardId, e);
				return Collections.emptyList();
			}
		} finally {
			possibleCheatsReportsIndex.releaseSearcher(searcher);
			repositoryServiceObjects.ungetService(repository);
		}
	}
}
